/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include <libnd_debug.h>
#include <libnd_filter.h>
#include <libnd_filter_registry.h>

static GList      *registry;          /* GList<LND_Filter*> */
static GList      *observers;         /* GList<LND_FilterRegObserver*> */

void     
libnd_filter_registry_init(void)
{
  registry = NULL;
  observers = NULL;
}


void     
libnd_filter_registry_add(LND_Filter *filter)
{
  D_ENTER;

  if (!filter)
    D_RETURN;

  if (g_list_find(registry, filter))
    D_RETURN;
  
  registry = g_list_prepend(registry, filter);
  libnd_filter_registry_tell_observers(filter, LND_FILTER_ADDED);
  D_RETURN;
}


void     
libnd_filter_registry_del(LND_Filter *filter)
{
  D_ENTER;

  if (!filter)
    D_RETURN;
  
  registry = g_list_remove(registry, filter);
  libnd_filter_registry_tell_observers(filter, LND_FILTER_DELETED);
  D_RETURN;
}


void     
libnd_filter_registry_foreach(LND_FilterRegFunc callback, void *user_data)
{
  LND_Filter *filter;
  GList *l;

  D_ENTER;

  if (!callback)
    D_RETURN;

  for (l = registry; l; l = g_list_next(l))
    {
      filter = (LND_Filter *) l->data;
      callback(filter, user_data);
    }

  D_RETURN;
}



LND_FilterRegObserver *
libnd_filter_registry_observer_new(void)
{
  return g_new0(LND_FilterRegObserver, 1);
}


void               
libnd_filter_registry_observer_free(LND_FilterRegObserver *ob)
{
  g_free(ob);
}


void           
libnd_filter_registry_add_observer(LND_FilterRegObserver *observer)
{
  if (!observer)
    return;

  observers = g_list_append(observers, observer);
}


void           
libnd_filter_registry_del_observer(LND_FilterRegObserver *observer)
{
  if (!observer)
    return;

  observers = g_list_remove(observers, observer);
}


void           
libnd_filter_registry_tell_observers(LND_Filter *filter, LND_FilterRegObserverOp op)
{
  GList *l;
  LND_FilterRegObserver *ob;

  if (!filter)
    return;

  for (l = observers; l; l = g_list_next(l))
    {
      ob = (LND_FilterRegObserver *) l->data;

      switch (op)
	{
	case LND_FILTER_ADDED:
	  if (ob->filter_added)
	    ob->filter_added(filter);
	  break;

	case LND_FILTER_DELETED:
	  if (ob->filter_deleted)
	    ob->filter_deleted(filter);
	  break;

	default:
	  D(("Unknown filter operation\n"));
	}
    }
}
