/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd_snap.h>

static LND_Protocol *snap;

static gboolean
snap_header_complete(const LND_Packet *packet, guchar *data)
{
  if (!data)
    return FALSE;

  return (data + LND_LLC_LEN <= libnd_packet_get_end(packet));
}


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("SNAP Plugin");
}


const char *
description(void)
{
  return ("A small plugin providing support for "
	  "LLC encapsulation and SNAP (Sub-Network "
	  "Access Protocol).");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION;
}


LND_Protocol *
init(void)
{
  snap = libnd_proto_new("LLC/SNAP", LND_PROTO_LAYER_LINK, LND_PROTO_SNAP);
			     		
  snap->init_packet     = libnd_snap_init_packet;
  snap->header_complete = libnd_snap_header_complete;

  /* That's all -- we don't need checksums or state maintenance
     for simple Ethernet. The other methods got initialized
     to dummy null operations in the constructor call above.
     
     We also don't need a special menu to appear in the Protocols
     menu for this plugin.
  */
	     
  return snap;
}


/* Protocol method implementations: ------------------------------------ */

void       
libnd_snap_init_packet(LND_Packet *packet, guchar *data, guchar *data_end)
{
  LND_Protocol *payload_proto = NULL;
  struct lnd_snap_header *sh;

  if (!snap_header_complete(packet, data))
    {
      libnd_raw_proto_get()->init_packet(packet, data, data_end);
      return;
    }

  libnd_packet_add_proto_data(packet, snap, data, data_end);  
  sh = (struct lnd_snap_header *) data;

  if (sh->ssap == 0xaa && sh->dsap == 0xaa)
    {
      /* It's a SNAP header -- use protocol field: */

      if (! (payload_proto = libnd_proto_registry_find(LND_PROTO_LAYER_NET, ntohs(sh->proto))))
	payload_proto = libnd_raw_proto_get();

      payload_proto->init_packet(packet, data + LND_SNAP_LEN, data_end);	    
    }
  else
    {
      /* Not sure if this is correct, but if I understand things right after
	 some digging on the web, it should be safe to start the payload data
	 directly after ssap, dsap and the control field. I have dug up a few
	 SAP codes on http://www.cisco.com/warp/public/90/22.html, I'm just
	 using IP for now. Can't test it anyway ...
      */

      /* Note: IPX uses an Ethertype of 0x8037,
               NetBIOS/NetBEUI/SMB uses an Ethertype of 0x8191. 

	       http://www.openwiki.com/ow.asp?NetworkInfo
      */

      switch (sh->ssap)
	{
	case 0x06:
	  payload_proto = libnd_proto_registry_find(LND_PROTO_LAYER_NET, ETHERTYPE_IP);
	  return;
	  
	case 0xf0:
	  payload_proto = libnd_proto_registry_find(LND_PROTO_LAYER_NET, 0x8191);
	  return;
	  
	default:
	  /* If all else fails, just use the hex editor ... */
	  payload_proto = NULL;
	}
      
      if (!payload_proto)
	payload_proto = libnd_raw_proto_get();
      
      payload_proto->init_packet(packet, data + LND_LLC_LEN, data_end);
    }
}


gboolean
libnd_snap_header_complete(const LND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = libnd_packet_get_data(packet, snap, nesting);

  return snap_header_complete(packet, data);
}


LND_Protocol *
libnd_snap_get(void)
{
  return snap;
}

