/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <libnd_fddi.h>
#include <protocols/snap/libnd_snap.h>

static LND_Protocol *fddi;

static gboolean
fddi_header_complete(const LND_Packet *packet, guchar *data)
{
  if (!data)
    return FALSE;

  return (data + 13 <= libnd_packet_get_end(packet));
}


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("FDDI Plugin");
}


const char *
description(void)
{
  return ("A plugin providing support for the FDDI protocol.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION;
}


LND_Protocol *
init(void)
{
  fddi = libnd_proto_new("FDDI", LND_PROTO_LAYER_LINK, DLT_FDDI);

  fddi->init_packet     = libnd_fddi_init_packet;
  fddi->header_complete = libnd_fddi_header_complete;

  return fddi;
}


/* Protocol method implementations: ------------------------------------ */

void       
libnd_fddi_init_packet(LND_Packet *packet, guchar *data, guchar *data_end)
{
  LND_Protocol *payload_proto;
  struct fddi_header *fh;

  if (!fddi_header_complete(packet, data))
    {
      libnd_raw_proto_get()->init_packet(packet, data, data_end);
      return;
    }

  libnd_packet_add_proto_data(packet, fddi, data, data_end);  
  
  /* Cast the data pointer into your protocol's header */
  fh = (struct fddi_header *) data;

  /* Check the appriopriate header field value to demultiplex
     packet initialization up to the next correct protocol: */

  if ((fh->fddi_fc >= 0x50 && fh->fddi_fc <= 0x5F) ||
      (fh->fddi_fc >= 0xD0 && fh->fddi_fc <= 0xD7))
    {
      /* This is an LLC frame. Pass on to the LLX/SNAP implementation: */
      if (! (payload_proto = libnd_proto_registry_find(LND_PROTO_LAYER_LINK, LND_PROTO_SNAP)))
	payload_proto = libnd_raw_proto_get();

      payload_proto->init_packet(packet, data + 13, data_end);
      return;
    }

  /* Otherwise, we don't handle this yet. Sorry. */
}


gboolean
libnd_fddi_header_complete(const LND_Packet *packet, guint nesting)
{
  guchar      *data;

  if (!packet)
    return FALSE;

  data = libnd_packet_get_data(packet, fddi, nesting);

  return fddi_header_complete(packet, data);
  TOUCH(nesting);
}


LND_Protocol *
libnd_fddi_get(void)
{
  return fddi;
}

