/* Copyright (C) 1998 Sean Gabriel <gabriel@korsoft.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <errno.h> 
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/time.h> 
#include <sys/types.h>

#include <glib.h>

#include "icq.h"
#include "libicq.h"
#include "send.h"
#include "receive.h"
#include "tcp.h"
#include "util.h"
#include "config.h"


void TCP_SendMessages(int i);
void UDP_SendMessages(int i);


int SRV_Addresses = 0;  /* The number of possible ICQ server IP addresses */
int SRV_AddressToUse = 0;  /* The index of the next address to try */
int Verbose = ICQ_VERB_NONE;
gboolean serv_mess[1024];
guint16 last_cmd[1024]; /* command issued for the first 1024 SEQ #'s */
guint16 seq_num = 1;  /* current sequence number */
guint32 our_ip = 0x0100007f; /* localhost for some reason */
guint32 our_port; /* the port to make tcp connections on */
Contact_Member Contacts[ 100 ]; /* no more than 100 contacts max */
int Num_Contacts;
guint32 Current_Status=STATUS_ONLINE;
SEARCH_RESULT* Search_Results;
guint32 last_recv_uin=0;
guint32 UIN;
gchar passwd[100];
gchar server[100];
guint32 remote_port;
guint32 set_status;
gboolean Done_Login=FALSE;
gint sok;
gint tcp_sok;

extern gchar contacts_rc[100];

/* #define DEBUG */

gint ICQ_Read_Config()
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Read_Config");
#endif

  return Get_Config_Info();
}


gint ICQ_Connect()
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Connect");
#endif

  Search_Results = NULL;
  if(Get_Config_Info() == 0) return 0;
  ICQ_Change_Status(Current_Status);

  return 1;
}


void ICQ_Disconnect()
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Disconnect");
#endif

  Send_Disconnect();
}


void ICQ_Change_Status(guint32 new_status)
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Change_Status(%ld)", new_status);
#endif

  if(new_status == STATUS_OFFLINE && sok != 0)
  {
    ICQ_Disconnect();
    Current_Status = new_status;
    return;
  }

  if(new_status != STATUS_OFFLINE && sok == 0)
  {
    Current_Status = new_status;
    if(Connect_Remote(server, remote_port) == 0)
    {
      if(Verbose & ICQ_VERB_ERR)
        fprintf(stderr, " - Connect_Remote failed.");
    }
    else
    {
      Send_BeginLogin(UIN, &passwd[0], our_ip, our_port);
    }
    return;
  }

  if(sok != 0) Send_ChangeStatus(new_status);
}


void ICQ_Keep_Alive()
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Keep_Alive");
#endif

  Send_KeepAlive();
}

/* Called with timeout = 1000 from SelectServer
  ie, called 100x per second as long we're connecting */
/* timeout is in milliseconds */
void ICQ_Check_Response(guint32 timeout)
{
  struct timeval tv;
  fd_set readfds, writefds;
  int numfds, i;
  int optval;
  int optlen;

/*
#ifdef DEBUG
  fprintf(stderr, ">");
#endif
*/
  numfds = ((sok > tcp_sok) ? sok : tcp_sok);

  tv.tv_sec = 0;
  tv.tv_usec = timeout;

  FD_ZERO(&readfds);
  FD_ZERO(&writefds);
  FD_SET(sok, &readfds);
  FD_SET(tcp_sok, &readfds);
  
  for(i = 0; i < Num_Contacts; i++)
  {
    if(Contacts[i].sok > 0)
    {
      FD_SET(Contacts[i].sok, &readfds);
      FD_SET(Contacts[i].sok, &writefds);
      if(Contacts[i].sok > numfds) numfds = Contacts[i].sok;
    }
  }

  /* don't care about exceptfds: */
  select(numfds + 1, &readfds, &writefds, NULL, &tv);

  if(FD_ISSET(sok, &readfds))
    Rec_Packet();

  if(FD_ISSET(tcp_sok, &readfds))
    TCP_ReadPacket(tcp_sok);
 
  for(i = 0; i < Num_Contacts; i++)
  {
    if(Contacts[i].sok > 0 && Contacts[i].connected == 0 && 
      FD_ISSET(Contacts[i].sok, &writefds))
    {
      optlen = sizeof(int);
      if(getsockopt(Contacts[i].sok, SOL_SOCKET, SO_ERROR, &optval, &optlen) < 0)
        optval = errno;

      if(!optval)
      {
        if(Verbose & ICQ_VERB_INFO)
          printf("TCP connection established\n");

        Contacts[i].connected = 1;
        TCP_SendHelloPacket(Contacts[i].sok);
        TCP_SendMessages(i);
      }
      else
      {
        if(Verbose & ICQ_VERB_INFO)
          printf("TCP connection failed\n");

        Contacts[i].connected = -1;  /* Connection failed */
        Contacts[i].sok = -1L;
        UDP_SendMessages(i);
      }
    }

    if(Contacts[i].sok > 0 && FD_ISSET(Contacts[i].sok, &readfds))
      TCP_ReadPacket(Contacts[i].sok);
  }
}


void ICQ_Register_Callback(gint type, CALLBACK func)
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Register_Callback");
#endif

  event[type] = func;
}


void ICQ_Send_Message(guint32 uin, gchar* text)
{
  MESSAGE_DATA_PTR msg;
  int i;

#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Send_Message");
#endif

  for(i = 0; i < Num_Contacts; i++)
    if(Contacts[i].uin == uin) break;
  
  if(i == Num_Contacts)
  {
    fprintf(stderr, " - ERR: Cannot send message; bad UIN???\n");
    return;
  }

  if(Contacts[i].connected == 1)
  {
    if(!TCP_SendMessage(uin, text))
      Send_Message(uin, text);  /* fall back on UDP send */
  }
  else if(Contacts[i].connected < 0 || Contacts[i].status == STATUS_OFFLINE)
  {
    Send_Message(uin, text);
  }
  else
  {
    msg = g_malloc(sizeof(MESSAGE_DATA));

    msg->type = MSG_MESS;
    msg->text = strdup(text);
    msg->url = NULL;

    Contacts[i].messages = g_list_append(Contacts[i].messages, (gpointer)msg);
    Contacts[i].sok = TCP_Connect(Contacts[i].current_ip, Contacts[i].port);
  }
}

void ICQ_Send_URL(guint32 uin, gchar* url, gchar* text)
{
  MESSAGE_DATA_PTR msg;
  int i;

#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Send_URL");
#endif

  for(i = 0; i < Num_Contacts; i++)
    if(Contacts[i].uin == uin) break;

  if(i == Num_Contacts)
  {
    fprintf(stderr, " - ERR: Cannot send message; bad UIN???");
    return;
  }

  if(Contacts[i].connected == 1)
  {
    if(!TCP_SendURL(uin, url, text))
      Send_URL(uin, url, text);
  }
  else if(Contacts[i].connected < 0 || Contacts[i].status == STATUS_OFFLINE)
  {
    Send_URL(uin, url, text);
  }
  else
  {
    msg = g_malloc(sizeof(MESSAGE_DATA));

    msg->type = URL_MESS;
    msg->text = strdup(text);
    msg->url = strdup(url);

    Contacts[i].messages = g_list_append(Contacts[i].messages, (gpointer)msg);
    Contacts[i].sok = TCP_Connect(Contacts[i].current_ip, Contacts[i].port);
  }
}

void ICQ_Get_Away_Message(guint32 uin)
{
  MESSAGE_DATA_PTR msg;
  int i;

#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Get_Away_Message(%ld)", uin);
#endif

  for(i = 0; i < Num_Contacts; i++)
    if(Contacts[i].uin == uin) break;

  if(i == Num_Contacts) return;

  if(Contacts[i].connected >= 1)
  {
    TCP_GetAwayMessage(uin);
  }
  else if(Contacts[i].connected == -1)
  {
    printf("ICQ_Get_Away_Message(): Connection timed out\n");
  }
  else
  {
    msg = g_malloc(sizeof(MESSAGE_DATA));

    msg->type = AWAY_MESS;
    msg->text = NULL;
    msg->url = NULL;

    Contacts[i].messages = g_list_append(Contacts[i].messages, (gpointer)msg);
    Contacts[i].sok = TCP_Connect(Contacts[i].current_ip, Contacts[i].port);
  }
}


void ICQ_Search(gchar *email, gchar *nick, gchar* first, gchar* last)
{
  /* Let's add some error checking, so that libicq doesn't accept
    NULLs in all fields, etc. */
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Search(%s, %s, %s, %s)", email, nick, first, last);
#endif

  Send_SearchRequest(email, nick, first, last);
}


void ICQ_Get_Info(guint32 uin)
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Get_Info(%ld)", uin);
#endif

  Send_InfoRequest(uin);
}


void ICQ_Get_Ext_Info(guint32 uin)
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Get_Info(%ld)", uin);
#endif

  Send_ExtInfoRequest(uin);
}


void ICQ_Add_User(guint32 uin, gchar* name)
{
  int cindex;

#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Add_User(%ld, %s)", uin, name);
#endif

  for(cindex = 0; cindex < Num_Contacts; cindex++)
    if(Contacts[cindex].uin == uin) break;

  if(cindex == Num_Contacts)
  {
    Contacts[Num_Contacts].uin = uin;
    Contacts[Num_Contacts].status = STATUS_OFFLINE;
    Contacts[Num_Contacts].last_time = -1L;
    Contacts[Num_Contacts].current_ip = -1L;
    Contacts[Num_Contacts].port = 0;
    Contacts[Num_Contacts].sok = -1L;
    Contacts[Num_Contacts].connected = 0;
    memcpy(Contacts[Num_Contacts].nick, name, sizeof(Contacts->nick));
    Num_Contacts++;
  }
  else
  {
    Contacts[cindex].status = STATUS_OFFLINE;
    Contacts[cindex].current_ip = -1L;
    Contacts[cindex].port = 0;
    if(Contacts[cindex].sok)
      close(Contacts[cindex].sok);
    Contacts[cindex].sok = -1L;
    Contacts[cindex].connected = 0;
  }

  Send_ContactList();
  Write_Contacts_RC(contacts_rc);
}


void ICQ_Rename_User(guint32 uin, gchar* name)
{
  int x;
  
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Rename_User(%ld, %s)", uin, name);
#endif

  for(x = 0; x < Num_Contacts; x++)
    if(Contacts[x].uin == uin) break;
    
  if(x >= Num_Contacts) return;
  
  memcpy(Contacts[x].nick, name, sizeof(Contacts->nick));
  Write_Contacts_RC(contacts_rc);
}


void ICQ_Delete_User(guint32 uin)
{
  int x;
  
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Delete_User(%ld)", uin);
#endif

  for(x = 0; x < Num_Contacts; x++)
    if(Contacts[x].uin == uin) break;
    
  if(x >= Num_Contacts) return;
  
  Contacts[x].uin = 0;
/*  if(Contacts[x].sok > 0)
  {
    set_nonblock(Contacts[x].sok);
    close(Contacts[x].sok);
  } */
  Contacts[x].sok = -1;
  Contacts[x].connected = 0;

  Write_Contacts_RC(contacts_rc);
}

void ICQ_Debug(gint debug)
{
#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> ICQ_Debug(%d)", debug);
#endif

  if((debug >= ICQ_VERB_NONE) && (debug <= ICQ_VERB_MAX))
    Verbose = debug;
  else
    fprintf(stderr, " - Bad debug value: %d", debug);
}


void TCP_SendMessages(gint i)
{
  GList* msg;
  MESSAGE_DATA_PTR curr_msg;

  #ifdef DEBUG
  fprintf(stderr, "\n LIBICQ> TCP_SendMessages");
  #endif
  while(msg = g_list_first(Contacts[i].messages))
  {
    curr_msg = ((MESSAGE_DATA_PTR)(msg->data));

    if(curr_msg->type == MSG_MESS)
      TCP_SendMessage(Contacts[i].uin, curr_msg->text);
    else if(curr_msg->type == URL_MESS)
      TCP_SendURL(Contacts[i].uin, curr_msg->url, curr_msg->text);
    else if(curr_msg->type == AWAY_MESS)
      TCP_GetAwayMessage(Contacts[i].uin);

    g_free(curr_msg->text);
    g_free(curr_msg->url);
    g_free(curr_msg);

    Contacts[i].messages = g_list_remove_link(Contacts[i].messages, 
      g_list_first(Contacts[i].messages));
  }
}

void UDP_SendMessages(gint i)
{
  GList* msg;
  MESSAGE_DATA_PTR curr_msg;

#ifdef DEBUG
  fprintf(stderr, "\nLIBICQ> UDP_SendMessages");
#endif

  while(msg = g_list_first(Contacts[i].messages))
  {
    curr_msg = ((MESSAGE_DATA_PTR)(msg->data));

    if(curr_msg->type == MSG_MESS)
      Send_Message(Contacts[i].uin, curr_msg->text);
    else if(curr_msg->type == URL_MESS)
      Send_URL(Contacts[i].uin, curr_msg->url, curr_msg->text);

    g_free(curr_msg->text);
    g_free(curr_msg->url);
    g_free(curr_msg);

    Contacts[i].messages = g_list_remove_link(Contacts[i].messages, 
      g_list_first(Contacts[i].messages));
  }
}

