/* $Id: visual.c,v 1.2 1998/12/24 00:51:12 marcus Exp $
******************************************************************************

   Graphics library for GGI. Handles visuals.

   Copyright (C) 1997 Jason McMullan	[jmcc@ggi-project.org]
  
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>

#include <ggi/internal/internal.h>

/*
******************************************************************************
 Default functions for op-structs
******************************************************************************
*/

static int _default_error(void)
{
        DPRINT_ALL("_default_error() called\n");
	return -1;
}

static int _default_zero(void)
{
	return 0;
}

/*
******************************************************************************
 Memory allocation
******************************************************************************
*/

void _ggi_mem_error(void) {
	ggiPanic("LibGGI is out of memory!\n");
}

void *_ggi_malloc(size_t siz)
{
	void *mem=calloc(1,siz);
	
	if (mem == NULL) {
		_ggi_mem_error();
	}
	return mem;
}

void *_ggi_calloc(size_t siz)
{
	void *mem=calloc(1, siz);
	
	if (mem == NULL) {
		_ggi_mem_error();
	}
	return mem;
}

void *_ggi_realloc(void *ptr, size_t siz)
{
	void *mem=realloc(ptr, siz);
	
	if (mem == NULL) {
		_ggi_mem_error();
	}
	return mem;
}

/*
******************************************************************************
 Allocate space for helper private data
******************************************************************************
*/

static uint32 _ggi_drvpriv_inuse = 0;

int _ggi_alloc_drvpriv(void)
{
	int idx, v;
	
	for (idx = 0, v = 1; idx < _GGI_NROF_HELPERS; idx++, v<<=1) {
		if ((_ggi_drvpriv_inuse & v) == 0) {
			_ggi_drvpriv_inuse |= v;
			return idx;
		}
	}

	return -1;
}

void _ggi_free_drvpriv(int idx)
{
	_ggi_drvpriv_inuse &= ~(1<<idx);
}

/*
******************************************************************************
 Creation and destruction of visuals
******************************************************************************
*/

typedef int   (*__simp_int)(void);

static void _ggiZapOpDisplay(ggi_visual *vis)
{
	__simp_int *funcarr;	
	int i;

	if (vis->opdisplay == NULL) {
		vis->opdisplay = 
			_ggi_malloc(sizeof(struct ggi_visual_opdisplay));
	} else {
		_ggiCloseDL(vis,GGI_DL_OPDISPLAY);
	}

	vis->opdisplay->version=GGI_VERSION_VISUAL_OPDISPLAY;
	vis->opdisplay->dlhandle=NULL;

	/** Display **/
	funcarr=(__simp_int *)(&vis->opdisplay->null);
	for (i=1;i<=GGI_OPDISPLAY_NUMFUNCS;i++)
		funcarr[i]=_default_error;
}

static void _ggiZapOpColor(ggi_visual *vis)
{
	__simp_int *funcarr;	
	int i;

	if (vis->opcolor == NULL) {
		vis->opcolor = 
			_ggi_malloc(sizeof(struct ggi_visual_opcolor));
	} else {
		_ggiCloseDL(vis,GGI_DL_OPCOLOR);
	}

	vis->opcolor->version=GGI_VERSION_VISUAL_OPCOLOR;
	vis->opcolor->dlhandle=NULL;

	/** Color **/
	funcarr=(__simp_int *)(&vis->opcolor->null);
	for (i=1;i<=GGI_OPCOLOR_NUMFUNCS;i++)
		funcarr[i]=_default_error;
	vis->opcolor->mapcolor=(ggi_pixel (*)(ggi_visual *, ggi_color *)) _default_zero;
}

static void _ggiZapOpDraw(ggi_visual *vis)
{
	__simp_int *funcarr;	
	int i;

	if (vis->opdraw == NULL) {
		vis->opdraw = 
			_ggi_malloc(sizeof(struct ggi_visual_opdraw));
	} else {
		_ggiCloseDL(vis,GGI_DL_OPDRAW);
	}

	vis->opdraw->version=GGI_VERSION_VISUAL_OPDRAW;
	vis->opdraw->dlhandle=NULL;
	
	/** 2D **/
	funcarr=(__simp_int *)(&vis->opdraw->null);
	for (i=1;i<=GGI_OPDRAW_NUMFUNCS;i++)
		funcarr[i]=_default_error;

	/* Exceptions...
	 */
}

static void _ggiZapOpGC(ggi_visual *vis)
{
	__simp_int *funcarr;	
	int i;

	if (vis->opgc == NULL) {
		vis->opgc = 
			_ggi_malloc(sizeof(struct ggi_visual_opgc));
	} else {
		_ggiCloseDL(vis,GGI_DL_OPGC);
	}

	vis->opgc->version=GGI_VERSION_VISUAL_OPGC;
	vis->opgc->dlhandle=NULL;
	
	/** GC **/
	funcarr=(__simp_int *)(&vis->opgc->null);
	for (i=1;i<=GGI_OPGC_NUMFUNCS;i++)
		funcarr[i]=NULL;
}

void _ggiZapMode(ggi_visual *vis,ggi_uint type)
{
	if (type & GGI_DL_OPDISPLAY)
		_ggiZapOpDisplay(vis);

	if (type & GGI_DL_OPCOLOR)
		_ggiZapOpColor(vis);

	if (type & GGI_DL_OPDRAW)
		_ggiZapOpDraw(vis);

	if (type & GGI_DL_OPGC)
		_ggiZapOpGC(vis);
}

ggi_visual *_ggiNewVisual(void)
{
	ggi_visual *vis;

	if ((vis = (ggi_visual *)malloc(sizeof(ggi_visual))) == NULL) {
		return NULL;
	}

	if ((vis->mutex = ggLockCreate()) == NULL) {
		free(vis);
		return NULL;
	}

	vis->version=GGI_VERSION_VISUAL;
	vis->numknownext=0;
	vis->extlist=NULL;
	
	LIBGGI_MODE(vis)=(ggi_mode *)_ggi_calloc(sizeof(ggi_mode));
	LIBGGI_PIXFMT(vis)=_ggi_calloc(sizeof(ggi_pixelformat));
	LIBGGI_APPLIST(vis)=_ggi_calloc(sizeof(ggi_db_list));
	LIBGGI_PRIVLIST(vis)=_ggi_calloc(sizeof(ggi_db_list));
	LIBGGI_APPLIST(vis)->num = LIBGGI_PRIVLIST(vis)->num = 0;
	LIBGGI_APPLIST(vis)->first_targetbuf
		= LIBGGI_PRIVLIST(vis)->first_targetbuf = -1;
	LIBGGI_APPBUFS(vis) = LIBGGI_PRIVBUFS(vis) = NULL;

	LIBGGI_FLAGS(vis)=0;
	LIBGGI_FD(vis) = -1;

	LIBGGI_DLHANDLE(vis) = NULL;
	vis->extlib = NULL;

	vis->opdisplay = NULL;
	vis->opcolor = NULL;
	vis->opdraw = NULL;
	vis->opgc = NULL;

	vis->d_frame_num = vis->r_frame_num = vis->w_frame_num = 0;
	vis->r_frame = vis->w_frame = NULL;
	vis->origin_x = vis->origin_y = 0;
	vis->gamma_red = vis->gamma_green = vis->gamma_blue = 1.0;
	vis->palette = NULL;
	vis->input = NULL;

	_ggiZapMode(vis,~0);

	return vis;
}

void _ggiDestroyVisual(ggi_visual *vis)
{
	_ggiCloseDL(vis,~0);
	if (vis->input) {
		giiClose(vis->input);
	}
	free(vis->opdisplay);
	free(vis->opcolor);
	free(vis->opdraw);
	free(vis->opgc);
	free(LIBGGI_MODE(vis));
	free(LIBGGI_PIXFMT(vis));
	free(LIBGGI_APPLIST(vis));
	free(LIBGGI_PRIVLIST(vis));
	ggLockDestroy(vis->mutex);
	free(vis);
}
