/* $Id: init.c,v 1.3 1998/12/17 02:08:59 marcus Exp $
******************************************************************************

   LibGGI initialization.

   Copyright (C) 1997 Jason McMullan	[jmcc@ggi-project.org]
   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]
  
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#include <ggi/internal/internal.h>
#include <ggi/gg.h>

/* Global variables */
uint32                _ggiDebugState = 0;
int                   _ggiDebugSync = 0;
void                 *_ggiConfigHandle;

/* Static variables */
static struct {
	void		*mutex;		/* Lock when changing.. */
	int		 visuals;	/* # of visuals active */
	ggi_visual	*visual;	/* Linked list of all visuals */
} _ggiVisuals;	/* This is for remembering visuals. */

static int 	      _ggiLibIsUp=0;
static char           ggiconfstub[512]=GGICONFFILE;
static char          *ggiconffile=ggiconfstub+GGITAGLEN;
static int            numextensions=0;
static ggi_extension *_ggiExtension;

/*
 * Initalize the strutures for the library
 */

int ggiInit(void)
{
	int err;
	const char *str;
	
	_ggiLibIsUp++;
	if (_ggiLibIsUp>1) return 0;	/* Initialize only at first call. */

	err = giiInit();
	if (err) {
		fprintf(stderr, "LibGGI: unable to initialize LibGII\n");
		return err;
	}

	if ((_ggiVisuals.mutex = ggLockCreate()) == NULL) {
		fprintf(stderr, "LibGGI: unable to initialize core mutex.\n");
		giiExit();
		return GGI_EUNKNOWN;
	}
	_ggiVisuals.visuals = 0;
	_ggiVisuals.visual = NULL;

	str=getenv("GGI_DEBUG");
	if (str!=NULL) {
		_ggiDebugState=atoi(str);
		DPRINT_CORE("Debugging=%d\n",_ggiDebugState);
	}

	str=getenv("GGI_DEBUGSYNC");
	if (str!=NULL) {
		_ggiDebugSync=1;
	}
		
	str=getenv("GGI_DEFMODE");
	if (str!=NULL) {
		_ggiSetDefaultMode(str);
	}

	err = ggLoadConfig(ggiconffile, &_ggiConfigHandle);
	if (err == GGI_OK)
		return GGI_OK;

	fprintf(stderr,"LibGGI: couldn't open %s.\n", ggiconffile );

	ggLockDestroy(_ggiVisuals.mutex);
	giiExit();
	_ggiLibIsUp--;

	return err;
}

int ggiExit(void)
{
	ggi_extension *tmp,*next;

	DPRINT_CORE("ggiExit called\n");
	if (!_ggiLibIsUp)
		return -1;

	if (_ggiLibIsUp > 1) {
		_ggiLibIsUp--;
		return _ggiLibIsUp;
	}

	DPRINT_CORE("ggiExit: really destroying.\n");
	while (_ggiVisuals.visual!=NULL) 
		ggiClose(_ggiVisuals.visual);

	ggLockDestroy(_ggiVisuals.mutex);
	
	for (tmp=_ggiExtension;tmp;tmp=next)
	{
		next=tmp->next;
		free(tmp);
	}

	ggFreeConfig(_ggiConfigHandle);
	giiExit();
	_ggiLibIsUp=0;

	DPRINT_CORE("ggiExit: done!\n");
	return 0;
}

void ggiPanic(const char *format,...)
{
	DPRINT_CORE("ggiPanic called\n");
	fprintf(stderr,"%s",format);
	fflush(stderr);
	while(ggiExit()>0);	/* kill all instances ! */
	exit(1);
}


/* Opens a visual.
 */
ggi_visual *ggiOpen(const char *driver,...)
{
	va_list drivers;
	ggi_visual *vis;
	char *cp, *inplist;
	char str[1024];
	char target[1024];
	int  success=0;
	void *argptr;
	static int globalopencount=0;

	if (!_ggiLibIsUp)
		return NULL;

	if (driver==NULL) {
		void *ret;
		
		/* If GGI_DISPLAY is set, use it. Fall back to "auto" 
		 * otherwise.
		 */

		cp=getenv("GGI_DISPLAY");
		if (cp!=NULL) {
			ret = ggiOpen(cp,NULL);
			return ret;
		}
		driver="auto";
	}
	if (strcmp(driver,"auto")==0) {

		void *ret;
		
		/* Try the X display.. */
		cp=getenv("DISPLAY");
		if (cp!=NULL) {
			strcpy(str,"display-x:");
			strcat(str,cp);
			ret = ggiOpen(str,NULL);
			if (ret != NULL)
				return ret;
		}

#if 0
		/* Try the KGI console.. */
		ret=ggiOpen("display-KGI:/dev/graphic",NULL);
		if (ret != NULL)
			return ret;
#endif
		
		/* Try the framebuffer console.. */
		ret = ggiOpen("display-fbdev", NULL);
		if (ret != NULL)
			return ret;

		/* Try svgalib target.. */
		ret = ggiOpen("display-svga",NULL);
		if (ret != NULL)
			return ret;

		/* Try AAlib target.. */
		ret = ggiOpen("display-aa",NULL);
		if (ret != NULL)
			return ret;

		/* This fallthrough is for a smooth transition, when someone 
		 * makes a display-auto target :
		 */
	}

	DPRINT_CORE("ggiOpen(\"%s\") called\n", driver);
	
	if ((vis = _ggiNewVisual()) == NULL) {
		return NULL;
	}

	va_start(drivers,driver);

	argptr=va_arg(drivers,void *);
	va_end(drivers);

	DPRINT_CORE("Loading driver %s\n",driver);

	do {
		if (ggParseTarget((char *)driver,target,1024) == NULL) {
			break;
		}

		if (strlen(target) == 0) {
			fprintf(stderr, "LibGGI: Missing target descriptor !\n");
			break;
		}
		
		cp=strchr(target, ':');

		if (cp != NULL) {
			*cp++ = 0;
		}
		if (_ggiOpenDL(vis,target,cp,argptr) != NULL) {
			success=1;
		}

	} while (0);
	

	va_end(drivers);

	if (success) {
		ggLock(_ggiVisuals.mutex);
		vis->next=_ggiVisuals.visual;
		_ggiVisuals.visual=vis;
		_ggiVisuals.visuals++;
		ggUnlock(_ggiVisuals.mutex);
		DPRINT_CORE("ggiOpen: returning %p\n", vis);
	} else {
		_ggiDestroyVisual(vis);
		DPRINT_CORE("ggiOpen: failure\n");
		return NULL;
	}

	DPRINT_CORE("Loading extra inputs for %s\n",driver);

	inplist=NULL;

	sprintf(str,"GGI_INPUT_%s_%d",target,++globalopencount);
	if (!inplist) { 
		inplist = getenv(str);
		DPRINT_CORE("Checking %s : %s\n",str,inplist);
	}

	sprintf(str,"GGI_INPUT_%s",target);
	if (!inplist) {
		inplist = getenv(str);
		DPRINT_CORE("Checking %s : %s\n",str,inplist);
	}

	strcpy(str,"GGI_INPUT");
	if (!inplist) {
		inplist = getenv(str);
		DPRINT_CORE("Checking %s : %s\n",str,inplist);
	}

	if (inplist) for (;;) {

		gii_input *inp;

		inplist = ggParseTarget(inplist, target, 1024);

		if (inplist == NULL) {
			break;
		}

		if (strlen(target) == 0) {
			fprintf(stderr,"LibGGI: Missing input descriptor !\n");
			break;
		}
		
		inp = giiOpen(target, NULL);

		if (inp == NULL) {
			fprintf(stderr, "LibGGI: failed to load input: %s\n",
				target);
		} else {
			vis->input = giiJoinInputs(vis->input, inp);
		}

		while (*inplist != '\0' && isspace((int) *inplist)) {
			inplist++;
		}

		if (*inplist != ':') {
			break;
		}

		inplist++;   /* skip ':' */
	}

	DPRINT_CORE("Loading input filters for %s\n",driver);

	inplist=NULL;

	sprintf(str,"GGI_FILTER_%s_%d",target,globalopencount);
	if (!inplist) {
		inplist = getenv(str);
		DPRINT_CORE("Checking %s : %s\n",str,inplist);
	}

	sprintf(str,"GGI_FILTER_%s",target);
	if (!inplist) {
		inplist = getenv(str);
		DPRINT_CORE("Checking %s : %s\n",str,inplist);
	}

	strcpy(str,"GGI_FILTER");
	if (!inplist) {
		inplist = getenv(str);
		DPRINT_CORE("Checking %s : %s\n",str,inplist);
	}

	if (inplist) for (;;) {

		gii_input *inp;

		inplist = ggParseTarget(inplist, target, 1024);

		if (inplist == NULL) {
			break;
		}

		if (strlen(target) == 0) {
			fprintf(stderr,"LibGGI: Missing filter descriptor !\n");
			break;
		}
		
		inp = giiOpen(target, vis->input);

		if (inp == NULL) {
			fprintf(stderr, "LibGGI: failed to load filter: %s\n",
				target);
		} else {
			vis->input = inp;
		}

		while (*inplist != '\0' && isspace((int) *inplist)) {
			inplist++;
		}

		if (*inplist != ':') {
			break;
		}

		inplist++;   /* skip ':' */
	}

	if (vis->input == NULL) {
		/* Add dummy input source so we can use sendevent */
		vis->input = giiOpen("null", NULL);
		if (vis->input == NULL) {
			/* Something is wrong here - bail out */
			ggiClose(vis);
			return NULL;
		}
	}
	
	return vis;
}
	
/* ggiClose
 *	Closes the requested visual
 *      Returns 0 on success, < 0 on error
 */
int ggiClose(ggi_visual *visual)
{
	ggi_visual *vis,*pvis=NULL;

	DPRINT_CORE("ggiClose(\"%p\") called\n", visual);

	if (!_ggiLibIsUp)
		return -1;

	DPRINT_CORE("ggiClose: closing\n");

	for (vis = _ggiVisuals.visual; vis != NULL; pvis=vis, vis=vis->next) {
		if (vis == visual) break;
	}

	if (vis == NULL) return -1;

	ggLock(_ggiVisuals.mutex);

	if (pvis == NULL) _ggiVisuals.visual = vis->next;
	else pvis->next = vis->next;

	_ggiVisuals.visuals--;
	
	ggUnlock(_ggiVisuals.mutex);

	_ggiDestroyVisual(vis);

	DPRINT_CORE("ggiClose: done!\n");
	return 0;
}

/*
 * Extension handling.
 */

/*
 * Register an Extension for usage. It will return an extension ID 
 * (ggi_extid) that can be used to address this extension.
 */
ggi_extid ggiExtensionRegister(char *name,int size,
			int (*change)(ggi_visual_t,int whatchanged))
{
	ggi_extension *tmp,*ext;

	DPRINT_CORE("ggiExtensionRegister(\"%s\", %d, %p) called\n",
		    name, size, change);
	if (_ggiExtension)
		for (tmp=_ggiExtension;tmp;tmp=tmp->next)
			if (strcmp(tmp->name,name)==0)
			{
				tmp->initcount++;
				DPRINT_CORE("ggiExtensionRegister: accepting copy #%d of extension %s\n",
				       tmp->initcount,tmp->name);
				return tmp->id;
			}

	ext=_ggi_malloc(sizeof(ggi_extension));

	ext->size=size;
	ext->paramchange=change;
	ext->next=NULL;
	ext->initcount=1;
	strncpy(ext->name,name,sizeof(ext->name));
	ext->name[sizeof(ext->name)-1]='\0';	/* Make sure it terminates */
	
	if (_ggiExtension) {
		for (tmp=_ggiExtension;tmp->next;tmp=tmp->next) ;
		tmp->next=ext;
		ext->prev=tmp;			
	} else {
		_ggiExtension=ext;
		ext->prev=NULL;
	}

	DPRINT_CORE("ggiExtensionRegister: installing first copy of extension %s\n", name);
	return ext->id=numextensions++;
}

/*
 * Unregister an Extension.It takes the ggi_extid gotten from
 * ggiExtensionRegister. It disallows further calls to 
 * ggiExtensionAttach (for that extid) and frees memory for the
 * extension registry. 
 * It dos NOT automatically ggiExtensionDetach it from all visuals.
 */
int ggiExtensionUnregister(ggi_extid id)
{
	ggi_extension *tmp;

	DPRINT_CORE("ggiExtensionUnregister(%d) called\n", id);
	if (_ggiExtension)
		for (tmp=_ggiExtension;tmp;tmp=tmp->next)
			if (tmp->id==id)
			{
				if (--tmp->initcount) {
					DPRINT_CORE("ggiExtensionUnregister: removing #%d copy of extension %s\n",tmp->initcount+1,tmp->name);
					return 0;	/* Removed copy */
				}

				if (tmp->prev==NULL) 
					_ggiExtension=tmp->next;
				else 
					tmp->prev->next=tmp->next;

				if (tmp->next!=NULL) 
					tmp->next->prev=tmp->prev;

				DPRINT_CORE("ggiExtensionUnregister: removing last copy of extension %s\n",tmp->name);

				free(tmp);
				return 0;
			}
	return -1;
}

/*
 * Make an extension available for a given visual.
 * The extension has to be registered for that. 
 * RC: -1 for trying to attach an unkown extension
 *      x for the number of times this extension had already been
 *        registered to that visual. So
 *      0 means you installed the extension as the first one. Note that
 *    >=0 should be regarded as "success". It is legal to attach an 
 *        extension multiple times. You might want to set up private
 *        data if RC==0.
 */
int ggiExtensionAttach(ggi_visual *vis,ggi_extid id)
{
	ggi_extension *tmp=NULL;

	DPRINT_CORE("ggiExtensionAttach(%p, %d) called\n", vis, id);
	if (_ggiExtension)
		for (tmp=_ggiExtension;tmp;tmp=tmp->next)
			if (tmp->id==id)
				break;
	if ( ! tmp )
		return -1;

	if ( vis->numknownext<=id)
	{
		vis->extlist=_ggi_realloc(vis->extlist,sizeof(vis->extlist[0])*(id+1));
		memset(&vis->extlist[vis->numknownext],0,
			sizeof(vis->extlist[0])*(id+1-vis->numknownext));
		vis->numknownext=id+1;
		DPRINT_CORE("ggiExtensionAttach: ExtList now at %p (%d)\n",vis->extlist,vis->numknownext);
	}

	if ( 0==vis->extlist[id].attachcount ) 
		vis->extlist[id].private=_ggi_malloc(tmp->size);

	return vis->extlist[id].attachcount++;
}

/*
 * Destroy an extension for a given visual.
 * The extension need not be registered for that anymore, though the extid
 * has to be valid. 
 * RC: -1 for trying to detach an unkown extension
 *      x for the number of times this extension remains attached. So
 *      0 means you removed the last copy of the extension. Note that
 *    >=0 should be regarded as "success". It is legal to attach an 
 *        extension multiple times.
 */
int ggiExtensionDetach(ggi_visual *vis,ggi_extid id)
{
	DPRINT_CORE("ggiExtensionDetach(%p, %d) called\n", vis, id);
	if ( vis->numknownext<=id ||
	     0==vis->extlist[id].attachcount ) 
	     	return -1;

	if (--vis->extlist[id].attachcount) 
		return vis->extlist[id].attachcount;
	
	free(vis->extlist[id].private);
	vis->extlist[id].private=NULL;	/* Make sure ... */
	return 0;
}

int ggiIndicateChange(ggi_visual_t vis, int whatchanged)
{
	ggi_extension *tmp=NULL;

	DPRINT_CORE("ggiIndicateChange(%p, 0x%x) called\n", vis, whatchanged);

	/* tell all attached extensions on this visual */

	DPRINT_CORE("ggiIndicateChange: %i changed for %p.\n",whatchanged,vis);

	if (_ggiExtension)
		for (tmp=_ggiExtension;tmp;tmp=tmp->next)
			if (tmp->id<vis->numknownext && vis->extlist[tmp->id].attachcount)
				tmp->paramchange(vis,whatchanged);
	return 0;
}

#include <ggi/internal/ggilibinit.h>
