/* $Id: dl.c,v 1.1 1998/12/09 04:27:29 marcus Exp $
******************************************************************************

   Graphics library for GGI. Library extensions dynamic loading.

   Copyright (C) 1997 Jason McMullan	[jmcc@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ggi/internal/internal.h>
#include <ggi/gg.h>

#define GGI_DLINIT_SYM          "GGIdlinit"
#define GGI_DLCLEANUP_SYM       "GGIdlcleanup"

/* Open the dynamic libary requested
 */
ggi_dlhandle *_ggiLoadDL(const char *name, const char *version, int type)
{
	ggi_dlhandle hand,*hp;

	DPRINT_LIBS("_ggiLoadDL(\"%s\",\"%s\", 0x%x) called \n", 
			name, version, type);

	hand.name=NULL;
	hand.use=0;

	if(type & GGI_DLTYPE_GLOBAL)
		hand.handle=ggMLoadModule(_ggiConfigHandle, name, version, GG_MODULE_GLOBAL);
	else
		hand.handle=ggMLoadModule(_ggiConfigHandle, name, version, 0);
	DPRINT_LIBS("hand.handle=%p\n",hand.handle);
	if (hand.handle == NULL) {
		DPRINT_LIBS("Error loading module %s\n", name);
		return NULL;
	}

	hand.init=(int (*)(struct ggi_visual *, const char *,void *))  ggGetSymbolAddress(hand.handle,GGI_DLINIT_SYM);
	hand.cleanup=(int (*)(struct ggi_visual *)) ggGetSymbolAddress(hand.handle,GGI_DLCLEANUP_SYM);

	DPRINT_LIBS("hand.init=%p\n",hand.init);
	DPRINT_LIBS("hand.cleanup=%p\n",hand.cleanup);
	if (hand.init==NULL || hand.cleanup==NULL) {
		ggFreeModule(hand.handle);
		return NULL;
	}

	hp=(ggi_dlhandle *)_ggi_malloc(sizeof(ggi_dlhandle));
	memcpy(hp,&hand,sizeof(ggi_dlhandle));

	return hp;
}

/****** Open and Close a DL *********/
ggi_dlhandle *_ggiAddDL(ggi_visual *vis,const char *driver,
		const char *args,void *argptr,int type)
{
	ggi_dlhandle *dlh;
	ggi_dlhandle_l *tmp;
	int err;

	DPRINT_LIBS("_ggiAddDL(%p, \"%s\", \"%s\", 0x%x) called\n",
		    vis, driver, args, type);
	
	dlh=_ggiLoadDL(driver, NULL, type);
	DPRINT_LIBS("_ggiLoadDL returned %p\n", dlh);
	if (dlh==NULL)
		return NULL;
	
	err=dlh->init(vis,args,argptr);
	DPRINT_LIBS("%d = dlh->init(%p,\"%s\",%p) - %s\n",
		    err, vis, args, argptr, driver);
	if (err & GGI_DL_ERROR) {
		ggFreeModule(dlh->handle);
		free(dlh);
		return NULL;
	}
	dlh->type=type;
	if (type==GGI_DLTYPE_INTERNAL) {
		if (err & GGI_DL_OPDISPLAY) {
			tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
			tmp->handle=dlh;
			tmp->next=vis->opdisplay->dlhandle;
			vis->opdisplay->dlhandle=tmp;
			dlh->use++;
		}
	
		if (err & GGI_DL_OPCOLOR) {
			tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
			tmp->handle=dlh;
			tmp->next=vis->opcolor->dlhandle;
			vis->opcolor->dlhandle=tmp;
			dlh->use++;
		}

		if (err & GGI_DL_OPDRAW) {
			tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
			tmp->handle=dlh;
			tmp->next=vis->opdraw->dlhandle;
			vis->opdraw->dlhandle=tmp;
			dlh->use++;
		}

		if (err & GGI_DL_OPGC) {
			tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
			tmp->handle=dlh;
			tmp->next=vis->opgc->dlhandle;
			vis->opgc->dlhandle=tmp;
			dlh->use++;
		}
	} else {
		dlh->use=1;
		tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
		tmp->handle=dlh;
		tmp->next=vis->extlib;
		vis->extlib=tmp;
	}

	if (dlh->use==0) {
		fprintf(stderr, "LibGGI: %s (%s) -> 0x%.8x - no operations in this library\n",driver,args,err);
		ggFreeModule(dlh->handle);
		free(dlh);
		return NULL;
	} else {
		tmp=(ggi_dlhandle_l *)_ggi_malloc(sizeof(ggi_dlhandle_l));
		tmp->handle=dlh;
		tmp->next=LIBGGI_DLHANDLE(vis);
		LIBGGI_DLHANDLE(vis)=tmp;
	}

	dlh->name=strdup(driver);

	return dlh;
}

ggi_dlhandle *_ggiOpenDL(ggi_visual *vis,const char *driver,const char *args,void *argptr)
{
	return _ggiAddDL(vis,driver,args,argptr,GGI_DLTYPE_INTERNAL);
}

void _ggiRemoveDL(ggi_visual *vis,ggi_dlhandle_l **lib)
{
	ggi_dlhandle_l *tmp,**prev;
	ggi_dlhandle_l *libtmp,**libprev,*libnext;

	for (libprev=lib,libtmp=*lib;libtmp!=NULL; libtmp=libnext) {
		libnext=libtmp->next;
		if (libtmp->handle->use<=0) {
			DPRINT_LIBS("Disposing \"%s\"\n",libtmp->handle->name);
			*libprev=libtmp->next;
			libtmp->handle->cleanup(vis);
			DPRINT_LIBS("Closing handle: 0x%x\n", libtmp->handle->handle);
			ggFreeModule(libtmp->handle->handle);
			
			/* Now, clean up the master visual */
			prev=&LIBGGI_DLHANDLE(vis);
			for (tmp=LIBGGI_DLHANDLE(vis);tmp;tmp=tmp->next) {
				if (tmp->handle==libtmp->handle) break;
				prev=&tmp->next;
			}
			if (!tmp) DPRINT_LIBS("Error: handle not in master list.\n");
			*prev=tmp->next;
			free(tmp);

			free(libtmp->handle->name);
			free(libtmp->handle);
			free(libtmp);
		} else {
			libprev=&libtmp->next;
		}
	}
}

static void _ggiZapDL(ggi_visual *vis,ggi_dlhandle_l **lib)
{
	ggi_dlhandle_l *tmp,*next;

	DPRINT_LIBS("_ggiZapDL(%p, %p) called\n", vis, lib);

	for (tmp=*lib;tmp;tmp=tmp->next) 
		tmp->handle->use--;

	_ggiRemoveDL(vis,lib);

	for (tmp=*lib;tmp;tmp=next) {
		next=tmp->next;
		free(tmp);
	}

	*lib=NULL;
}

void _ggiCloseDL(ggi_visual *vis,ggi_uint type)
{
	LIBGGI_ASSERT(vis != NULL, "vis == NULL");
	
	if ((type & GGI_DL_OPDISPLAY)
	    && vis->opdisplay
	    && vis->opdisplay->dlhandle) {
		_ggiZapDL(vis,&vis->opdisplay->dlhandle);
	}

	if ((type & GGI_DL_OPCOLOR)
	    && vis->opcolor
	    && vis->opcolor->dlhandle) {
		_ggiZapDL(vis,&vis->opcolor->dlhandle);
	}

	if ((type & GGI_DL_OPDRAW)
	    && vis->opdraw
	    && vis->opdraw->dlhandle) {
		_ggiZapDL(vis,&vis->opdraw->dlhandle);
	}

	if ((type & GGI_DL_OPGC)
	    && vis->opgc
	    && vis->opgc->dlhandle) {
		_ggiZapDL(vis,&vis->opgc->dlhandle);
	}

	if ((type & GGI_DL_EXTENSION)
	    && vis->extlib) {
		_ggiZapDL(vis,&vis->extlib);
	}

}
