/* $Id: visual.c,v 1.9 1998/12/20 21:30:55 marcus Exp $
******************************************************************************

   Display-VCSA: visual management

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/vt.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/vcsa.h>


#define VCSA_OPTIONS  4

static gg_option vcsa_options[VCSA_OPTIONS] =
{
	{ "noinput", "no" },
	{ "nokbd",   "no" },
	{ "nomouse", "no" },
	{ "ascii",   "no" }
};

static int which_console(void)
{
	int fd; 
	struct vt_stat vtinfo;

	fd = open("/dev/tty", O_RDWR);

	if (fd < 0) {
		perror("display-vcsa: cannot open /dev/tty");
		return -1;
	}

	if (ioctl(fd, VT_GETSTATE, &vtinfo) != 0) {
		perror("display-vcsa: VT_GETSTATE failed");
		fprintf(stderr, "display-vcsa: (You need to be running "
			"directly on a virtual console).\n");
		close(fd);
		return -1;
	}
	
	close(fd);

	return vtinfo.v_active;
}

int GGIdlinit(ggi_visual *vis, const char *args,void *argptr)
{
	VcsaHook *vv;
	char filename[80];
	unsigned char buf[4];
	int vt;
	gii_input *inp;

	DPRINT("display-vcsa: GGIdlinit start.\n");

	vv = LIBGGI_PRIVATE(vis) = _ggi_malloc(sizeof(VcsaHook));
	
	LIBGGI_GC(vis) = _ggi_malloc(sizeof(ggi_gc));


	if (getenv("GGI_VCSA_OPTIONS") != 0) {
		if (ggParseOptions(getenv("GGI_VCSA_OPTIONS"),
			vcsa_options, VCSA_OPTIONS) == NULL) {
			
			fprintf(stderr, "display-vcsa: error in "
				"$GGI_VCSA_OPTIONS.\n");
			free(vv);
			return GGI_DL_ERROR;
		}
	}
	
	if (args) {
		args = ggParseOptions((char *)args, vcsa_options,
				VCSA_OPTIONS);

		if (args == NULL) {
			
			fprintf(stderr, "display-vcsa: error in "
				"arguments.\n");
			free(vv);
			return GGI_DL_ERROR;
		}
	}


	/* handle args */
	
	filename[0] = 0;
	while (args && *args && isspace(*args)) {
		args++;
	}

	if (args && *args) {
		filename[79] = 0;
		strncpy(filename, args, 79);
	}
	
	vv->inputs = VCSA_INP_KBD | VCSA_INP_MOUSE;

	if (toupper(vcsa_options[0].result[0]) != 'N') {
		vv->inputs = 0;
	}
	if (toupper(vcsa_options[1].result[0]) != 'N') {
		vv->inputs &= ~VCSA_INP_KBD;
	}
	if (toupper(vcsa_options[2].result[0]) != 'N') {
		vv->inputs &= ~VCSA_INP_MOUSE;
	}
	
	vv->flags = 0;

	if (toupper(vcsa_options[3].result[0]) != 'N') {
		vv->flags |= VCSA_FLAG_ASCII;
	}

	/* work out which console we're on */
	if (filename[0] == 0) {
		vt = which_console();
		if (vt < 0) {
			return GGI_DL_ERROR;
		}
		sprintf(filename, "/dev/vcsa%d", vt);
	}
	
	/* now open the vcsa device */
	DPRINT("display-vcsa: Using file `%s'.\n", filename);

	LIBGGI_FD(vis) = open(filename, O_RDWR);

	if (LIBGGI_FD(vis) < 0) {
		perror("display-vcsa: Couldn't open vcsa device");
		free(vv);
		return GGI_DL_ERROR;
	}

	/* read vcsa dimensions */
	if (read(LIBGGI_FD(vis), buf, 4) < 4) {
		perror("display-vcsa: Couldn't read vcsa device");
		close(LIBGGI_FD(vis));
		free(vv);
		return GGI_DL_ERROR;
	}

	vv->width  = buf[1];
	vv->height = buf[0];

	/* move cursor somewhere relatively out of the way */
        buf[2] = buf[1];  /* cursor_x */
        buf[3] = buf[0];  /* cursor_y */

	lseek(LIBGGI_FD(vis), 0, SEEK_SET);
	write(LIBGGI_FD(vis), buf, 4);

	/* Open keyboard and mouse input */

	if (vv->inputs & VCSA_INP_KBD) {
		vis->input = giiOpen("input-linux-kbd", NULL);

		if (vis->input == NULL) {
			fprintf(stderr, "display-vcsa: Couldn't open kbd.\n");
			close(LIBGGI_FD(vis));
			free(vv);
			return GGI_DL_ERROR;
		}
	}
	if (vv->inputs & VCSA_INP_MOUSE) {
		inp = giiOpen("linux-mouse:file", &args, NULL);
		if (inp != NULL) {
			vis->input = giiJoinInputs(vis->input, inp);
		}
	}
	
	/* mode management */
	vis->opdisplay->getmode   = GGI_vcsa_getmode;
	vis->opdisplay->setmode   = GGI_vcsa_setmode;
	vis->opdisplay->checkmode = GGI_vcsa_checkmode;
	vis->opdisplay->getapi    = GGI_vcsa_getapi;
	vis->opdisplay->flush     = GGI_vcsa_flush;
	vis->opdisplay->setflags  = GGI_vcsa_setflags;
		
	DPRINT("display-vcsa: GGIdlinit success.\n");

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	VcsaHook *vv = VCSA_HOOK(vis);

	DPRINT("display-vcsa: GGIdlcleanup start.\n");

	if (LIBGGI_FD(vis) >= 0) {

		GGI_vcsa_resetmode(vis);
		
		if (vis->input != NULL) {
			giiClose(vis->input);
			vis->input = NULL;
		}

		close(LIBGGI_FD(vis));
		LIBGGI_FD(vis) = -1;
	}

	free(LIBGGI_GC(vis));
	free(vv);

	DPRINT("display-vcsa: GGIdlcleanup done.\n");

	return 0;
}

int GGI_vcsa_flush(ggi_visual *vis, int tryflag)
{
	return 0;
}


#include <ggi/internal/ggidlinit.h>
