/* $Id: visual.c,v 1.24 1998/12/18 21:36:23 marcus Exp $
******************************************************************************

   Display-palemu: initialization

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/palemu.h>


static gg_option palemu_options[1] =
{
	{ "parent", "[C]" }
};


int GGIdlinit(ggi_visual *vis, const char *args,void *argptr)
{
	PalemuHook *ph;

	char target[1024];


	/* initialize */

	DPRINT("display-palemu: GGIdlinit start.\n");

	LIBGGI_PRIVATE(vis) = ph = _ggi_malloc(sizeof(PalemuHook));

	if ((ph->flush_lock = ggLockCreate()) == NULL) {
		free(ph);
		return GGI_DL_ERROR;
	}

	vis->gc = _ggi_malloc(sizeof(ggi_gc));
	ph->opmansync = _ggi_malloc(sizeof(_ggi_opmansync));

	ph->flags  = 0;
	ph->fb_ptr = NULL;


	/* handle arguments */

	if (getenv("GGI_PALEMU_OPTIONS") != NULL) {
		if (ggParseOptions(getenv("GGI_PALEMU_OPTIONS"),
			palemu_options, 1) == NULL) {
			
			fprintf(stderr, "display-palemu: error in "
				"$GGI_PALEMU_OPTIONS.\n");
			free(ph);
			return GGI_DL_ERROR;
		}
	}
	
	if (args) {
		args = ggParseOptions((char *) args, palemu_options, 3);
			
		if (args == NULL) {
			fprintf(stderr, "display-palemu: error in "
				"arguments.\n");
			free(ph);
			return GGI_DL_ERROR;
		}
	}

	DPRINT("display-palemu: parent mode is '%s'\n",
		palemu_options[0].result);

	ggiParseMode(palemu_options[0].result, &ph->mode);
	

	/* setup mansync */
	if (! _ggiAddDL(vis, "helper-mansync", NULL, ph->opmansync, 0)) {
		fprintf(stderr, "display-palemu: Cannot load "
			"helper-mansync!\n");
		return GGI_DL_ERROR;
	}

	MANSYNC_init(vis);


	/* open the parent visual */
	while (args && *args && isspace((unsigned char) *args)) {
		args++;
	}

	DPRINT("display-palemu: opening target: %s\n", args);

	strcpy(target, "");

	if (args) {
		if (ggParseTarget((char *) args, target, 1024) == NULL) {
			/* error occurred */
			free(ph);
			return GGI_DL_ERROR;
		}
	}

	if (*target == 0) {
		strcpy(target, "auto");
	}

	ph->parent = ggiOpen(target, NULL);

	if (ph->parent == NULL) {
		fprintf(stderr, "display-palemu: Failed to open "
			"target: '%s'\n", target);
		free(ph);
		return GGI_DL_ERROR;
	}
	
	ggiSetFlags(ph->parent, GGIFLAG_ASYNC);

	/* add giiInputs, if we have them */

	if (ph->parent->input)
	{
		vis->input=giiJoinInputs(vis->input, ph->parent->input);
		ph->parent->input=NULL;	/* destroy old reference */
	}

	/* Has mode management */
	
	vis->opdisplay->getmode = GGI_palemu_getmode;
	vis->opdisplay->setmode = GGI_palemu_setmode;
	vis->opdisplay->checkmode = GGI_palemu_checkmode;
	vis->opdisplay->getapi = GGI_palemu_getapi;
	vis->opdisplay->flush = GGI_palemu_flush;
	vis->opdisplay->setflags = GGI_palemu_setflags;
	
	DPRINT("display-palemu: GGIdlinit succeeded.\n");

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	DPRINT("display-palemu: GGIdlcleanup start.\n");

	MANSYNC_deinit(vis);

	if (ph->fb_ptr != NULL) {
		GGI_palemu_resetmode(vis);
	}

	if (ph->parent != NULL) {
		ggiClose(ph->parent);
		ph->parent = NULL;
	}

	LIBGGI_PRIVATE(vis) = NULL;

	ggLockDestroy(ph->flush_lock);

	free(ph->opmansync);
	free(vis->gc);
	free(ph);

	DPRINT("display-palemu: GGIdlcleanup done.\n");

	return 0;
}

#include <ggi/internal/ggidlinit.h>
