/* $Id: mode.c,v 1.12 1998/12/24 00:59:51 marcus Exp $
******************************************************************************

   Display-file: mode management

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/file.h>

#include "../common/pixfmt-setup.inc"
#include "../common/ggi-auto.inc"
#include "../common/gt-auto.inc"


#define write_byte    _ggi_file_write_byte
#define write_word    _ggi_file_write_word
#define write_string  _ggi_file_write_string
#define write_zeros   _ggi_file_write_zeros
#define write_flush   _ggi_file_flush


static int _ggi_rawstuff(ggi_visual *vis)
{
	FileHook *ff = FILE_PRIV(vis);

	ggi_graphtype gt = LIBGGI_GT(vis);

	int padding;


	/* Calculate stuff
	 */
	ff->offset_pal = 8+2+2+4+2+2;

	ff->offset_image = ff->offset_pal + ff->num_cols * 3;
	ff->offset_image += ff->fb_stride-1;
	ff->offset_image -= (ff->offset_image % ff->fb_stride);

	ff->file_size = ff->offset_image + ff->fb_size;
	ff->file_size += ROUND_UP_SIZE-1;
	ff->file_size -= (ff->file_size % ROUND_UP_SIZE);

	padding = ff->offset_image - ff->offset_pal - ff->num_cols*3;
	
	DPRINT("display-file: stride=0x%x padding=0x%x "
		"offset_image=0x%x file_size=0x%x", ff->fb_stride, 
		padding, ff->offset_image, ff->file_size);
		

	/* Write initial file contents 
	 */
	write_string(vis, "\020GGIFILE");	/* magic */
	write_word(vis, LIBGGI_VIRTX(vis));	/* width */
	write_word(vis, LIBGGI_VIRTY(vis));	/* height */

	/* graphtype */
	write_byte(vis, (gt & GT_SCHEME_MASK) >> GT_SCHEME_SHIFT);
	write_byte(vis, (gt & GT_SUBSCHEME_MASK) >> GT_SUBSCHEME_SHIFT);
	write_byte(vis, (gt & GT_SIZE_MASK)  >> GT_SIZE_SHIFT);
	write_byte(vis, (gt & GT_DEPTH_MASK) >> GT_DEPTH_SHIFT);

	write_word(vis, ff->fb_stride);		/* stride */
	write_word(vis, padding);		/* padsize */

	write_zeros(vis, ff->num_cols * 3);	/* palette */
	write_zeros(vis, padding);		/* padding */

	/* image */
	write_zeros(vis, ff->file_size - ff->offset_image);
	write_flush(vis);


	/* Map the file into memory
	 */
	ff->file_mmap = (uint8 *) mmap(0, ff->file_size, 
		PROT_READ | PROT_WRITE, MAP_SHARED, LIBGGI_FD(vis), 0);

	DPRINT("display-file: File mmap'd at 0x%x.\n", ff->file_mmap);

	if (ff->file_mmap == (uint8 *) -1) {
		perror("display-file: mmap failed");
		close(LIBGGI_FD(vis));
		return -1;
	}

	ff->fb_ptr = ff->file_mmap + ff->offset_image;

	return 0;
}

static int _ggi_getmmap(ggi_visual *vis)
{
	FileHook *ff = FILE_PRIV(vis);

	ggi_graphtype gt = LIBGGI_GT(vis);


	ff->fb_stride = ((GT_SIZE(gt) * LIBGGI_VIRTX(vis)) + 7) / 8;
	ff->fb_size = LIBGGI_FB_SIZE(LIBGGI_MODE(vis));

	if (GT_SCHEME(gt) == GT_PALETTE) {
		ff->num_cols = 1 << GT_DEPTH(gt);
	} else {
		ff->num_cols = 0;
	}


	/* create file */
	
	if (_ggi_file_create_file(vis, ff->filename) < 0) {
		return -1;
	}


	/* map the file into memory */

	if (ff->flags & FILEFLAG_RAW) {
		if (_ggi_rawstuff(vis) < 0) {
			return -1;
		}
	} else {
		ff->fb_ptr = malloc(ff->fb_size);

		if (ff->fb_ptr == NULL) {
			DPRINT_MODE("display-file: Out of memory!");
			return -1;
		}
	}

	/* set up pixel format */
	memset(LIBGGI_PIXFMT(vis), 0, sizeof(ggi_pixelformat));
	setup_pixfmt(LIBGGI_PIXFMT(vis), gt);
	_ggi_build_pixfmt(LIBGGI_PIXFMT(vis));

	/* set up DirectBuffer */
	_ggi_db_add_buffer(LIBGGI_APPLIST(vis), _ggi_db_get_new());

	LIBGGI_APPBUFS(vis)[0]->frame = 0;
	LIBGGI_APPBUFS(vis)[0]->type  = GGI_DB_NORMAL | GGI_DB_SIMPLE_PLB;
	LIBGGI_APPBUFS(vis)[0]->read  = ff->fb_ptr;
	LIBGGI_APPBUFS(vis)[0]->write = ff->fb_ptr;
	LIBGGI_APPBUFS(vis)[0]->layout = blPixelLinearBuffer;
	LIBGGI_APPBUFS(vis)[0]->buffer.plb.stride = ff->fb_stride;
	LIBGGI_APPBUFS(vis)[0]->buffer.plb.pixelformat = LIBGGI_PIXFMT(vis);

	/* set up palette */
	if (GT_SCHEME(gt) == GT_PALETTE) {
		vis->palette=_ggi_malloc(sizeof(ggi_color) * ff->num_cols);
	}
	
	return 0;
}

int GGI_file_getapi(ggi_visual *vis,int num, char *apiname ,char *arguments)
{
	ggi_graphtype gt = LIBGGI_GT(vis);

	strcpy(arguments,"");

	switch(num) { 

	case 0: strcpy(apiname, "display-file");
		return 0;

	case 1: strcpy(apiname, "generic-stubs");
		return 0;
		
	case 2: if (GT_SCHEME(gt) == GT_TEXT) {
			sprintf(apiname, "generic-text-%d", GT_SIZE(gt));
			return 0;
		}

		sprintf(apiname, "generic-linear-%d%s", GT_SIZE(gt),
			(gt & GT_SUB_HIGHBIT_RIGHT) ? "-r" : "");
		return 0;

	case 3: if (GT_SCHEME(gt) == GT_TEXT) {
			return -1;
		}

		strcpy(apiname, "generic-color");
		return 0;
	}

	return -1;
}

static void _ggi_freedbs(ggi_visual *vis) 
{
	int i;

	for (i=LIBGGI_APPLIST(vis)->num-1; i >= 0; i--) {
		_ggi_db_free(LIBGGI_APPBUFS(vis)[i]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i);
	}
}

static int _ggi_domode(ggi_visual *vis)
{
	int err, i;

	char name[256];
	char args[256];

	_ggiZapMode(vis,~GGI_DL_OPDISPLAY);
	_ggi_freedbs(vis);

	DPRINT("display-file: _ggi_domode: zapped\n");

	if ((err = _ggi_getmmap(vis)) != 0) {
		return err;
	}

	DPRINT("display-file: _ggi_domode: got mmap\n");

	for(i=1; GGI_file_getapi(vis, i, name, args) == 0; i++) {

		if (_ggiOpenDL(vis, name, args, NULL) == NULL) {
			fprintf(stderr,"display-file: Can't open the "
				"%s (%s) library.\n", name, args);
			return err;
		} else {
			DPRINT("display-file: Success in loading "
				"%s (%s)\n", name, args);
		}
	}

	if (GT_SCHEME(LIBGGI_GT(vis)) == GT_PALETTE) {
		vis->opcolor->setpalvec = GGI_file_setpalvec;
	}
	
	return 0;
}

int GGI_file_setmode(ggi_visual *vis, ggi_mode *mode)
{ 
	int err;

	if (vis==NULL || mode==NULL || LIBGGI_MODE(vis)==NULL) {
		return -1;
	}
	
	DPRINT_MODE("display-file: setmode %dx%d#%dx%dF%d[0x%02x]\n",
			mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);

	if ((err = ggiCheckMode(vis, mode)) != 0) {
		return err;
	}

	*LIBGGI_MODE(vis) = *mode;

	err = _ggi_domode(vis);

	if (err) {
		DPRINT("display-file: domode failed (%d)\n",err);
		return err;
	}

	ggiIndicateChange(vis, GGI_CHG_APILIST);
	DPRINT("display-file: change indicated\n",err);

	return 0;
}

int GGI_file_resetmode(ggi_visual *vis)
{
	FileHook *ff = FILE_PRIV(vis);

	DPRINT("display-file: GGIresetmode(%p)\n", vis);

	if (ff->flags & FILEFLAG_RAW) {
		munmap((void *)ff->file_mmap, ff->file_size);
	} else {
		/* This is where we write the non-raw file */

		(* ff->writer)(vis);

		free((void *)ff->fb_ptr);
	}

	ff->file_mmap = ff->fb_ptr = NULL;

	_ggi_freedbs(vis);
	_ggi_file_close_file(vis);

	return 0;
}

int GGI_file_checkmode(ggi_visual *vis, ggi_mode *mode)
{
	/* FileHook *ff = FILE_PRIV(vis); */

	int err = 0;

	if (vis==NULL || mode==NULL || LIBGGI_MODE(vis)==NULL) {
		return -1;
	}

	DPRINT_MODE("display-file: checkmode %dx%d#%dx%dF%d[0x%02x]\n",
			mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);


	/* handle GT_AUTO and GGI_AUTO */

	_GGIhandle_ggiauto(mode, 320, 200);

	mode->graphtype = _GGIhandle_gtauto(mode->graphtype);


	/* do some checks */

	if (mode->frames != 1) {
		mode->frames = 1;
		err--;
	}

	if (GT_SIZE(mode->graphtype) < 8) {
	
		int align = 8 / GT_SIZE(mode->graphtype);

		if (mode->visible.x % align != 0) {
			mode->visible.x += align-(mode->visible.x % align);
			err--;
		}
		
		if (mode->virt.x % align != 0) {
			mode->virt.x += align-(mode->virt.x % align);
			err--;
		}
	}
	
	if (mode->virt.x < mode->visible.x) {
		mode->virt.x = mode->visible.x;
		err--;
	}

	if (mode->virt.x < mode->visible.x) {
		mode->virt.y = mode->visible.y;
		err--;
	}

	DPRINT_MODE("display-file: result %d %dx%d#%dx%dF%d[0x%02x]\n",
			err, mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);

	return err;	
}

int GGI_file_getmode(ggi_visual *vis, ggi_mode *mode)
{
	DPRINT("display-file: GGIgetmode(%p,%p)\n", vis, mode);

	if (vis==NULL || mode==NULL || LIBGGI_MODE(vis)==NULL) {
		return -1;
	}

	memcpy(mode, LIBGGI_MODE(vis), sizeof(ggi_mode));

	return 0;
}

int GGI_file_setflags(ggi_visual *vis, ggi_flags flags)
{
	LIBGGI_FLAGS(vis) = flags;

	return 0;
}

int GGI_file_setpalvec(ggi_visual *vis, int start, int len, ggi_color *colormap)
{
	FileHook *ff = FILE_PRIV(vis);

	uint8 *file_pal = ff->file_mmap + ff->offset_pal;


	DPRINT("display-file: setpalette.\n");

	if (GT_SCHEME(LIBGGI_GT(vis)) != GT_PALETTE)
		return -1;

	if (start == -1) {
		start = 0;
	}

	if (colormap==NULL || (start+len) > ff->num_cols) {
		return -1;
	}
                        
	file_pal += start * 3;

	for (; len > 0; len--, start++, colormap++) {

		vis->palette[start] = *colormap;
		
		if (ff->flags & FILEFLAG_RAW) {
			*file_pal++ = colormap->r >> 8;
			*file_pal++ = colormap->g >> 8;
			*file_pal++ = colormap->b >> 8;
		}
	}

	return 0;
}
