/* $Id: visual.c,v 1.42 1998/12/22 14:25:22 jmcc Exp $
******************************************************************************

   Display-FBDEV: visual handling

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/fbdev.h>


#define FBDEV_OPTIONS  3

static gg_option fbdev_options[FBDEV_OPTIONS] =
{
	{ "noinput", "no" },
	{ "nokbd",   "no" },
	{ "nomouse", "no" }
};

int GGIdlinit(ggi_visual *vis, const char *args,void *argptr)
{
	fbdev_hook *ff;
	gii_input *inp;

	DPRINT("display-fbdev: GGIdlinit start.\n");


	LIBGGI_PRIVATE(vis) = ff = _ggi_malloc(sizeof(fbdev_hook));
	
	ff->fb_ptr = NULL;
	ff->need_redraw = 0;
	ff->have_kgi = 0;
		
	if (getenv("GGI_FBDEV_OPTIONS") != 0) {
		if (ggParseOptions(getenv("GGI_FBDEV_OPTIONS"),
			fbdev_options, FBDEV_OPTIONS) == NULL) {
			
			fprintf(stderr, "display-fbdev: error in "
				"$GGI_FBDEV_OPTIONS.\n");
			free(ff);
			return GGI_DL_ERROR;
		}
	}
	
	if (args) {
		args = ggParseOptions((char *)args, fbdev_options,
				FBDEV_OPTIONS);

		if (args == NULL) {
			
			fprintf(stderr, "display-fbdev: error in "
				"arguments.\n");
			free(ff);
			return GGI_DL_ERROR;
		}
	}


	/* handle args */

	while (args && *args && isspace(*args)) args++;

	if (args && *args) {
		ff->devicetype = strdup(args);
	} else if (getenv("FRAMEBUFFER") != NULL) {
		ff->devicetype = strdup(getenv("FRAMEBUFFER"));
	} else {
		ff->devicetype = strdup("/dev/fb0");
	}
	
	ff->inputs = FBDEV_INP_KBD | FBDEV_INP_MOUSE;

	if (toupper(fbdev_options[0].result[0]) != 'N') {
		ff->inputs = 0;
	}
	if (toupper(fbdev_options[1].result[0]) != 'N') {
		ff->inputs &= ~FBDEV_INP_KBD;
	}
	if (toupper(fbdev_options[2].result[0]) != 'N') {
		ff->inputs &= ~FBDEV_INP_MOUSE;
	}
	

	/* now open the framebuffer device */

	LIBGGI_FD(vis) = open(ff->devicetype, O_RDWR);

	if (LIBGGI_FD(vis) < 0) {
		fprintf(stderr, "display-fbdev: Couldn't open "
			"framebuffer device %s: %s\n", ff->devicetype,
			strerror(errno));

		free(ff->devicetype); free(ff);
		return GGI_DL_ERROR;
	}

	/* read original mode on framebuffer */

	if ((ioctl(LIBGGI_FD(vis),FBIOGET_FSCREENINFO,&ff->orig_fix) < 0) ||
	    (ioctl(LIBGGI_FD(vis),FBIOGET_VSCREENINFO,&ff->orig_var) < 0)) {
	    
		perror("display-fbdev: GET_SCREENINFO");

		close(LIBGGI_FD(vis)); LIBGGI_FD(vis) = -1;
		free(ff->devicetype); free(ff);
		return GGI_DL_ERROR;
	}

	LIBGGI_GC(vis) = ff->normalgc = _ggi_malloc(sizeof(ggi_gc));


	/* Open keyboard and mouse input */

	if (ff->inputs & FBDEV_INP_KBD) {
		vis->input = giiOpen("input-linux-kbd", NULL);

		if (vis->input == NULL) {
			fprintf(stderr, "display-fbdev: Couldn't open kbd.\n");
			close(LIBGGI_FD(vis));
			free(ff->normalgc); free(ff);
			return GGI_DL_ERROR;
		}
	}
	if (ff->inputs & FBDEV_INP_MOUSE) {
		if ((inp = giiOpen("linux-mouse:file", &args, NULL)) != NULL) {
			vis->input = giiJoinInputs(vis->input, inp);
		}
	}

	/* Mode management */
	vis->opdisplay->getmode   = GGI_fbdev_getmode;
	vis->opdisplay->setmode   = GGI_fbdev_setmode;
	vis->opdisplay->checkmode = GGI_fbdev_checkmode;
	vis->opdisplay->getapi    = GGI_fbdev_getapi;
	vis->opdisplay->flush     = GGI_fbdev_flush;
	vis->opdisplay->setflags  = GGI_fbdev_setflags;
		
	DPRINT("display-fbdev: GGIdlinit success.\n");

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	DPRINT("display-fbdev: GGIdlcleanup start.\n");

	if (LIBGGI_FD(vis) >= 0) {

		GGI_fbdev_resetmode(vis);

		if (vis->input != NULL) {
			giiClose(vis->input);
			vis->input = NULL;
		}

		close(LIBGGI_FD(vis));
	}

	if (ff->devicetype) free(ff->devicetype);

	free(ff->normalgc);
	free(ff);

	DPRINT("display-fbdev: GGIdlcleanup done.\n");

	return 0;
}

static void fbdev_send_expose_event(ggi_visual *vis)
{
	ggi_event ev;

	ev.any.size   = sizeof(ggi_expose_event);
	ev.any.type   = evExpose;
	ev.any.origin = GII_EV_ORIGIN_NONE;
	ev.any.target = GII_EV_TARGET_ALL;

	ev.expose.x = ev.expose.y = 0;
	ev.expose.w = LIBGGI_VIRTX(vis);
	ev.expose.h = LIBGGI_VIRTY(vis);

	if (vis->input) {
		giiEventSend(vis->input, &ev);
	}

	DPRINT_EVENTS("display-fbdev: EXPOSE sent.\n");
}

int GGI_fbdev_flush(ggi_visual *vis, int tryflag)
{
	fbdev_hook *ff = LIBGGI_PRIVATE(vis);

	/* Putting this here is far from ideal.  If the application
	 * never calls ggiFlush (and when in SYNC mode, that is
	 * perfectly OK), then it will never receive evExpose events.  
	 *
	 * But it can't be put into handle_switched_back() since that
	 * function is called from a signal handler -- which could have
	 * interrupted a "critical section" in the event queuing code.
	 */

	if (tryflag && ff->need_redraw) {
		fbdev_send_expose_event(vis);
		ff->need_redraw = 0;
	}
	 
	return 0;
}


#include <ggi/internal/ggidlinit.h>
