/***************************************************************************
                          engine.c  -  description
                             -------------------
    begin                : Sat Feb 3 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "sdl.h"
#include "dynlist.h"
#include "tools.h"
#include "config.h"
#include "date.h"
#include "nation.h"
#include "unit.h"
#include "player.h"
#include "theme.h"
#include "map.h"
#include "scenario.h"
#include "gui.h"
#include "ai_action.h"
#include "engine_tools.h"
#include "deploy.h"
#include "strat_map.h"
#include "ai_tools.h"
#include "ai.h"
#include "window.h"
#include "slot.h"
#include "engine.h"

#define ENGINE_DEBUG

extern Sdl sdl;
extern Config config;
extern Map map;
extern Scen scen;
extern int cur_weather; /* used in unit.c to compute weather influence on units; set by set_player() */

/* --- forwarded --- */
/* get primary unit by checking engine->air_mode */
inline Unit* get_prim_unit( Engine *engine, int x, int y );
/* get primary unit by checking engine->air_mode */
inline Unit* get_prim_deploy_unit( Engine *engine, int x, int y );
/* get primary target; returns 0 if there is none */
Unit* get_prim_target( Engine *engine, int x, int y );
/* while engine->sel_pred->support_count > 0 initiate fight between att and supporter else between
attacker and defender */
int init_next_fight( Engine *engine );
/* deactivate all unit buttons */
void clear_unit_buttons( Engine *engine );
/* init last round; set control to nobody so the players can have a look on the map */
void init_final_view( Engine *engine );

/* create engine -- get mem and set theme pointer and create windows */
Engine *create_engine( Theme *theme )
{
    Engine *engine = 0;

    /* get and reset memory */
    engine = malloc( sizeof( Engine ) );
    memset( engine, 0, sizeof( Engine ) );

    /* fog alpha */
    engine->fog_alpha = 128;

    /* set theme and create windows */
    engine->theme = theme;
    create_windows( engine ); /* uses the current theme */

    /* screen copy */
    engine->screen_copy_type = NO_SCREEN_COPY;

    /* cursors */
    engine->stan_cursor = create_cursor( 16, 16, 0, 0,
                                       "bbb             "
                                       "bwwbb           "
                                       "bwwwwbb         "
                                       " bwwwwwbb       "
                                       " bwwwwwwwbb     "
                                       "  bwwwwwwwwbb   "
                                       "  bwwwwwwwwwwbb "
                                       "   bwwwwwwwwwwwb"
                                       "   bwwwwwwwwwwwb"
                                       "    bwwwwwwwwwwb"
                                       "    bwwwwwwwwwb "
                                       "     bwwwwwwwb  "
                                       "     bwwwwwwb   "
                                       "      bwwwwb    "
                                       "      bwwwb     "
                                       "       bbb      " );
    engine->up_cursor = create_cursor( 16, 16, 8, 0,
                                       "       bb       "
                                       "      bwwb      "
                                       "      bwwb      "
                                       "     bwwwwb     "
                                       "     bwwwwb     "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "   bwwwwwwwwb   "
                                       "   bwwwwwwwwb   "
                                       "  bwwwwwwwwwwb  "
                                       "  bwwwwwwwwwwb  "
                                       " bwwwwwbbwwwwwb "
                                       " bwwwwbbbbwwwwb "
                                       "bwwwwbb  bbwwwwb"
                                       "bwwwbb    bbwwwb"
                                       "bbbbb      bbbbb" );
    engine->down_cursor = create_cursor( 16, 16, 8, 15,
                                       "bbbbb      bbbbb"
                                       "bwwwbb    bbwwwb"
                                       "bwwwwbb  bbwwwwb"
                                       " bwwwwbbbbwwwwb "
                                       " bwwwwwbbwwwwwb "
                                       "  bwwwwwwwwwwb  "
                                       "  bwwwwwwwwwwb  "
                                       "   bwwwwwwwwb   "
                                       "   bwwwwwwwwb   "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "     bwwwwb     "
                                       "     bwwwwb     "
                                       "      bwwb      "
                                       "      bwwb      "
                                       "       bb       " );
    engine->left_cursor = create_cursor( 16, 16, 0, 8,
                                       "             bbb"
                                       "           bbwwb"
                                       "         bbwwwwb"
                                       "       bbwwwwwwb"
                                       "     bbwwwwwwwbb"
                                       "   bbwwwwwwwwbb "
                                       " bbwwwwwwwwwbb  "
                                       "bwwwwwwwwwwbb   "
                                       "bwwwwwwwwwwbb   "
                                       " bbwwwwwwwwwbb  "
                                       "   bbwwwwwwwwbb "
                                       "     bbwwwwwwwbb"
                                       "       bbwwwwwwb"
                                       "         bbwwwwb"
                                       "           bbwwb"
                                       "             bbb" );
    engine->right_cursor = create_cursor( 16, 16, 15, 8,
                                       "bbb             "
                                       "bwwbb           "
                                       "bwwwwbb         "
                                       "bwwwwwwbb       "
                                       "bbwwwwwwwbb     "
                                       " bbwwwwwwwwbb   "
                                       "  bbwwwwwwwwwbb "
                                       "   bbwwwwwwwwwwb"
                                       "   bbwwwwwwwwwwb"
                                       "  bbwwwwwwwwwbb "
                                       " bbwwwwwwwwbb   "
                                       "bbwwwwwwwbb     "
                                       "bwwwwwwbb       "
                                       "bwwwwbb         "
                                       "bwwbb           "
                                       "bbb             " );
    engine->sel_cursor = create_cursor( 16, 16, 8, 8,
                                       "bbbbb      bbbbb"
                                       "bwwwwb    bwwwwb"
                                       "bwwwwb    bwwwwb"
                                       "bwwwwwb  bwwwwwb"
                                       "bwwwwwb  bwwwwwb"
                                       " bbwwwb  bwwwbb "
                                       "   bbbb  bbbb   "
                                       "                "
                                       "                "
                                       "   bbbb  bbbb   "
                                       " bbwwwb  bwwwbb "
                                       "bwwwwwb  bwwwwwb"
                                       "bwwwwwb  bwwwwwb"
                                       "bwwwwb    bwwwwb"
                                       "bwwwwb    bwwwwb"
                                       "bbbbb      bbbbb" );
    engine->move_cursor = create_cursor( 16, 16, 8, 8,
                                       "                "
                                       "                "
                                       "        bbb     "
                                       "        bwwb    "
                                       "        bwwwb   "
                                       "bbbbbbbbbwwwwb  "
                                       "bwwwwwwwwwwwwwb "
                                       "bwwwwwwwwwwwwwwb"
                                       "bwwwwwwwwwwwwwwb"
                                       "bwwwwwwwwwwwwwb "
                                       "bbbbbbbbbwwwwb  "
                                       "        bwwwb   "
                                       "        bwwb    "
                                       "        bbb     "
                                       "                "
                                       "                " );
    engine->attack_cursor = create_cursor( 16, 16, 8, 8,
                                       "    bbbwwbbb    "
                                       "   bwwwwwwwwb   "
                                       "  bwwbbwwbbwwb  "
                                       " bwwb bwwb bwwb "
                                       "bwwb  bwwb  bwwb"
                                       "bwb   bwwb   bwb"
                                       "bwbbbb bb bbbbwb"
                                       "wwwwwwb  bwwwwww"
                                       "wwwwwwb  bwwwwww"
                                       "bwbbbb bb bbbbwb"
                                       "bwb   bwwb   bwb"
                                       "bwwb  bwwb  bwwb"
                                       " bwwb bwwb bwwb "
                                       "  bwwbbwwbbwwb  "
                                       "   bwwwwwwwwb   "
                                       "    bbbwwbbb    " );
    engine->ground_tran_cursor = create_cursor( 16, 16, 8, 8,
                                       "                "
                                       "                "
                                       "                "
                                       "bbbbbbbbb       "
                                       "bwwwbwwwbbbbb   "
                                       "bwwwbwwwbwbwwb  "
                                       "bwwwbwwwbwbwwb  "
                                       "bwwwbwwwbbbbbbbb"
                                       "bbbbbbbbbwwwwwwb"
                                       "bwwwwwwwwwwwwwwb"
                                       "bbbbbbwwbbbbbbwb"
                                       " bwwwbbbbbwwwbbb"
                                       " bwbwb   bwbwb  "
                                       " bwwwb   bwwwb  "
                                       "  bbb     bbb   "
                                       "                " );
    engine->debark_cursor = create_cursor( 16, 16, 8, 8,
                                       "       bb       "
                                       "      bwwb      "
                                       "     bwwwwb     "
                                       "    bwwwwwwb    "
                                       "   bwwwwwwwwb   "
                                       "  bwwwwwwwwwwb  "
                                       "  bbbbwwwwbbbb  "
                                       "     bwwwwb     "
                                       "     bwwwwb     "
                                       "     bbbbbb     "
                                       "bbbb        bbbb"
                                       "bwwbbbbbbbbbbwwb"
                                       "bwwwbwwwwwwbwwwb"
                                       "bbwwbwwwwwwbwwbb"
                                       " bbwbwwwwwwbwbb "
                                       "  bbbbbbbbbbbb  " );
    engine->embark_cursor = create_cursor( 16, 16, 8, 8,
                                       "     bbbbbb     "
                                       "     bwwwwb     "
                                       "     bwwwwb     "
                                       "  bbbbwwwwbbbb  "
                                       "  bwwwwwwwwwwb  "
                                       "   bwwwwwwwwb   "
                                       "    bwwwwwwb    "
                                       "     bwwwwb     "
                                       "      bwwb      "
                                       "       bb       "
                                       "bbbb        bbbb"
                                       "bwwbbbbbbbbbbwwb"
                                       "bwwwbwwwwwwbwwwb"
                                       "bbwwbwwwwwwbwwbb"
                                       " bbwbwwwwwwbwbb "
                                       "  bbbbbbbbbbbb  " );
    engine->merge_cursor = create_cursor( 16, 16, 8, 8,
                                       "  bbbbbbbbbbbb  "
                                       "  bwwwwwwwwwwb  "
                                       "   bwwwwwwwwb   "
                                       "    bwwwwwwb    "
                                       "     bwwwwb     "
                                       "      bwwb      "
                                       "       bb       "
                                       "                "
                                       "       bb       "
                                       "      bwwb      "
                                       "     bwwwwb     "
                                       "    bwwwwwwb    "
                                       "   bwwwwwwwwb   "
                                       "  bwwwwwwwwwwb  "
                                       "  bwwwwwwwwwwb  "
                                       "  bbbbbbbbbbbb  " );
    engine->deploy_cursor = create_cursor( 16, 16, 8, 15,
                                       "    bbbbbbbb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "bbbbbwwwwwwbbbbb"
                                       "bwwwwwwwwwwwwwwb"
                                       " bwwwwwwwwwwwwb "
                                       "  bwwwwwwwwwwb  "
                                       "   bwwwwwwwwb   "
                                       "    bwwwwwwb    "
                                       "     bwwwwb     "
                                       "      bwwb      "
                                       "       bb       " );
    engine->undeploy_cursor = create_cursor( 16, 16, 8, 0,
                                       "       bb       "
                                       "      bwwb      "
                                       "     bwwwwb     "
                                       "    bwwwwwwb    "
                                       "   bwwwwwwwwb   "
                                       "  bwwwwwwwwwwb  "
                                       " bwwwwwwwwwwwwb "
                                       "bwwwwwwwwwwwwwwb"
                                       "bbbbbwwwwwwbbbbb"
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bwwwwwwb    "
                                       "    bbbbbbbb    " );
/* template
    engine->up_cursor = create_cursor( 16, 16, 8, 0,
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                "
                                       "                " );
*/

    /* set standard cursor */
    SDL_SetCursor( engine->stan_cursor );

    /* moving velocity */
    engine->unit_vel = 0.3;

    /* set timers */
    set_delay( &engine->scroll_delay, 100 );
    set_delay( &engine->cross_delay, 750 );
    set_delay( &engine->expl_delay, 50 );

    /* set status to AWAITING_INIT */
    engine->status = INIT;

    /* init save slots */
    init_slots( engine );

    return engine;
}

/* delete engine completly */
void delete_engine( Engine *engine )
{
    if ( !engine ) return;

    /* clear stuff allocated by init engine */
    clear_engine( engine );

    /* windows */
    delete_windows( engine );

    /* cursors */
    if ( engine->stan_cursor ) SDL_FreeCursor( engine->stan_cursor );
    if ( engine->up_cursor ) SDL_FreeCursor( engine->up_cursor );
    if ( engine->down_cursor ) SDL_FreeCursor( engine->down_cursor );
    if ( engine->left_cursor ) SDL_FreeCursor( engine->left_cursor );
    if ( engine->right_cursor ) SDL_FreeCursor( engine->right_cursor );
    if ( engine->sel_cursor ) SDL_FreeCursor( engine->sel_cursor );
    if ( engine->move_cursor ) SDL_FreeCursor( engine->move_cursor );
    if ( engine->attack_cursor ) SDL_FreeCursor( engine->attack_cursor );
    if ( engine->ground_tran_cursor ) SDL_FreeCursor( engine->ground_tran_cursor );
    if ( engine->debark_cursor ) SDL_FreeCursor( engine->debark_cursor );
    if ( engine->embark_cursor ) SDL_FreeCursor( engine->embark_cursor );
    if ( engine->merge_cursor ) SDL_FreeCursor( engine->merge_cursor );
    if ( engine->deploy_cursor ) SDL_FreeCursor( engine->deploy_cursor );
    if ( engine->undeploy_cursor ) SDL_FreeCursor( engine->undeploy_cursor );

    FREE( engine );
    engine = 0;
}

/* prepare engine (set dyn variables etc ) */
int init_engine( Engine *engine, char *scen_name, int *player_cont )
{
    int i, j, bytes;
    int pos, line_length;

    /* check status -- must be INIT */
    if ( engine->status != INIT ) {

        fprintf( stderr, "cannot init engine... bad status!\n");
        return 0;

    }

    if ( !load_scen( scen_name, DEEP ) ) return 0;
    /* overwrite default controls? */
    if ( player_cont != 0 ) {
        for ( i = 0; i < scen.player_count; i++ )
            if ( player_cont[i] == HUMAN )
                scen.players[i]->cont = HUMAN;
            else
                scen.players[i]->cont = CPU;
        FREE( player_cont );
    }
    /* initiate strategic map from scenario and engine info */
    strat_map_init( engine );

    /* compute map offset and width; set position to 0,0 */
    engine->start_x = -map.def->tile_x_offset;
    engine->start_y = engine->upper_bar->height - map.def->tile_y_offset;
    engine->x_offset = map.def->tile_x_offset;
    engine->y_offset = map.def->tile_y_offset;
    engine->map_x = engine->map_y = 0;
    engine->map_width = ( sdl.screen->w + map.def->tile_x_offset - ( map.def->tile_width - engine->x_offset ) ) / engine->x_offset + 2;
    engine->map_height = ( sdl.screen->h + map.def->tile_y_offset - engine->lower_bar->height - engine->upper_bar->height ) / map.def->tile_height + 1;

    /* create tile mask */
    /*  1 = map tile directly hit
        0 = neighbor */
    engine->tile_mask = calloc( map.def->tile_width * map.def->tile_height,
                                sizeof ( int ) );
    memset( engine->tile_mask, 0,
            sizeof ( int ) * map.def->tile_width * map.def->tile_height );
    bytes = map.def->fog_pic->format->BytesPerPixel;
    line_length = map.def->fog_pic->pitch;
    pos = 0;
    lock_surf( map.def->fog_pic );
    for ( j = 0; j < map.def->tile_height; j++ ) {

        for ( i = 0; i < map.def->tile_width; i++ )
            if ( ((char*)map.def->fog_pic->pixels)[ pos + ( i * bytes ) ] )
                engine->tile_mask[j * map.def->tile_width + i] = 1;
        pos += line_length;

    }
    unlock_surf( map.def->fog_pic );

    /* screen copy buffer */
    engine->screen_buffer = create_surf( sdl.screen->w, sdl.screen->h, SDL_SWSURFACE );

    /* set remaining/current turns */
    scen.rem_turns = scen.minor_turn_limit;
    scen.cur_turn = 0;

    /* explosion size */
    engine->expl_frame_height = map.def->expl_pic->h;
    engine->expl_frame_width = engine->expl_frame_height;
    engine->expl_frame_count = map.def->expl_pic->w / engine->expl_frame_width;

    /* no unit selected */
    engine->sel_unit = 0;

    /* clear save and end turn buttons */
    set_active( get_child( engine->menu, ID_SAVE_MENU ), 1, NO_REFRESH );
    set_active( get_child( engine->lower_bar, ID_END_TURN_BUTTON ), 1, NO_REFRESH );

    /* check unit buttons */
    clear_unit_buttons( engine );

    /* clear cpu actions */
    engine->cpu_actions_count = 0;

    /* set status */
    engine->status = WAITING;

    /* clear various flags */
    engine->ambush = 0;
    engine->leave_scen = 0;
    engine->final_view = 0;

    return 1;
}

/*
free used memory allocated by init_engine ( not create_engine! )
must be called after running engine and before initiating next scenario
*/
void clear_engine( Engine *engine )
{
    if ( !engine ) return;

    /* tile mask */
    if ( engine->tile_mask ) FREE( engine->tile_mask );
    engine->tile_mask = 0;

    /* screen buffer */
    if ( engine->screen_buffer ) SDL_FreeSurface( engine->screen_buffer );
    engine->screen_buffer = 0;

    /* way points */
    FREE( engine->way );

    /* clear target predictions */
    FREE( engine->pred )

    /* deploy unit pointers */
    FREE( engine->deploy_units );

    /* scenario */
    clear_scen();
    /* clear strategic map */
    strat_map_clear( engine );

    engine->status = INIT;
}

/* run engine -- run game */
int run_engine( Engine *engine )
{
    Window *used_child = 0;
    SDL_Event event;
    int ms = 0;
    int map_x, map_y;
    int screenshot_count = 0;
    char aux_str[128]; /* for screenshot name */
    int ret = 0;
    Unit *partner;
    Unit *unit;
    int i;

    if ( engine->status != WAITING ) {
        fprintf( stderr, "can't run engine: bad engine status: %i\n", engine->status );
        return 0;
    }

    /* if this isn't a game that was loaded at the title menu the players must be reset */
    if ( !engine->scen_loaded ) {
        /* set player_id to -1 */
        engine->player_id = -1;
        end_turn( engine ); /* will draw engine the first time */
    }
    else {
        engine->scen_loaded = 0;
        update_full_screen( engine );
    }

    /* reset time */
    reset_timer();

    /* main game loop */
    while ( !engine->leave_scen ) {

        /* only handle events if HUMAN or NOBODY is in control and status is WAITING or SCROLLING */
        /* remember that only SCROLLING is allowed for NOBODY */
        if ( ( engine->cont_status == HUMAN || engine->cont_status == NOBODY ) &&
             ( engine->status == WAITING ||
               engine->status == SHOWING_MERGE_UNITS ||
               engine->status == DEPLOYING_UNITS ||
               engine->status == SCROLLING_BY_MOUSE ||
               engine->status == SCROLLING_BY_KEY ||
               engine->status == SHOWING_STRAT_MAP ) )
            if ( SDL_PollEvent( &event ) ) {

                switch ( event.type ) {

                    case SDL_KEYDOWN:
                        if ( engine->status == SHOWING_STRAT_MAP ) break;

                        /* scrolling is allowed for any status (if we got here)
                        the old status is safed in engine::old_status */
                        if ( engine->status != SCROLLING_BY_KEY && engine->status != SCROLLING_BY_MOUSE )
                            engine->old_status = engine->status;

                        reset_delay( &engine->scroll_delay );

                        switch ( event.key.keysym.sym ) {
                            case SDLK_UP:
                                scroll_map_by_key( engine, SCROLL_UP );
                                break;
                            case SDLK_DOWN:
                                scroll_map_by_key( engine, SCROLL_DOWN );
                                break;
                            case SDLK_RIGHT:
                                scroll_map_by_key( engine, SCROLL_RIGHT );
                                break;
                            case SDLK_LEFT:
                                scroll_map_by_key( engine, SCROLL_LEFT );
                                break;
                            default:
                                break;
                        }
                        break;

                    case SDL_KEYUP:
                        if ( engine->status == SHOWING_STRAT_MAP ) break;

                        /* ESC calls the menu */
                        if ( event.key.keysym.sym == SDLK_ESCAPE ) {
                            engine->window_id = ID_MENU;
                            break;
                        }
                        /* scrolling ? */
                        if ( engine->status == SCROLLING_BY_KEY )
                            engine->status = engine->old_status;
                        /* screenshot */
                        if ( event.key.keysym.sym == SDLK_TAB ) {
                            sprintf( aux_str, "screenshot_%i.bmp", screenshot_count++ );
                            SDL_SaveBMP( sdl.screen, aux_str );
                            printf( "Screenshot taken: %s\n", aux_str );
                        }

                        break;


                    case SDL_MOUSEBUTTONUP:

                        /* if deploying only deplyoing or undeploying is allowed */
                        if ( engine->status == DEPLOYING_UNITS ) {
                            if ( !get_map_pos( engine, event.button.x, event.button.y, &map_x, &map_y ) )
                                break;
                            /* right click on unit means undeploy else deploy window */
                            if ( event.button.button == RIGHT_BUTTON ) {
                                if ( ( unit = get_prim_deploy_unit( engine, map_x, map_y ) ) != 0 ) {
                                    if ( unit->delay > 0 ) {
                                        /* not inserted yet: undeploy */
                                        remove_unit_from_map( unit );
                                        unit->x = -1;
                                        /* recreate list */
                                        create_deploy_list( engine );
                                        /* as this may change the position of the current unit we simply
                                           switch to the removed one something we'd have to do anyway when
                                           list was empty */
                                        engine->sel_unit = unit;
                                        for ( i = 0; i < engine->deploy_unit_count; i++ )
                                            if ( engine->deploy_units[i] == unit ) {
                                                engine->deploy_sel_id = i;
                                                break;
                                            }
                                        /* deploy mask may have changed */
                                        set_deploy_mask( engine->sel_unit );
                                        engine->action = UPDATE_FULL_MAP;
                                    }
                                }
                                else {
                                    engine->window_id = ID_DEPLOY_UNITS;
                                }
                                break;
                            }
                            /* left click means deploy current selected unit and get next unit from list */
                            if ( engine->sel_unit )
                            if ( event.button.button == LEFT_BUTTON && mask_tile( map_x, map_y )->deploy ) {
                                /* set position */
                                engine->deploy_units[engine->deploy_sel_id]->x = map_x;
                                engine->deploy_units[engine->deploy_sel_id]->y = map_y;
                                /* insert to map */
                                insert_unit_to_map( engine->deploy_units[engine->deploy_sel_id] );
                                /* clear cursor */
                                SDL_SetCursor( engine->stan_cursor );
                                /* the next unit got the same selection index we just need to recreate
                                the list of deployable units */
                                create_deploy_list( engine );
                                /* get next deployable unit in list */
                                engine->sel_unit = get_next_deploy_unit( engine );
                                update_unit_info( engine, 0, 0 );
                                /* if selected unit is 0 all tiles with a deployed unit on it will be highlighted */
                                set_deploy_mask( engine->sel_unit );
                                engine->action = UPDATE_FULL_MAP;
                            }
                            break;
                        }

                        /* if SHOWING_STRAT_MAP this will goto position or show unit info */
                        if ( engine->status == SHOWING_STRAT_MAP ) {
                            if ( event.button.button == LEFT_BUTTON ) {
                                if ( get_strat_map_pos( event.button.x, event.button.y, &map_x, &map_y ) ) {
                                    /* use position as center and check for borders */
                                    engine->map_x = map_x - engine->map_width / 2;
                                    engine->map_y = map_y - engine->map_height / 2;
                                    if ( engine->map_x + engine->map_width >= map.width )
                                        engine->map_x =  map.width - engine->map_width;
                                    if ( engine->map_y + engine->map_height >= map.height )
                                        engine->map_y = map.height - engine->map_height;
                                    if ( engine->map_x < 0 ) engine->map_x = 0;
                                    if ( engine->map_y < 0 ) engine->map_y = 0;
                                    engine->action = END_STRAT_MAP;
                                }
                            }
                            else {
                                /* display info? */
                                if ( get_strat_map_pos( event.button.x, event.button.y, &map_x, &map_y ) )
                                    if ( get_prim_unit( engine, map_x, map_y ) )
                                        if ( !mask_tile( map_x, map_y )->fog ) {
                                            engine->window_id = ID_UNIT_INFO;
                                            engine->sel_info_unit = get_prim_unit( engine, map_x, map_y );
                                            break;
                                        }
                            }
                            break;
                        }

                        /* no action to be taken by NOBODY! execpt info */
                        if ( engine->cont_status == NOBODY ) {
                            if ( event.button.button != RIGHT_BUTTON ) break;
                            if ( !get_map_pos( engine, event.button.x, event.button.y, &map_x, &map_y ) )
                                break;
                            /* display info? */
                            if ( get_prim_unit( engine, map_x, map_y ) ) {
                                engine->window_id = ID_UNIT_INFO;
                                engine->sel_info_unit = get_prim_unit( engine, map_x, map_y );
                            }
                            break;
                        }

                        /* check if unit has been selected for mergeing */
                        if ( event.button.button == LEFT_BUTTON )
                            if ( engine->status == SHOWING_MERGE_UNITS )
                                if ( get_map_pos( engine, event.button.x, event.button.y, &map_x, &map_y ) )
                                    if ( mask_tile( map_x, map_y )->merge )
                                        if ( ( partner = get_merge_unit( engine->sel_unit, map_x, map_y ) ) != 0 ) {
                                            merge_units( engine->sel_unit, partner );
                                            select_unit( engine, engine->sel_unit );
                                            remove_unit_from_map( partner );
                                            dl_delete_poi( &scen.units, partner );
                                            engine->action = END_MERGE_UNITS;
                                            update_cursor( engine, event.button.x, event.button.y );
                                            engine->check_unit_buttons = 1;
                                        }

                        /* select ( sets action SELECT ) */
                        if ( engine->status != SHOWING_MERGE_UNITS )
                            if ( engine->sel_unit == 0 && event.button.button == LEFT_BUTTON)
                                if ( select_unit_by_mouse( engine, event.button.x, event.button.y ) ) {
#ifdef WITH_SOUND
                                    if ( map.def->select_sound )
                                        sound_play( map.def->select_sound );
#endif
                                    break;
                                }

                        /* unselect */
                        if ( event.button.button == MIDDLE_BUTTON ) {

                            /* unselect */
                            if ( !get_map_pos( engine, event.button.x, event.button.y, &map_x, &map_y ) )
                                break;
                            select_unit( engine, 0 );
                            engine->check_unit_buttons = 1;
                            if ( engine->status == SHOWING_MERGE_UNITS )
                                engine->action = END_MERGE_UNITS;
                            else
                                engine->action = UNSELECT;
                            break;

                        }

                        /* info or unselect */
                        if ( event.button.button == RIGHT_BUTTON ) {

                            if ( !get_map_pos( engine, event.button.x, event.button.y, &map_x, &map_y ) )
                                break;
                            /* display info? */
                            if ( get_prim_unit( engine, map_x, map_y ) )
                                if ( !mask_tile( map_x, map_y )->fog ) {
                                    engine->window_id = ID_UNIT_INFO;
                                    engine->sel_info_unit = get_prim_unit( engine, map_x, map_y );
                                    break;
                                }
                            /* unselect */
                            select_unit( engine, 0 );
                            engine->check_unit_buttons = 1;
                            if ( engine->status == SHOWING_MERGE_UNITS )
                                engine->action = END_MERGE_UNITS;
                            else
                                engine->action = UNSELECT;
                            break;

                        }

                        /* only checked when WAITING */
                        if ( engine->status != WAITING ) break;

                        /* move, attack, sea-debark, sea-embark ? */
                        if ( engine->sel_unit && event.button.button == LEFT_BUTTON ) {

                            if ( !get_map_pos( engine, event.button.x, event.button.y, &map_x, &map_y ) )
                                break;

                            /* sea- debark,embark ? */
                            if ( mask_tile( map_x, map_y )->sea_embark ) {

                                if ( engine->sel_unit->embark == SEA_EMBARK ) {
                                    /* backup! */
                                    backup_unit( engine->sel_unit, &engine->backup );
                                    backup_spot_mask( engine );
                                    engine->undo_ok = 1;
                                    /* debark */
                                    debark_unit( engine, engine->sel_unit, map_x, map_y, SEA_EMBARK );
                                }
                                else
                                    if ( aban_ground_tran( engine, engine->sel_unit ) ) {
                                        /* backup! */
                                        backup_unit( engine->sel_unit, &engine->backup );
                                        backup_spot_mask( engine );
                                        engine->undo_ok = 1;
                                        /* embark */
                                        embark_unit( engine, engine->sel_unit, map_x, map_y, SEA_EMBARK );
                                    }

                                break;

                            }

                            /* attack ? */
                            if ( start_unit_attack( engine, map_x, map_y ) )
                                break;

                            /* move? */
                            if ( start_unit_move( engine, map_x, map_y ) )
                                break;

                        }
                        break;

                    case SDL_MOUSEMOTION:
                        /* show map tile info and unit info */
                        update_map_tile_info( engine, event.motion.x, event.motion.y );
                        update_unit_info( engine, event.motion.x, event.motion.y );
                        /* update non-scrolling cursor */
                        update_cursor( engine, event.motion.x, event.motion.y );
                        /* initiate scrolling */
                        if ( engine->status == WAITING || engine->status == DEPLOYING_UNITS || engine->status == SHOWING_MERGE_UNITS ) {
                            engine->old_status = engine->status;
                            reset_delay( &engine->scroll_delay );
                            scroll_map_by_mouse( engine, event.motion.x, event.motion.y );
                        }
                        /* save as last motion position */
                        engine->last_motion_x = event.motion.x;
                        engine->last_motion_y = event.motion.y;
                        break;

                }

                /* handle lower window */
                handle_window_event( engine->lower_bar, &event, &used_child );
                /* react on buttons pressed by setting action or window_id */
                if ( used_child )
                    switch ( used_child->id ) {

                        case ID_DEPLOY_BUTTON:
                            if ( engine->status == WAITING )
                                engine->action = START_DEPLOY_UNITS;
                            else
                                engine->window_id = ID_DEPLOY_UNITS;
                            break;

                        case ID_MERGE_BUTTON:
                            if ( engine->status != SHOWING_MERGE_UNITS )
                                engine->action = SHOW_MERGE_UNITS;
                            else
                                engine->action = END_MERGE_UNITS;
                            break;

                        case ID_SUPPLY_BUTTON:
                            if ( supply_unit( engine->sel_unit ) ) {

                                select_unit( engine, engine->sel_unit );
                                engine->check_unit_buttons = 1;
                                engine->action = SELECT;

                            }
                            break;

                        case ID_UNDO_BUTTON:
                            if ( !engine->undo_ok ) break;
                            engine->action = UNDO_UNIT_MOVE;
                            break;

                        case ID_RENAME_BUTTON:
                            engine->window_id = ID_RENAME_UNIT;
                            break;

                        case ID_MENU_BUTTON:
                            engine->window_id = ID_MENU;
                            break;

                        case ID_END_TURN_BUTTON:
                            engine->window_id = ID_CONF_END_TURN;
                            break;

                        case ID_AIR_MODE_BUTTON:
                            engine->air_mode = !engine->air_mode;
                            if ( engine->status != SHOWING_STRAT_MAP )
                                engine->action = SWITCH_AIR_GROUND;
                            else
                                engine->action = UPDATE_STRAT_MAP;
                            break;

                        case ID_SCEN_INFO_BUTTON:
                            engine->window_id = ID_SCEN_INFO;
                            break;

                        case ID_STRAT_MAP_BUTTON:
                            if ( engine->status == SHOWING_STRAT_MAP )
                                engine->action = END_STRAT_MAP;
                            else
                                engine->action = SHOW_STRAT_MAP;
                            break;

                        case ID_EMBARK_BUTTON:
                            if ( engine->sel_unit->embark == NO_EMBARK ) {

                                if ( unit_can_embark( engine, engine->sel_unit, engine->sel_unit->x, engine->sel_unit->y, AIR_EMBARK ) )
                                    if ( aban_ground_tran( engine, engine->sel_unit ) ) {
                                        /* backup! */
                                        backup_unit( engine->sel_unit, &engine->backup );
                                        backup_spot_mask( engine );
                                        engine->undo_ok = 1;
                                        /* embark */
                                        embark_unit( engine, engine->sel_unit, engine->sel_unit->x, engine->sel_unit->y, AIR_EMBARK );
                                        engine->check_unit_buttons = 1;
                                    }

                            }
                            else {

                                if ( unit_can_debark( engine, engine->sel_unit, engine->sel_unit->x, engine->sel_unit->y, AIR_EMBARK ) ) {
                                    /* backup! */
                                    backup_unit( engine->sel_unit, &engine->backup );
                                    backup_spot_mask( engine );
                                    engine->undo_ok = 1;
                                    /* debark */
                                    debark_unit( engine, engine->sel_unit, engine->sel_unit->x, engine->sel_unit->y, AIR_EMBARK );
                                    engine->check_unit_buttons = 1;
                                }

                            }
                            break;

                    }

            }

        /* get time */
        ms = get_time();

        /* update screen buffer and refresh the changed rects */
        update_map( engine, ms );
        if ( engine->check_unit_buttons ) {
            engine->check_unit_buttons = 0;
            check_unit_buttons( engine );
        }
        update_window( engine->lower_bar, IF_NEEDED | NO_SCREEN_REF );
        update_window( engine->upper_bar, IF_NEEDED | NO_SCREEN_REF );
        refresh_rects();
        if ( engine->block_time ) {
            /* block whole engine for some time */
            SDL_Delay( engine->block_time );
            engine->block_time = 0;
        }

        /* display map without fog when attacker won */
        if ( scen.att_team_won && !engine->final_view ) init_final_view( engine );

        /* if status is THINKING and CPU is in control initiate next action */
        if ( engine->status == THINKING && engine->cont_status == CPU )
            ai_handle( engine );

        /* now the screen is fully refreshed. if any window is asked to pop up this is done here */
        if ( engine->window_id != ID_NONE ) {

            switch ( run_window( engine ) ) {

                case END_TURN:
                    end_turn( engine );
                    break;
                case APPLY_VIDEO_MODE:
                    apply_video_mode( engine );
                    break;

            }

        }

        SDL_Delay( 5 );
    }

    FULL_DEST( sdl.screen );
    fill_surf( 0x0 );
    refresh_screen( 0, 0, 0, 0 );

    /* check return value */
    if ( scen.att_team_won ) {
         if ( scen.cur_turn < scen.major_turn_limit )
            ret = MAJOR_VIC;
         else
            ret = MINOR_VIC;
    }
    else
        if ( scen.cur_turn >= scen.minor_turn_limit )
            ret = DEFEAT;
        else
            ret = NONE;

    SDL_SetCursor( engine->stan_cursor );

    return ret;
}

/* draw map and screen pos tile */
void draw_map_tile( Engine *engine, int map_x, int map_y, int screen_x, int screen_y, int flags, int ref )
{
    /* check if this tile is fogged or not :
    DRAW_FOG is set: if map:mask::fog kep this flag
    DRAW_MOVE_RANGE: if map::mask::in_range keep this flag
    else no fog */

    /* is this tile fogged or not ? */
    if ( engine->cont_status == NOBODY )
        flags = flags & ~DRAW_FOG;
    else
    if ( flags & DRAW_FOG ) {
        if ( !mask_tile( map_x, map_y )->fog )
            flags = flags & ~DRAW_FOG;
    } else
    /* in range or not? */
    if ( flags & DRAW_MOVE_RANGE ) {
        if ( mask_tile( map_x, map_y )->in_range == -1 && !mask_tile( map_x, map_y )->sea_embark )
            flags = flags | DRAW_FOG;
        if ( mask_tile( map_x, map_y )->blocked )
            flags = flags | DRAW_FOG;
    }
    /* check if merge mask is used */
    if ( flags & DRAW_MERGE_MASK ) {
        flags = flags & ~DRAW_FOG;
        if ( !mask_tile( map_x, map_y )->merge )
            flags |= DRAW_FOG;
    }
    /* check if deploy mask is used */
    if ( flags & DRAW_DEPLOY_MASK ) {
        flags = flags & ~DRAW_FOG;
        if ( !mask_tile( map_x, map_y )->deploy )
            flags |= DRAW_FOG;
    }
    /* check if FOG was forced */
    if ( flags & DRAW_FORCED_FOG )
        flags |= DRAW_FOG;

    /* treat fog as always cached; therefore config.cache_fog MUST BE ALWAYS SET! */

    /* add cached fog -- else draw unfogged map tile */
    DEST( sdl.screen, screen_x, screen_y,
          map.def->tile_width, map.def->tile_height );
    if ( flags & DRAW_FOG )
        SOURCE( map_tile( map_x, map_y )->prop->fogged_pic,
                map_tile( map_x, map_y )->pic_offset, 0 )
    else
        SOURCE( map_tile( map_x, map_y )->prop->pic,
                map_tile( map_x, map_y )->pic_offset, 0 )
    blit_surf();

    /* draw nation flag */
    if ( map_tile( map_x, map_y )->nation_id != -1 ) {

        /* add colored frame if military target */
        if ( map_tile( map_x, map_y )->obj ) {

            DEST( sdl.screen,
                  screen_x + ( ( map.def->tile_width - scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic->w ) >> 1 ) - 1,
                  screen_y + map.def->tile_height - scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic->h - 2 - 1,
                  scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic->w + 2,
                  scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic->h + 2);
            fill_surf( 0xffff00 );

        }

        DEST( sdl.screen,
              screen_x + ( ( map.def->tile_width - scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic->w ) >> 1 ) ,
              screen_y + map.def->tile_height - scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic->h - 2,
              scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic->w,
              scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic->h );
        SOURCE( scen.nations[map_tile( map_x, map_y )->nation_id]->flag_pic, 0, 0 );
        blit_surf();

    }

    /* add grid */
    if ( config.grid ) {

        DEST( sdl.screen, screen_x, screen_y,
              map.def->tile_width, map.def->tile_height );
        SOURCE( map.def->grid_pic, 0, 0 );
        blit_surf();

    }

    if ( ref == REFRESH )
        add_refresh_rect( screen_x, screen_y,
                          map.def->tile_width, map.def->tile_height );
}

/* draw map and screen pos tile */
void draw_unit( Engine *engine, int map_x, int map_y, int screen_x, int screen_y, int flags, int ref )
{
    int bar_x = -1, bar_y = -1;
    Unit *unit = 0;

    /* check if this unit is seen or not:
    no fog: set DRAW_UNIT
    nobody in control: set DRAW_UNIT */
    if ( !mask_tile( map_x, map_y )->fog )
        flags = flags | DRAW_UNIT;
    else
    if ( engine->cont_status == NOBODY )
        flags = flags | DRAW_UNIT;

    /* draw unit ? */
    if ( flags & DRAW_UNIT ) {

        /* check in which way to draw air/ground units */
        if ( map_tile( map_x, map_y )->unit && map_tile( map_x, map_y )->air_unit ) {

            /* if one of the two units is currently selected, ignore air_mode and draw the
                selected unit big */
            if ( engine->sel_unit && engine->sel_unit == map_tile( map_x, map_y )->air_unit )
                flags = flags | DRAW_SMALL_GROUND_UNIT | DRAW_AIR_UNIT;
            else
                if ( engine->sel_unit && engine->sel_unit == map_tile( map_x, map_y )->unit )
                    flags = flags | DRAW_SMALL_AIR_UNIT | DRAW_GROUND_UNIT;
                else
                    if ( engine->air_mode )
                        flags = flags | DRAW_SMALL_GROUND_UNIT | DRAW_AIR_UNIT;
                    else
                        flags = flags | DRAW_SMALL_AIR_UNIT | DRAW_GROUND_UNIT;

        }
        else
            if ( map_tile( map_x, map_y )->unit )
                flags = flags | DRAW_GROUND_UNIT;
            else
                if ( map_tile( map_x, map_y )->air_unit )
                    flags = flags | DRAW_AIR_UNIT;

    }

    /* draw unit */
    if ( flags & DRAW_GROUND_UNIT ) {

        /* add normal-size unit */
        bar_x = screen_x + ( ( map.def->tile_width - map_tile( map_x, map_y )->unit->sel_prop->width ) >> 1 );
        bar_y = screen_y + ( ( map.def->tile_height - map_tile( map_x, map_y )->unit->sel_prop->height ) >> 1 );
        DEST( sdl.screen, bar_x, bar_y,
              map_tile( map_x, map_y )->unit->sel_prop->width,
              map_tile( map_x, map_y )->unit->sel_prop->height );
        SOURCE( map_tile( map_x, map_y )->unit->sel_prop->pic, map_tile( map_x, map_y )->unit->pic_offset, 0 );
        blit_surf();

        /* unit bar stuff */
        unit = map_tile( map_x, map_y )->unit;
        /* bar_x += ( unit->sel_prop->width - BAR_WIDTH ) >> 1; */
        bar_x = screen_x + ( ( map.def->tile_width - BAR_WIDTH + 1 ) >> 1 );
        bar_y = screen_y + map.def->tile_height - BAR_HEIGHT - 4;
        /* bar_y += unit->sel_prop->height; */

    }
    else
        if ( flags & DRAW_SMALL_GROUND_UNIT ) {

            /* add small unit */
            DEST( sdl.screen,
                  screen_x +
                  ( ( map.def->tile_width - ( map_tile( map_x, map_y )->unit->sel_prop->width >> 1 ) ) >> 1 ),
                  screen_y + map.def->tile_height - ( map_tile( map_x, map_y )->unit->sel_prop->height >> 1 ),
                  map_tile( map_x, map_y )->unit->sel_prop->width >> 1,
                  map_tile( map_x, map_y )->unit->sel_prop->height >> 1);
            SOURCE( map_tile( map_x, map_y )->unit->sel_prop->small_pic, map_tile( map_x, map_y )->unit->pic_offset >> 1, 0 );
            blit_surf();

        }

    if ( flags & DRAW_AIR_UNIT ) {

        /* add normal-size unit */
        bar_x = screen_x + ( ( map.def->tile_width - map_tile( map_x, map_y )->air_unit->sel_prop->width ) >> 1 );
        bar_y = screen_y + ( ( map.def->tile_height - map_tile( map_x, map_y )->air_unit->sel_prop->height ) >> 1 );
        DEST( sdl.screen, bar_x, bar_y,
              map_tile( map_x, map_y )->air_unit->sel_prop->width,
              map_tile( map_x, map_y )->air_unit->sel_prop->height );
        SOURCE( map_tile( map_x, map_y )->air_unit->sel_prop->pic, map_tile( map_x, map_y )->air_unit->pic_offset, 0 );
        blit_surf();

        /* bar stuff */
        unit = map_tile( map_x, map_y )->air_unit;
        /* bar_x += ( unit->sel_prop->width - BAR_WIDTH ) >> 1; */
        bar_x = screen_x + ( ( map.def->tile_width - BAR_WIDTH + 1 ) >> 1 );
        bar_y = screen_y + map.def->tile_height - BAR_HEIGHT - 4;
        /* bar_y += unit->sel_prop->height; */

    }
    else
        if ( flags & DRAW_SMALL_AIR_UNIT ) {

            /* add small unit */
            DEST( sdl.screen,
                  screen_x +
                  ( ( map.def->tile_width - ( map_tile( map_x, map_y )->air_unit->sel_prop->width >> 1 ) ) >> 1 ),
                  screen_y,
                  map_tile( map_x, map_y )->air_unit->sel_prop->width >> 1,
                  map_tile( map_x, map_y )->air_unit->sel_prop->height >> 1);
            SOURCE( map_tile( map_x, map_y )->air_unit->sel_prop->small_pic, map_tile( map_x, map_y )->air_unit->pic_offset >> 1, 0 );
            blit_surf();

        }

    /* show unit status */
    /* if bar_x = bar_y = -1 no unit was drawn */
    if ( bar_x != -1 && config.show_bar && unit && !( flags & NO_SEL_FRAME ) /* && unit == engine->sel_unit */ ) {

        /* background box; if unit can't move: black else white */
        DEST( sdl.screen, bar_x, bar_y, BAR_WIDTH, BAR_HEIGHT );
        fill_surf( 0x111111 );
        /* copy from map::life_icons */
        DEST( sdl.screen, bar_x, bar_y, unit->damage_bar_width, BAR_TILE_HEIGHT );
        SOURCE( map.def->damage_bar_icons, 0, unit->damage_bar_offset );
        blit_surf();

    }

    /* add selection frame */
    if ( engine->cont_status == HUMAN && engine->sel_unit && engine->sel_unit->x == map_x &&
         engine->sel_unit->y == map_y && !( flags & NO_SEL_FRAME ) &&
         !mask_tile( map_x, map_y )->fog )
    {

        DEST( sdl.screen, screen_x, screen_y,
              map.def->tile_width, map.def->tile_height );
        SOURCE( map.def->sel_pic, 0, 0 );
        blit_surf();

    }

    /* add crosshair */
    if ( flags & DRAW_CROSS ) {

        DEST( sdl.screen, screen_x, screen_y,
              map.def->tile_width, map.def->tile_height );
        SOURCE( map.def->cross_pic, 0, 0 );
        blit_surf();

    }

    if ( ref == REFRESH )
        add_refresh_rect( screen_x, screen_y,
                          map.def->tile_width, map.def->tile_height );
}

/* redraw map tile and unit by getting position first; only use for updating single tiles */
void update_map_tile( Engine *engine, int map_x, int map_y, int flags )
{
    int screen_x, screen_y;
    get_screen_pos( engine, map_x, map_y, &screen_x, &screen_y );
    draw_map_tile( engine, map_x, map_y, screen_x, screen_y, flags, REFRESH );
    draw_unit( engine, map_x, map_y, screen_x, screen_y, flags, REFRESH );
}

/*
====================================================================
Determine wether to use no fog (dont change flag), fog of war
(add DRAW_FOG ) or move_range (add DRAW_MOVE_RANGE ).
====================================================================
*/
void check_fog_style( Engine *engine, int *flags )
{
    Unit *att_unit;

    /* a computer does not show a moving range if OWN_FOG is set and if FOREIGN_FOG */
    /* is set nothing happens to the fog either */
    if ( engine->fog_status == FOREIGN_FOG || engine->cont_status == CPU )
        *flags = DRAW_FOG;
    else {
        /* human with OWN_FOG is in control */
        /* if the attacking unit has no move points left use fog */
        /* determine attacking unit by checking sel_pred */
        /* if engine::sel_pred is != 0 supporting fire is going on */
        /* and the attacker is engine::sel_pred::att else it's simply engine::sel_unit */
        att_unit = engine->sel_unit;
        if ( engine->sel_pred ) att_unit = engine->sel_pred->att;
        if ( att_unit ) {
            if ( att_unit->cur_prop.mov == 0 )
                *flags = DRAW_FOG;
            else
                *flags = DRAW_MOVE_RANGE;
        }
    }
}

/* update map by checking action/status types and change the screen buffer */
void update_map( Engine *engine, int ms )
{
    int screen_x, screen_y;
    int goto_x, goto_y;
    float vec_length;
    int i;
    int kill = 0;
    char str[128];
    int att_damage, def_damage;
    int att_died;
    int enemy_spotted = 0;
    int flags;

    /* clip to map */
    set_surf_clip( sdl.screen, 0, engine->upper_bar->height, sdl.screen->w,
                   sdl.screen->h - engine->upper_bar->height  - engine->lower_bar->height );

    /* actions which must be checked before status */
    switch ( engine->action ) {

        case BATTLE:
            combat( engine->sel_unit, engine->sel_target, engine->combat_type, &att_damage, &def_damage );
            /* only animate when seen */
            if ( engine->cont_status != CPU || config.show_cpu_turn ) {
                /* rugged defence? ambush ?*/
                str[0] = 0;
                if ( engine->ambush ) {
                    if ( engine->sel_unit->sel_prop->flags & FLYING )
                        sprintf( str, "Out of the Sun!" );
                    else
                        if ( engine->sel_unit->sel_prop->flags & SWIMMING )
                            sprintf( str, "Surprise Contact!" );
                        else
                            sprintf( str, "Ambush!" );
                }
                else
                    if ( engine->sel_target->rugged_def )
                        sprintf( str, "Rugged Defence!" );
                /* display message and wait for a second */
                if ( str[0] != 0 ) {
                    /* reset clip */
                    set_surf_clip( sdl.screen, 0, 0, 0, 0 );
                    /* display message */
                    set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ), str, REFRESH );
                    set_simple_window_text( get_child( engine->upper_bar, ID_LEFT_LABEL ), "", REFRESH );
                    set_simple_window_text( get_child( engine->upper_bar, ID_RIGHT_LABEL ), "", REFRESH );
                    update_window( engine->upper_bar, 0 );
                    SDL_Delay( 1000 );
                    set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ), "", REFRESH );
                    /* clip to map */
                    set_surf_clip( sdl.screen, 0, engine->upper_bar->height, sdl.screen->w,
                                   sdl.screen->h - engine->upper_bar->height  - engine->lower_bar->height );
                }
#ifdef WITH_SOUND
            if ( engine->cont_status != CPU || config.show_cpu_turn )
                if ( map.def->expl_sound ) {
                    if ( def_damage )
                        sound_play( map.def->expl_sound );
                    if ( att_damage )
                        sound_play( map.def->expl_sound );
                }
#endif
                /* show real damages in percent in upper bar */
                sprintf( str, "Attacker: %i%%", att_damage * 10 );
                set_simple_window_text( get_child( engine->upper_bar, ID_LEFT_LABEL ), str, REFRESH );
                sprintf( str, "Defender: %i%%", def_damage * 10 );
                set_simple_window_text( get_child( engine->upper_bar, ID_RIGHT_LABEL ), str, REFRESH );
                /* set up both explosions if seen */
                /* positions */
                get_screen_pos( engine, engine->sel_unit->x, engine->sel_unit->y,
                                &engine->expl_pos.x, &engine->expl_pos.y );
                engine->expl_pos.x += ( map.def->tile_width -
                                        engine->expl_frame_width ) / 2;
                engine->expl_pos.y += ( map.def->tile_height -
                                        engine->expl_frame_height ) / 2;
                get_screen_pos( engine, engine->sel_target->x, engine->sel_target->y,
                                &engine->expl_pos2.x, &engine->expl_pos2.y );
                engine->expl_pos2.x += ( map.def->tile_width -
                                         engine->expl_frame_width ) / 2;
                engine->expl_pos2.y += ( map.def->tile_height -
                                         engine->expl_frame_height ) / 2;
                /* buffers -- don't use if unit not seen */
                engine->expl_buffer = engine->expl_buffer2 = 0;
                if ( att_damage > 0 &&
                     !mask_tile( engine->sel_unit->x, engine->sel_unit->y )->fog
                )
                    engine->expl_buffer = create_surf( engine->expl_frame_width, engine->expl_frame_height,
                                                       SDL_SWSURFACE );
                if ( def_damage > 0 &&
                     !mask_tile( engine->sel_target->x, engine->sel_target->y )->fog
                )
                    engine->expl_buffer2 = create_surf( engine->expl_frame_width, engine->expl_frame_height,
                                                        SDL_SWSURFACE );
                /* save background */
                if ( engine->expl_buffer ) {

                    FULL_DEST( engine->expl_buffer );
                    SOURCE( sdl.screen, engine->expl_pos.x, engine->expl_pos.y );
                    blit_surf();

                }
                if ( engine->expl_buffer2 ) {

                    FULL_DEST( engine->expl_buffer2 );
                    SOURCE( sdl.screen, engine->expl_pos2.x, engine->expl_pos2.y );
                    blit_surf();

               }
                /* first frame */
                engine->expl_delay.cur = engine->expl_delay.limit; /*use TAKING_DAMAGE code to initiate */
                engine->expl_frame_offset = -engine->expl_frame_width;
                /* start it! */
                engine->action = NONE;
                engine->status = TAKING_DAMAGE;
            }
            else
                engine->action = END_BATTLE;
            break;

        case SHOW_ATT_CROSS:
            /* CPU in control and not seen ? */
            if ( engine->cont_status == CPU )
                if ( !config.show_cpu_turn || mask_tile( engine->sel_unit->x, engine->sel_unit->y )->fog ) {
                    engine->action = SHOW_DEF_CROSS; /* try next one */
                    break;
                }
            /* maybe it's supporting fire and this tile is fogged? */
            if ( mask_tile( engine->sel_unit->x, engine->sel_unit->y )->fog ) {
                engine->action = SHOW_DEF_CROSS;
                break;
            }
            /* not seen? goto_map_pos! */
            if ( engine->sel_unit->x <= engine->map_x || engine->sel_unit->y <= engine->map_y ||
                 engine->sel_unit->x >= engine->map_x + engine->map_width - 1 ||
                 engine->sel_unit->y >= engine->map_y + engine->map_height - 1 )
                {

                goto_x = engine->sel_unit->x - ( engine->map_width >> 1 );
                goto_y = engine->sel_unit->y - ( engine->map_height >> 1 );
                goto_map_pos( engine, goto_x, goto_y, SCREEN_COPY );
                update_full_map( engine, NO_CLIP );

            }
            /* show crosshair */
            get_screen_pos( engine, engine->sel_unit->x, engine->sel_unit->y, &screen_x, &screen_y );
            /* redraw map tile with crosshair */
            /* CPU does not show move limitation therefore ignore it */
            flags = 0;
            check_fog_style( engine, &flags );
            draw_map_tile( engine, engine->sel_unit->x, engine->sel_unit->y, screen_x, screen_y, flags, REFRESH );
            draw_unit( engine, engine->sel_unit->x, engine->sel_unit->y, screen_x, screen_y, DRAW_CROSS, REFRESH );
            engine->action = NONE;
            engine->status = SHOWING_ATT_CROSS;
            break;

        case SHOW_DEF_CROSS:
            /* CPU in control and not seen ? */
            if ( engine->cont_status == CPU )
                if ( !config.show_cpu_turn || mask_tile( engine->sel_target->x, engine->sel_target->y )->fog ) {
                    engine->action = BATTLE; /* compute damages */
                    break;
                }
            /* not seen? goto_map_pos! */
            /* not seen? goto_map_pos! */
            if ( engine->sel_target->x <= engine->map_x || engine->sel_target->y <= engine->map_y ||
                 engine->sel_target->x >= engine->map_x + engine->map_width - 1 ||
                 engine->sel_target->y >= engine->map_y + engine->map_height - 1 )
                {

                goto_x = engine->sel_target->x - ( engine->map_width >> 1 );
                goto_y = engine->sel_target->y - ( engine->map_height >> 1 );
                goto_map_pos( engine, goto_x, goto_y, SCREEN_COPY );
                update_full_map( engine, NO_CLIP );

            }
            /* show crosshair */
            get_screen_pos( engine, engine->sel_target->x, engine->sel_target->y, &screen_x, &screen_y );
            /* redraw map tile with crosshair */
            /* CPU does not show move limitation therefore ignore it */
            flags = 0;
            check_fog_style( engine, &flags );
            draw_map_tile( engine, engine->sel_target->x, engine->sel_target->y, screen_x, screen_y, flags, REFRESH );
            draw_unit( engine, engine->sel_target->x, engine->sel_target->y, screen_x, screen_y, DRAW_CROSS, REFRESH );
            engine->action = NONE;
            engine->status = SHOWING_DEF_CROSS;
            break;

        case START_MOVE:
            /* check if CPU is in control, FOREIGN_FOG or !show_cpu_turn or covered by fog;
               loop until end or tile is found where source and dest are not covered with fog */

            /* delete unit at the very beginning of the way (way_pos = 0 ) and insert at the very end and
            not within because then own units might vanish */

            /* delete the unit from map tile */
            if ( engine->way_pos == 0 )
                remove_unit_from_map( engine->sel_unit );

#ifdef WITH_SOUND
            /* play move sound once in the beginning */
            if ( engine->way_pos == 0 )
                if ( engine->cont_status != CPU || config.show_cpu_turn )
                    if ( engine->sel_unit->sel_prop->move_sound )
                        sound_play( engine->sel_unit->sel_prop->move_sound );
#endif

            /* get new way_pos id */
            if ( engine->cont_status == CPU && !config.show_cpu_turn )
                engine->way_pos = engine->way_length - 1;
            else
                if ( engine->fog_status == FOREIGN_FOG ) {

                    i = engine->way_pos;

                    /* show cpu turn but skip movement on covered map tiles */
                    while ( i + 1 < engine->way_length &&
                            mask_tile( engine->way[i].x, engine->way[i].y )->fog &&
                            mask_tile( engine->way[i].x + 1, engine->way[i].y + 1 )->fog )
                        i++;
                    engine->way_pos = i;
                }

            /* change selected unit position */
            engine->sel_unit->x = engine->way[engine->way_pos].x;
            engine->sel_unit->y = engine->way[engine->way_pos].y;

            /* check if last way_point has been reached  */
            /* if engine::ambush_unit is set an ambush follows */
            if ( engine->way_pos == engine->way_length - 1 ) {

                insert_unit_to_map( engine->sel_unit );

                if ( engine->cont_status == HUMAN )
                    engine->status = WAITING; /* wait for new input */
                else
                    engine->status = THINKING; /* request next cpu action */

                /* unit is selected anyway and needs to be redrawn */
                select_unit( engine, engine->sel_unit );
                if ( engine->cont_status != CPU || config.show_cpu_turn )
                    engine->check_unit_buttons = 1;
                engine->action = SELECT;

                /* capture flag if there is one */
                if ( !( engine->sel_unit->sel_prop->flags & FLYING ) &&
                     map_tile(engine->sel_unit->x,engine->sel_unit->y)->nation_id != - 1
                ) {

                    map_tile( engine->sel_unit->x, engine->sel_unit->y )->nation_id = engine->sel_unit->sel_prop->nation;
                    map_tile( engine->sel_unit->x, engine->sel_unit->y )->player_id = engine->sel_unit->player_id;

                }

                /* check if victory conditions are fullfilled if player is in
                attacking team */
                if ( is_att( engine->player ) )
                    check_victory_cond( );

                /* free back_buffer */
                SDL_FreeSurface( engine->back_buffer );

                if ( engine->ambush_unit ) {
                    start_unit_ambush( engine );
                    engine->ambush_unit = 0;
                }

                break;

            }

            /* first goto_map_pos() if needed ( is needed when unit moves out of screen ) */
            if ( engine->sel_unit->x <= engine->map_x + 1 || engine->sel_unit->y <= engine->map_y + 1 ||
                 engine->sel_unit->x >= engine->map_x + engine->map_width - 1 - 2 ||
                 engine->sel_unit->y >= engine->map_y + engine->map_height - 1 - 2 )
                {

                goto_x = engine->sel_unit->x - ( engine->map_width >> 1 );
                goto_y = engine->sel_unit->y - ( engine->map_height >> 1 );
                goto_map_pos( engine, goto_x, goto_y, SCREEN_COPY );
                update_full_map( engine, NO_CLIP );

            }

            /* check direction */
            check_unit_dir( engine->sel_unit,
                            engine->way[engine->way_pos + 1].x, engine->way[engine->way_pos + 1].y );

            /* get screen position */
            get_screen_pos( engine, engine->way[engine->way_pos].x, engine->way[engine->way_pos].y,
                            &screen_x, &screen_y );

            /* draw_map_tile with fog if not OWN_FOG */
            if ( engine->fog_status == FOREIGN_FOG ) {
                draw_map_tile( engine, engine->way[engine->way_pos].x, engine->way[engine->way_pos].y, screen_x, screen_y, DRAW_FOG | NO_SEL_FRAME, REFRESH );
                draw_unit( engine, engine->way[engine->way_pos].x, engine->way[engine->way_pos].y, screen_x, screen_y, NO_SEL_FRAME, REFRESH );
            }
            else {
                draw_map_tile( engine, engine->way[engine->way_pos].x, engine->way[engine->way_pos].y, screen_x, screen_y, DRAW_MOVE_RANGE | NO_SEL_FRAME, REFRESH );
                draw_unit( engine, engine->way[engine->way_pos].x, engine->way[engine->way_pos].y, screen_x, screen_y, NO_SEL_FRAME, REFRESH );
            }

            /* get start and end position for movement from one to next tile */
            engine->start_pos.x = screen_x +
                                  ( ( map.def->tile_width - engine->sel_unit->sel_prop->width ) >> 1 );
            engine->start_pos.y = screen_y +
                                  ( ( map.def->tile_height - engine->sel_unit->sel_prop->height ) >> 1 );
            get_screen_pos( engine, engine->way[engine->way_pos + 1].x, engine->way[engine->way_pos + 1].y,
                            &screen_x, &screen_y );
            engine->end_pos.x = screen_x +
                                ( ( map.def->tile_width - engine->sel_unit->sel_prop->width ) >> 1 );
            engine->end_pos.y = screen_y +
                                ( ( map.def->tile_height - engine->sel_unit->sel_prop->height ) >> 1 );

            /* get background */
            DEST( engine->back_buffer, 0, 0, engine->sel_unit->sel_prop->width, engine->sel_unit->sel_prop->height );
            SOURCE( sdl.screen, (int)engine->start_pos.x, (int)engine->start_pos.y );
            blit_surf();

            /* compute move_vector from current way_point and destination way_point */
            engine->move_vector.x = engine->end_pos.x - engine->start_pos.x;
            engine->move_vector.y = engine->end_pos.y - engine->start_pos.y;
            /* norm move vector */
            vec_length = sqrt( engine->move_vector.x * engine->move_vector.x +
                               engine->move_vector.y * engine->move_vector.y );
            engine->move_vector.x /= vec_length;
            engine->move_vector.y /= vec_length;

            /* compute time unit will need to move */
            set_delay( &engine->move_time, (int)( vec_length / engine->unit_vel ) );

            /* clear action */
            engine->action = NONE;
            break;

    }

    /* check status */
    switch ( engine->status ) {

        case TAKING_DAMAGE:
            if ( timed_out( &engine->expl_delay, ms ) ) {

                engine->expl_frame_offset += engine->expl_frame_width;
                if ( engine->expl_frame_offset >= map.def->expl_pic->w ) {

                    /* free explosion buffers */
                    if ( engine->expl_buffer ) SDL_FreeSurface( engine->expl_buffer );
                    if ( engine->expl_buffer2 ) SDL_FreeSurface( engine->expl_buffer2 );

                    /* clean up */
                    engine->action = END_BATTLE;
                    break;

                }

                if ( engine->expl_buffer ) {

                    /* put background */
                    DEST( sdl.screen, engine->expl_pos.x, engine->expl_pos.y,
                          engine->expl_frame_width, engine->expl_frame_height );
                    SOURCE( engine->expl_buffer, 0, 0 );
                    blit_surf();
                    /* save background */
                    FULL_DEST( engine->expl_buffer );
                    SOURCE( sdl.screen, engine->expl_pos.x, engine->expl_pos.y );
                    blit_surf();
                    /* put explosion */
                    DEST( sdl.screen, engine->expl_pos.x, engine->expl_pos.y,
                          engine->expl_frame_width, engine->expl_frame_height );
                    SOURCE( map.def->expl_pic, engine->expl_frame_offset, 0 );
                    blit_surf();
                    add_refresh_rect( engine->expl_pos.x, engine->expl_pos.y,
                                      engine->expl_frame_width, engine->expl_frame_height );

                }
                if ( engine->expl_buffer2 ) {

                    /* put background */
                    DEST( sdl.screen, engine->expl_pos2.x, engine->expl_pos2.y,
                          engine->expl_frame_width, engine->expl_frame_height );
                    SOURCE( engine->expl_buffer2, 0, 0 );
                    blit_surf();
                    /* save background */
                    FULL_DEST( engine->expl_buffer2 );
                    SOURCE( sdl.screen, engine->expl_pos2.x, engine->expl_pos2.y );
                    blit_surf();
                    /* put explosion */
                    DEST( sdl.screen, engine->expl_pos2.x, engine->expl_pos2.y,
                          engine->expl_frame_width, engine->expl_frame_height );
                    SOURCE( map.def->expl_pic, engine->expl_frame_offset, 0 );
                    blit_surf();
                    add_refresh_rect( engine->expl_pos2.x, engine->expl_pos2.y,
                                      engine->expl_frame_width, engine->expl_frame_height );

               }

            }
            break;

        case SHOWING_ATT_CROSS:
            /* next action is show_def_cross */
            if ( timed_out( &engine->cross_delay, ms ) ) {

                /* redraw maptile without crosshair */
                get_screen_pos( engine, engine->sel_unit->x, engine->sel_unit->y, &screen_x, &screen_y );
                flags = 0;
                check_fog_style( engine, &flags );
                draw_map_tile( engine, engine->sel_unit->x, engine->sel_unit->y, screen_x, screen_y, flags, REFRESH );
                draw_unit( engine, engine->sel_unit->x, engine->sel_unit->y, screen_x, screen_y, 0, REFRESH );

                /* next action */
                engine->action = SHOW_DEF_CROSS;
                engine->status = ATTACKING;

            }
            break;

        case SHOWING_DEF_CROSS:
            /* next action is show_def_cross */
            if ( timed_out( &engine->cross_delay, ms ) ) {

                /* redraw maptile without crosshair */
                get_screen_pos( engine, engine->sel_target->x, engine->sel_target->y, &screen_x, &screen_y );
                flags = 0;
                check_fog_style( engine, &flags );
                draw_map_tile( engine, engine->sel_target->x, engine->sel_target->y, screen_x, screen_y, flags, REFRESH );
                draw_unit( engine, engine->sel_target->x, engine->sel_target->y, screen_x, screen_y, 0, REFRESH );

                /* next action */
                engine->action = BATTLE;
                engine->status = ATTACKING;

            }
            break;

        case MOVING:
            /* animate movement along ( engine->start_pos + t * engine->move_vector ) */
            /* put background */
            DEST( sdl.screen, (int)engine->start_pos.x, (int)engine->start_pos.y,
                  engine->sel_unit->sel_prop->width, engine->sel_unit->sel_prop->height );
            SOURCE( engine->back_buffer, 0, 0 );
            blit_surf();
            add_refresh_rect( (int)engine->start_pos.x, (int)engine->start_pos.y,
                              engine->sel_unit->sel_prop->width, engine->sel_unit->sel_prop->height );
            /* update position -- engine->start_pos */
            engine->start_pos.x += engine->unit_vel * engine->move_vector.x * ms;
            engine->start_pos.y += engine->unit_vel * engine->move_vector.y * ms;
            /* get background */
            DEST( engine->back_buffer, 0, 0, engine->sel_unit->sel_prop->width, engine->sel_unit->sel_prop->height );
            SOURCE( sdl.screen, (int)engine->start_pos.x, (int)engine->start_pos.y );
            blit_surf();
            /* draw unit */
            DEST( sdl.screen, (int)engine->start_pos.x, (int)engine->start_pos.y,
                  engine->sel_unit->sel_prop->width, engine->sel_unit->sel_prop->height );
            SOURCE( engine->sel_unit->sel_prop->pic, engine->sel_unit->pic_offset, 0 );
            blit_surf();
            add_refresh_rect( (int)engine->start_pos.x, (int)engine->start_pos.y,
                              engine->sel_unit->sel_prop->width, engine->sel_unit->sel_prop->height );

            /* when engine->move_time timed out next START_MOVE */
            if ( timed_out( &engine->move_time, ms ) ) {

                /* next way point */
                engine->way_pos++;
                engine->action = START_MOVE;

                /* change selected unit position */
                engine->sel_unit->x = engine->way[engine->way_pos].x;
                engine->sel_unit->y = engine->way[engine->way_pos].y;

                /* update spot mask; if fog_status is OWN_FOG this will update the fog as well */
                enemy_spotted = update_spot_mask( engine, engine->sel_unit );
                /* if you spotted an enemy it's not allowed to undo the turn */
                if ( enemy_spotted ) {
                    deny_undo( engine );
                    engine->check_unit_buttons = 1;
                }
                /* unit moved so check if influencing enemy was spotted */
                set_vis_infl_mask( engine );

            }
            break;

        case SCROLLING_BY_MOUSE:
            /* just scroll, don't update screen because SCROLLING causes GOTO_POS action which
            updates screen below */
            if ( timed_out( &engine->scroll_delay, ms ) )
                scroll_map_by_mouse( engine, engine->last_motion_x, engine->last_motion_y );
            break;

        case SCROLLING_BY_KEY:
            if ( timed_out( &engine->scroll_delay, ms ) )
                scroll_map_by_key( engine, engine->scroll_dir );
            break;

    }

    /* check other action */
    switch ( engine->action ) {

        case UPDATE_FULL_MAP:
            update_full_map( engine, NO_CLIP );
            engine->action = NONE;
            break;

        case START_DEPLOY_UNITS:
            select_unit( engine, 0 ); /* clear selection */
            engine->check_unit_buttons = 1; /* clear unit buttons */
            reset_avail_reinf( engine->player );
            engine->window_id = ID_DEPLOY_UNITS;
            engine->action = NONE;
            engine->status = DEPLOYING_UNITS;
            /* deny save and end turn */
            set_active( get_child( engine->lower_bar, ID_END_TURN_BUTTON ), 0, REFRESH );
            set_active( get_child( engine->menu, ID_SAVE_MENU ), 0, NO_REFRESH );
            break;

        case END_DEPLOY_UNITS:
            engine->action = NONE;
            if ( engine->cont_status == CPU )
                engine->status = THINKING;
            else
                engine->status = WAITING;
            /* unselect deploy unit */
            engine->sel_unit = 0;
            /* check deploy button */
            check_deploy_button( engine );
            /* update */
            engine->old_status = WAITING;
            update_full_map( engine, NO_CLIP );
            /* reallow save and end turn */
            set_active( get_child( engine->lower_bar, ID_END_TURN_BUTTON ), 1, REFRESH );
            set_active( get_child( engine->menu, ID_SAVE_MENU ), 1, NO_REFRESH );
            break;

        case SHOW_MERGE_UNITS:
            engine->action = NONE;
            engine->status = SHOWING_MERGE_UNITS;
            update_full_map( engine, NO_CLIP );
            break;
        case END_MERGE_UNITS:
            if ( engine->cont_status == CPU )
                engine->status = THINKING;
            else
                engine->status = WAITING;
            engine->old_status = WAITING;
            update_full_map( engine, NO_CLIP );
            engine->action = NONE;
            break;

        case SHOW_STRAT_MAP:
            strat_map_update( engine, RECREATE_STRAT_MAP | RECREATE_UNIT_LAYER | RECREATE_FLAG_LAYER );
            engine->action = NONE;
            engine->old_status = engine->status;
            engine->status = SHOWING_STRAT_MAP;
            set_active( get_child( engine->lower_bar, ID_SUPPLY_BUTTON ), 0, REFRESH );
            set_active( get_child( engine->lower_bar, ID_EMBARK_BUTTON ), 0, REFRESH );
            set_active( get_child( engine->lower_bar, ID_RENAME_BUTTON ), 0, REFRESH );
            set_active( get_child( engine->lower_bar, ID_UNDO_BUTTON ), 0, REFRESH );
            set_active( get_child( engine->lower_bar, ID_MERGE_BUTTON ), 0, REFRESH );
            /* while strat map's displayed you may not select a deploy unit */
            set_active( get_child( engine->lower_bar, ID_DEPLOY_BUTTON ), 0, REFRESH );
            break;
        case UPDATE_STRAT_MAP:
            strat_map_update( engine, RECREATE_UNIT_LAYER | RECREATE_FLAG_LAYER );
            engine->action = NONE;
            break;
        case END_STRAT_MAP:
            engine->action = NONE;
            engine->status = engine->old_status;
            engine->old_status = WAITING;
            /* do only update unit buttons if status is WAITING */
            if ( engine->status == WAITING )
                engine->check_unit_buttons = 1;
            /* restore deploy button */
            check_deploy_button( engine );
            update_full_map( engine, NO_CLIP );
            break;

        case GOTO_POS:
        case SWITCH_AIR_GROUND:
        case SELECT:
        case UNSELECT:
            if ( engine->cont_status != CPU || config.show_cpu_turn ) {
                update_full_map( engine, NO_CLIP );
                engine->action = NONE;
            }
            break;
        case DEBARK:
        case EMBARK:
            if ( engine->cont_status != CPU || config.show_cpu_turn ) {
                select_unit( engine, engine->sel_unit );
                engine->check_unit_buttons = 1;
                update_full_map( engine, NO_CLIP );
                engine->action = NONE;
            }
            break;
        case UNDO_UNIT_MOVE:
            remove_unit_from_map( engine->sel_unit );
            restore_unit( engine->sel_unit, &engine->backup );
            insert_unit_to_map( engine->sel_unit );
            deny_undo( engine );
            /* restore spot mask */
            restore_spot_mask( engine );
            /* update screen */
            if ( engine->cont_status != CPU || config.show_cpu_turn ) {
                select_unit( engine, engine->sel_unit );
                engine->check_unit_buttons = 1;
                update_full_map( engine, NO_CLIP );
            }
            engine->action = NONE;
            break;
        case END_BATTLE:
            att_died = 0;
            /* update current properties */
            update_cur_unit_prop( engine->sel_unit );
            update_cur_unit_prop( engine->sel_target );
            /* reset rugged defence flag */
            engine->sel_unit->rugged_def = engine->sel_target->rugged_def = 0;
            /* check if units got killed */
            if ( engine->sel_unit->killed ) {
                remove_unit( engine, engine->sel_unit );
                engine->sel_unit = 0;
            }
            if ( engine->sel_target->killed ) {
                /* if sel_target == pred->att this was supporting fire and the attacker died */
                if ( engine->sel_pred && engine->sel_pred->att == engine->sel_target ) {
                    att_died = 1;
                    engine->sel_unit = 0;
                }
                remove_unit( engine, engine->sel_target );
                engine->sel_target = 0;
                kill = 1;
            }
            /* clear ambush */
            engine->ambush = 0;
            /* initiate next fight; */
            if ( att_died || !init_next_fight( engine ) ) {
                /* clear suppression from defensive fire */
                if ( engine->sel_unit )
                    engine->sel_unit->sup = 0;
                /* selected unit attacked */
                if ( engine->sel_unit )
                    engine->sel_unit->no_action_yet = 0;
                /* either clear selection or update moving range by reselection */
                if ( !engine->sel_unit || kill ) {
                    select_unit( engine, engine->sel_unit );
                }
                if ( engine->cont_status != CPU || config.show_cpu_turn )
                    engine->check_unit_buttons = 1;
                /* update predicitions */
                if ( engine->sel_unit ) {
                    FREE( engine->pred );
                    engine->pred = get_pred( engine->sel_unit, &engine->pred_count );
                }
                /* reset status */
                engine->action = NONE;
                if ( engine->cont_status == HUMAN )
                    engine->status = WAITING; /* attack finished */
                else
                    engine->status = THINKING;
                /* redraw map */
                if ( engine->cont_status != CPU || config.show_cpu_turn ) {
                    /* update full map */
                    update_full_map( engine, NO_CLIP );
                }
            }
            if ( engine->cont_status != CPU || config.show_cpu_turn ) {
                if ( engine->cont_status != CPU || config.show_cpu_turn ) {
                    /* determine the update flag */
                    flags = 0;
                    check_fog_style( engine, &flags );
                    /* update map tiles */
                    /* selected unit */
                    if ( engine->sel_unit && !mask_tile( engine->sel_unit->x, engine->sel_unit->y )->fog )
                        update_map_tile( engine, engine->sel_unit->x, engine->sel_unit->y, flags );
                    /* selected target */
                    if ( engine->sel_target && !mask_tile( engine->sel_target->x, engine->sel_target->y )->fog )
                        update_map_tile( engine, engine->sel_target->x, engine->sel_target->y, flags );
                }
                /* clear damage info */
                set_simple_window_text( get_child( engine->upper_bar, ID_LEFT_LABEL ), "", REFRESH );
                set_simple_window_text( get_child( engine->upper_bar, ID_RIGHT_LABEL ), "", REFRESH );
                /* check unit buttons */
                engine->check_unit_buttons = 1;
            }
            break;


    }

    /* reset clip */
    set_surf_clip( sdl.screen, 0, 0, 0, 0 );
}

/* simply update full map */
void update_full_map( Engine *engine, int clip )
{
    int x, y, abs_y;
    int i, j;
    int start_map_x, start_map_y, end_map_x, end_map_y;
    int buffer_height, buffer_width, buffer_offset;
    int flags; /* for draw_map_tile/unit */

    /* TEST */
/*    reset_timer(); */

    /* clip to map */
    if ( clip == CLIP )
        set_surf_clip( sdl.screen, 0, engine->upper_bar->height, sdl.screen->w,
                       sdl.screen->h - engine->upper_bar->height  - engine->lower_bar->height );

    /* which mask to use? */
    flags = DRAW_FOG;
    if ( engine->fog_status == NO_FOG )
        flags = 0;
    else
        if ( engine->sel_unit && engine->fog_status == OWN_FOG &&
             engine->cont_status == HUMAN && engine->sel_unit->cur_prop.mov > 0
        )
            flags = DRAW_MOVE_RANGE;
    if ( engine->status == SHOWING_MERGE_UNITS || engine->old_status == SHOWING_MERGE_UNITS )
        flags = DRAW_MERGE_MASK;
    if ( engine->status == DEPLOYING_UNITS || engine->old_status == DEPLOYING_UNITS )
        flags = DRAW_DEPLOY_MASK;

    /* screen copy? */
    start_map_x = engine->map_x;
    start_map_y = engine->map_y;
    end_map_x = engine->map_x + engine->map_width;
    end_map_y = engine->map_y + engine->map_height;
    if ( engine->screen_copy_type == VERT_SCREEN_COPY ) {

        /* clear flag */
        engine->screen_copy_type = NO_SCREEN_COPY;

        /* set buffer offset and height */
        buffer_offset = abs( engine->screen_copy_diff ) * map.def->tile_height;
        buffer_height = sdl.screen->h - engine->upper_bar->height - engine->lower_bar->height -
                        buffer_offset;
        /* going down */
        if ( engine->screen_copy_diff > 0 ) {

            /* copy screen to buffer */
            DEST( engine->screen_buffer, 0, 0, sdl.screen->w, buffer_height );
            SOURCE( sdl.screen, 0, engine->upper_bar->height + buffer_offset );
            blit_surf();
            /* copy buffer to new pos */
            DEST( sdl.screen, 0, engine->upper_bar->height, sdl.screen->w, buffer_height );
            SOURCE( engine->screen_buffer, 0, 0 );
            blit_surf();
            /* set loop range to redraw lower lines */
            start_map_y += engine->map_height - engine->screen_copy_diff - 2;

        }
        /* going up */
        else {

            /* copy screen to buffer */
            DEST( engine->screen_buffer, 0, 0, sdl.screen->w, buffer_height );
            SOURCE( sdl.screen, 0, engine->upper_bar->height );
            blit_surf();
            /* copy buffer to new pos */
            DEST( sdl.screen, 0, engine->upper_bar->height + buffer_offset, sdl.screen->w, buffer_height );
            SOURCE( engine->screen_buffer, 0, 0 );
            blit_surf();
            /* set loop range to redraw upper lines */
            end_map_y = engine->map_y + abs( engine->screen_copy_diff ) + 1;

        }

    }
    else
        if ( engine->screen_copy_type == HORI_SCREEN_COPY ) {

            /* clear flag */
            engine->screen_copy_type = NO_SCREEN_COPY;

            /* set buffer offset and width */
            buffer_offset = abs( engine->screen_copy_diff ) * engine->x_offset;
            buffer_width = sdl.screen->w - buffer_offset;
            buffer_height = sdl.screen->h - engine->upper_bar->height - engine->lower_bar->height;

            /* going right */
            if ( engine->screen_copy_diff > 0 ) {

                /* copy screen to buffer */
                DEST( engine->screen_buffer, 0, 0, buffer_width, buffer_height );
                SOURCE( sdl.screen, buffer_offset, engine->upper_bar->height );
                blit_surf();
                /* copy buffer to new pos */
                DEST( sdl.screen, 0, engine->upper_bar->height, buffer_width, buffer_height );
                SOURCE( engine->screen_buffer, 0, 0 );
                blit_surf();
                /* set loop range to redraw right lines */
                start_map_x += engine->map_width - engine->screen_copy_diff - 2;

            }
            /* going left */
            else {

                /* copy screen to buffer */
                DEST( engine->screen_buffer, 0, 0, buffer_width, buffer_height );
                SOURCE( sdl.screen, 0, engine->upper_bar->height );
                blit_surf();
                /* copy buffer to new pos */
                DEST( sdl.screen, buffer_offset, engine->upper_bar->height, buffer_width, buffer_height );
                SOURCE( engine->screen_buffer, 0, 0 );
                blit_surf();
                /* set loop range to redraw right lines */
                end_map_x = engine->map_x + abs( engine->screen_copy_diff ) + 1;

            }

        }

    /* start position for drawing */
    x = engine->start_x + ( start_map_x - engine->map_x ) * engine->x_offset;
    y = engine->start_y + ( start_map_y - engine->map_y ) * map.def->tile_height;

    /* end_map_xy must not exceed map's size */
    if ( end_map_x >= map.width ) end_map_x = map.width;
    if ( end_map_y >= map.height ) end_map_y = map.height;

    /* loop to draw map tile */
    for ( j = start_map_y; j < end_map_y; j++ ) {

        for ( i = start_map_x; i < end_map_x; i++ ) {

            /* update each map tile */
            if ( i & 1 )
                abs_y = y + engine->y_offset;
            else
                abs_y = y;
            draw_map_tile( engine, i, j, x, abs_y, flags, NO_REFRESH );

            x += engine->x_offset;

        }
        y += map.def->tile_height;
        x = engine->start_x + ( start_map_x - engine->map_x ) * engine->x_offset;

    }

    /* start position for drawing */
    x = engine->start_x + ( start_map_x - engine->map_x ) * engine->x_offset;
    y = engine->start_y + ( start_map_y - engine->map_y ) * map.def->tile_height;

    /* loop again to draw units */
    for ( j = start_map_y; j < end_map_y; j++ ) {

        for ( i = start_map_x; i < end_map_x; i++ ) {

            /* update each map tile */
            if ( i & 1 )
                abs_y = y + engine->y_offset;
            else
                abs_y = y;
            draw_unit( engine, i, j, x, abs_y, flags, NO_REFRESH );

            x += engine->x_offset;

        }
        y += map.def->tile_height;
        x = engine->start_x + ( start_map_x - engine->map_x ) * engine->x_offset;

    }

    /* reset clip */
    if ( clip == CLIP )
        set_surf_clip( sdl.screen, 0, 0, 0, 0 );

    /* was a full update so full screen needs to be refreshed */
    sdl.rect_count = RECT_LIMIT;

    /* TEST */
/*    printf( "time needed: %i ms\n", get_time() ); */
}

/* update map tile info */
void update_map_tile_info( Engine *engine, int mouse_x, int mouse_y )
{
    int x, y;
    char str[128];
    int ok = 0;

    if ( engine->status == SHOWING_STRAT_MAP )
        ok = get_strat_map_pos( mouse_x, mouse_y, &x, &y );
    else
        ok = get_map_pos( engine, mouse_x, mouse_y, &x, &y );

    if ( ok )
        if ( engine->status == DEPLOYING_UNITS || engine->old_status == DEPLOYING_UNITS ) {
            set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ), "Deploy Unit", REFRESH );
            return;
        }

    if ( ok ) {

        /* display name and coordinates */
        sprintf( str, "%s (%i,%i)", map_tile( x, y )->name, x, y );
        set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ), str, REFRESH );

    }
    else
        if ( engine->last_motion_y >= engine->upper_bar->height && engine->last_motion_y < sdl.screen->h - engine->lower_bar->height ) {

            /* was last position in map? then clear info */
            str[0] = 0;
            set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ), str, REFRESH );

        }
}

/* update unit info */
void update_unit_info( Engine *engine, int mouse_x, int mouse_y)
{
    int x, y;
    char str[128];
    Unit *unit;
    int i;
    int ok = 0;

    if ( engine->status == SHOWING_STRAT_MAP )
        ok = get_strat_map_pos( mouse_x, mouse_y, &x, &y );
    else
        ok = get_map_pos( engine, mouse_x, mouse_y, &x, &y );

    if ( ok && ( !mask_tile( x, y )->fog || engine->cont_status == NOBODY || engine->status == DEPLOYING_UNITS || engine->old_status == DEPLOYING_UNITS ) ) {

        if ( engine->status == DEPLOYING_UNITS || engine->old_status == DEPLOYING_UNITS ) {
            /* use deploy unit*/
            unit = engine->sel_unit;
        }
        else {
            /* get unit pointer */
            if ( engine->sel_unit && engine->sel_unit->x == x && engine->sel_unit->y == y )
                unit = engine->sel_unit;
            else
                if ( engine->air_mode ) {
                    unit = map_tile( x, y )->air_unit;
                    if ( unit == 0 )
                        unit = map_tile( x, y )->unit;
                }
                else {
                    unit = map_tile( x, y )->unit;
                    if ( unit == 0 )
                        unit = map_tile( x, y )->air_unit;
                }
        }

        /* display name and coordinates */
        if ( unit == 0 ) {

            str[0] = 0;
            set_simple_window_text( get_child( engine->lower_bar, ID_STATUS_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->lower_bar, ID_EXP_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->lower_bar, ID_NAME_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->lower_bar, ID_CAP_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->upper_bar, ID_LEFT_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->upper_bar, ID_RIGHT_LABEL ), str, REFRESH );
            set_icon( get_child( engine->lower_bar, ID_MOVE_ICON ), 0, 0, 0, 0, 0, REFRESH );
            set_icon( get_child( engine->lower_bar, ID_ATTACK_ICON ), 0, 0, 0, 0, 0, REFRESH );

        }
        else {

            /* if own unit check if it can move/attack and display icon */
            set_icon( get_child( engine->lower_bar, ID_MOVE_ICON ), 0, 0, 0, 0, 0, REFRESH );
            set_icon( get_child( engine->lower_bar, ID_ATTACK_ICON ), 0, 0, 0, 0, 0, REFRESH );
            if ( unit->player_id == engine->player_id ) {

                if ( unit->cur_prop.mov > 0 )
                    set_icon( get_child( engine->lower_bar, ID_MOVE_ICON ),
                              engine->theme->status_icons, 0, 0,
                              engine->theme->status_icon_width,
                              engine->theme->status_icon_height,
                              REFRESH );
                if ( unit->cur_prop.attack_count > 0 &&
                     unit->embark == NO_EMBARK &&
                     unit->cur_prop.ammo > 0
                )
                    set_icon( get_child( engine->lower_bar, ID_ATTACK_ICON ),
                              engine->theme->status_icons,
                              engine->theme->status_icon_height, 0,
                              engine->theme->status_icon_width,
                              engine->theme->status_icon_height,
                              REFRESH );

            }

            /* own unit? if not no fuel,ammo info */
            if ( unit->player_id != engine->player_id )
                sprintf( str, "|? {? }%i", unit->entr );
            else
                if ( use_fuel( unit ) )
                    sprintf( str, "|%i {%i }%i", unit->cur_prop.ammo, unit->cur_prop.fuel, unit->entr );
                else
                    sprintf( str, "|%i {- }%i", unit->cur_prop.ammo, unit->entr );
            set_simple_window_text( get_child( engine->lower_bar, ID_STATUS_LABEL ), str, REFRESH );
            sprintf( str, "##### '%i%%", ( MAX_DAMAGE - unit->damage ) *10 );
            for ( i = 0; i < unit->exp_level; i++ )
                str[i] = '~';
            set_simple_window_text( get_child( engine->lower_bar, ID_EXP_LABEL ), str, REFRESH );
            if ( unit->cur_prop.attack_count > 0 )
                sprintf( str, "%s", unit->name );
            else
                strcpy( str, unit->name );
            set_simple_window_text( get_child( engine->lower_bar, ID_NAME_LABEL ), str, REFRESH );
            strcpy( str, unit->prop.cap );
            set_simple_window_text( get_child( engine->lower_bar, ID_CAP_LABEL ), str, REFRESH );

            /* if unit is target, add damage prediction */
            if ( !engine->sel_unit ) return;
            for ( i = 0; i < engine->pred_count; i++ ) {

                if ( engine->pred[i].def == get_prim_target( engine, x, y ) ||
                     ( engine->pred[i].def->x == x && engine->pred[i].def->y == y &&
                       engine->sel_unit->x == x && engine->sel_unit->y == y )
                ) {

                    sprintf( str, "Attacker: %i%%", engine->pred[i].att_damage );
                    set_simple_window_text( get_child( engine->upper_bar, ID_LEFT_LABEL ), str, REFRESH );
                    sprintf( str, "Defender: %i%%", engine->pred[i].def_damage );
                    set_simple_window_text( get_child( engine->upper_bar, ID_RIGHT_LABEL ), str, REFRESH );
                    break;

                }
            }

        }

    }
    else
        if ( engine->last_motion_y >= engine->upper_bar->height && engine->last_motion_y < sdl.screen->h - engine->lower_bar->height ) {

            /* was last position in map? then clear info */
            str[0] = 0;
            set_simple_window_text( get_child( engine->lower_bar, ID_STATUS_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->lower_bar, ID_EXP_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->lower_bar, ID_NAME_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->lower_bar, ID_CAP_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->upper_bar, ID_LEFT_LABEL ), str, REFRESH );
            set_simple_window_text( get_child( engine->upper_bar, ID_RIGHT_LABEL ), str, REFRESH );
            set_icon( get_child( engine->lower_bar, ID_MOVE_ICON ), 0, 0, 0, 0, 0, REFRESH );
            set_icon( get_child( engine->lower_bar, ID_ATTACK_ICON ), 0, 0, 0, 0, 0, REFRESH );

        }
}

/* update cursor */
void update_cursor( Engine *engine, int mouse_x, int mouse_y )
{
    int map_x, map_y;

    if ( engine->status == SHOWING_STRAT_MAP ) {
        SDL_SetCursor( engine->stan_cursor );
        return;
    }

    if ( engine->cont_status == NOBODY ) {

        SDL_SetCursor( engine->stan_cursor );
        return;

    }

    if ( get_map_pos( engine, mouse_x, mouse_y, &map_x, &map_y ) ) {

        if ( engine->status == DEPLOYING_UNITS || engine->old_status == DEPLOYING_UNITS ) {
            /* shot deploy cursor if valid on tile */
            /* if no unit is selected and mask::deploy is set show undeploy cursor */
            if ( !engine->sel_unit && mask_tile( map_x, map_y )->deploy )
                SDL_SetCursor( engine->undeploy_cursor );
            else
                if ( engine->sel_unit ) {
                    if ( mask_tile( map_x, map_y )->deploy )
                        SDL_SetCursor( engine->deploy_cursor );
                    else
                        if ( get_prim_deploy_unit( engine, map_x, map_y ) && get_prim_deploy_unit( engine, map_x, map_y )->delay > 0 )
                            SDL_SetCursor( engine->undeploy_cursor );
                         else
                            SDL_SetCursor( engine->stan_cursor );
                }
                else
                    SDL_SetCursor( engine->stan_cursor );
            return;
        }
        /* if trying to merge units do only show merge cursor */
        if ( engine->status == SHOWING_MERGE_UNITS ) {
            if ( mask_tile( map_x, map_y )->merge )
                SDL_SetCursor( engine->merge_cursor );
            else
                SDL_SetCursor( engine->stan_cursor );
            return;
        }

        /* on selectable unit and no unit selected ? */
        if ( !engine->sel_unit &&
             get_prim_unit( engine, map_x, map_y ) &&
             get_prim_unit( engine, map_x, map_y )->player_id == engine->player_id )
        {

            SDL_SetCursor( engine->sel_cursor );
            return;

        }

        /* unit selected and can attack ? */
        if ( engine->sel_unit )
            if ( engine->sel_unit->cur_prop.attack_count > 0 )
                if ( get_prim_target( engine, map_x, map_y ) )
                    if ( !mask_tile( map_x, map_y )->fog ) {
                        SDL_SetCursor( engine->attack_cursor );
                        return;
                    }

        /* unit selected and can move there ? */
        if ( engine->sel_unit && mask_tile( map_x, map_y )->in_range != -1 &&
             !mask_tile( map_x, map_y )->blocked &&
             ( map_x != engine->sel_unit->x || map_y != engine->sel_unit->y ) )
        {

            if ( mask_tile( map_x, map_y )->mount )
                SDL_SetCursor( engine->ground_tran_cursor );
            else
                SDL_SetCursor( engine->move_cursor );
            return;

        }

        /* unit selected and can de/embark there ? */
        if ( engine->sel_unit && mask_tile( map_x, map_y )->sea_embark ) {

            if ( engine->sel_unit->embark == SEA_EMBARK )
                SDL_SetCursor( engine->debark_cursor );
            else
                SDL_SetCursor( engine->embark_cursor );
            return;

        }

    }

    SDL_SetCursor( engine->stan_cursor );
}

/* goto to map position */
void goto_map_pos( Engine *engine, int x, int y, int type )
{
    int x_diff, y_diff;

    /* check range */
    if ( x < 0 ) x = 0;
    if ( y < 0 ) y = 0;
    /* if more tiles are displayed then map has ( black space the rest ) no change in position allowed */
    if ( engine->map_width >= map.width ) x = 0;
    else
        if ( x > map.width - engine->map_width )
            x = map.width - engine->map_width;
    if ( engine->map_height >= map.height ) y = 0;
    else
        if ( y > map.height - engine->map_height )
            y = map.height - engine->map_height;

    /* if SCREEN_COPY is enabled check if this is allowed */
    if ( type == SCREEN_COPY ) {

        x_diff = x - engine->map_x;
        y_diff = y - engine->map_y;

        /* if one diff is ==0 and one diff !=0 do it! */
        if ( x_diff == 0 && y_diff != 0 ) {

            engine->screen_copy_type = VERT_SCREEN_COPY;
            engine->screen_copy_diff = y_diff;

        }
        else
            if ( x_diff != 0 && y_diff == 0 ) {

                engine->screen_copy_type = HORI_SCREEN_COPY;
                engine->screen_copy_diff = x_diff;

            }

    }

    /* set action */
    if ( x != engine->map_x || y != engine->map_y ) {

        engine->map_x = x; engine->map_y = y;
        engine->action = GOTO_POS;

    }
}

/* set action to scroll map and update position and cursor if needed */
void scroll_map_by_mouse( Engine *engine, int x, int y )
{
    int tol = 2; /* tolerance in pixels when to start with scrolling */

    if ( y < tol && engine->map_y > 0 ) {

        SDL_SetCursor( engine->up_cursor );
        goto_map_pos( engine, engine->map_x, engine->map_y - 2, SCREEN_COPY );
        engine->status = SCROLLING_BY_MOUSE;

    }
    else
        if ( y >= sdl.screen->h - tol && engine->map_y < map.height - engine->map_height ) {

            SDL_SetCursor( engine->down_cursor );
            goto_map_pos( engine, engine->map_x, engine->map_y + 2, SCREEN_COPY );
            engine->status = SCROLLING_BY_MOUSE;

        }
        else
            if ( x < tol && engine->map_x > 0 ) {

                SDL_SetCursor( engine->left_cursor );
                goto_map_pos( engine, engine->map_x - 2, engine->map_y, SCREEN_COPY);
                engine->status = SCROLLING_BY_MOUSE;

            }
            else
                if ( x >= sdl.screen->w - tol && engine->map_x < map.width - engine->map_width ) {

                    SDL_SetCursor( engine->right_cursor );
                    goto_map_pos( engine, engine->map_x + 2, engine->map_y, SCREEN_COPY );
                    engine->status = SCROLLING_BY_MOUSE;

                }
                else
                    if ( engine->status == SCROLLING_BY_MOUSE ) {
                        engine->status = engine->old_status;
                        SDL_SetCursor( engine->stan_cursor );
                        engine->old_status = WAITING;
                    }
}

/* set action to scroll map and update position and cursor if needed */
void scroll_map_by_key( Engine *engine, int scroll_dir )
{
    engine->scroll_dir = scroll_dir;

    if ( scroll_dir == 0 ) {
        engine->status = engine->old_status;
        engine->old_status = WAITING;
    }

    switch ( scroll_dir ) {

        case SCROLL_UP:
            goto_map_pos( engine, engine->map_x, engine->map_y - 2, SCREEN_COPY );
            engine->status = SCROLLING_BY_KEY;
            /* update info */
            update_map_tile_info( engine, engine->last_motion_x, engine->last_motion_y );
            update_unit_info( engine, engine->last_motion_x, engine->last_motion_y );
            break;

        case SCROLL_DOWN:
            goto_map_pos( engine, engine->map_x, engine->map_y + 2, SCREEN_COPY );
            engine->status = SCROLLING_BY_KEY;
            /* update info */
            update_map_tile_info( engine, engine->last_motion_x, engine->last_motion_y );
            update_unit_info( engine, engine->last_motion_x, engine->last_motion_y );
            break;

        case SCROLL_RIGHT:
            goto_map_pos( engine, engine->map_x + 2, engine->map_y, SCREEN_COPY );
            engine->status = SCROLLING_BY_KEY;
            /* update info */
            update_map_tile_info( engine, engine->last_motion_x, engine->last_motion_y );
            update_unit_info( engine, engine->last_motion_x, engine->last_motion_y );
            break;

        case SCROLL_LEFT:
            goto_map_pos( engine, engine->map_x - 2, engine->map_y, SCREEN_COPY );
            engine->status = SCROLLING_BY_KEY;
            /* update info */
            update_map_tile_info( engine, engine->last_motion_x, engine->last_motion_y );
            update_unit_info( engine, engine->last_motion_x, engine->last_motion_y );
            break;

    }
}


/* various engine tool functions */

/* set player id, pointer and cont_status */
void set_player( Engine *engine, int player_id )
{
    int human_count;
    int human_id;
    int i;
    Unit *unit;

    /* check range */
    if ( player_id >= scen.player_count ) {
        /* all players have moved, start a new turn */
        player_id = 0;
        scen.rem_turns--;
        scen.cur_turn++;
    }
    if ( player_id < 0 ) player_id = scen.player_count - 1;

    /* get current weather type */
    cur_weather = scen.weather[scen.cur_turn];

    /* set player info */
    engine->player_id = player_id;
    engine->player = scen.players[player_id];
    engine->cont_status = engine->player->cont;

    /* update the list of available reinforcements */
    /* add all units from scen::reinf whose delay <= cur_turn */
    for ( i = 0; i < scen.reinf.count; i++ ) {
        unit = (Unit*)dl_get( &scen.reinf, i );
        if ( unit->player_id == engine->player_id && unit->delay <= scen.cur_turn ) {
            transfer_unit( unit, &scen.reinf, &engine->player->avail_reinf );
            /* index must be reset if unit was added */
            i--;
        }
    }
    check_deploy_button( engine );

    /* set status */
    if ( engine->cont_status == HUMAN )
        engine->status = WAITING;
    else {

        /* initiate cpu turn and set status to THINKING */
        /* ai_init_turn( engine ); *//* init is done after info was shown in end_turn() */
        engine->status = THINKING;

    }

    /* set fog and status */
    if ( engine->cont_status == HUMAN ) {
        engine->fog_status = OWN_FOG;
        set_spot_mask( engine ); /* includes fog */
    }
    else {
        /* if only one human, use his fog and set status to FOREIGNFOG */
        human_count = 0;
        human_id = 0;
        for ( i = 0; i < scen.player_count; i++ )
            if ( scen.players[i]->cont == HUMAN ) {
                if ( human_count == 0 )
                    human_id = i;
                human_count++;
            }
        if ( human_count > 1 ) {
            engine->fog_status = OWN_FOG;
            set_spot_mask( engine ); /* includes fog */
        }
        else {
            engine->fog_status = FOREIGN_FOG;
            set_spot_mask( engine ); /* does not include fog because FOREIGN_FOG is set */
            set_fog( engine, human_id );
        }
    }
    /* set influence mask */
    set_infl_mask( engine );
}

/* init last round; set control to nobody so the players can have a look on the map */
void init_final_view( Engine *engine )
{
    char text[400];

    SDL_SetCursor( engine->stan_cursor );

    /* show message about who won */
    FULL_DEST( sdl.screen );
    fill_surf( 0x0 );
    engine->theme->stan_font->align = ALIGN_X_CENTER | ALIGN_Y_CENTER;
    if ( scen.rem_turns == 0 )
        sprintf( text, "Defender Minor Victory" );
    else {
        if ( scen.cur_turn < scen.major_turn_limit )
            sprintf( text, "Attacker Major Victory" );
        else
            sprintf( text, "Attacker Minor Victory" );
    }
    write_text( engine->theme->stan_font, sdl.screen, sdl.screen->w >> 1, sdl.screen->h >> 1, text, OPAQUE );
    refresh_screen( 0, 0, 0, 0 );
    wait_for_click();

    /* set cont_status to NOBODY */
    clear_mask( F_INVERSE_FOG );
    engine->cont_status = NOBODY;

    /* do not allow save */
    set_active( get_child( engine->menu, ID_SAVE_MENU ), 0, NO_REFRESH );

    /* unselect unit */
    select_unit( engine, 0 );
    engine->check_unit_buttons = 1;

    update_full_screen( engine );

    engine->final_view = 1;
}

/* end turn, display summary and select next player */
void end_turn( Engine *engine )
{
    int i;
    Unit *unit;
    char text[400];

    /* unselect any unit if selected: may cause some strange effects else */
    select_unit( engine, 0 );

    /* if rem_turns == 0 scenario is over and player don't want to look around any longer */
    if ( engine->final_view ) {
        engine->leave_scen = 1;
        return;
    }

    /* init next player */
    set_player( engine, engine->player_id + 1);

    /* prepare units for turn -- fuel, mov-points, entr, weather etc */
    /* delete killed units */
    for ( i = 0; i < scen.units.count; i++ ) {
        unit = (Unit*)dl_get( &scen.units, i );
        if ( unit->player_id == engine->player_id ) {
            if ( scen.cur_turn == 0  )
                prep_unit_for_turn( unit, scen.cur_turn, NO_FUEL_CHECK | NO_ENTR_CHECK );
            else
                prep_unit_for_turn( unit, scen.cur_turn, 0 );
        }
        if ( unit->killed ) {
            remove_unit( engine, unit ); /* die bastard!!! */
            i--; /* adjust index */
        }
    }

    /* clear selected unit */
    engine->sel_unit = 0;
    engine->check_unit_buttons = 1;

    /* if rem_turns == 0 or scen.att_team_won is set
    check and display who won the scenario */
    /* human player may now take a look at the whole map without any fog */
    /* the scenario will end after pressing end_turn again */
    if ( scen.rem_turns == 0 ) {
        init_final_view( engine );
        return;
    }

    /* show information about next player, weather, turns etc */
    draw_turn_info( engine );
    /* if CPU is in control init AI turn here */
    if ( engine->cont_status == CPU )
        ai_init_turn( engine );

    /* restore screen */
    /* if turn is shown display normal map -- if not show_cpu_turn, take black screen with CPU thinks... */
    if ( engine->cont_status != CPU || config.show_cpu_turn ) {

        update_full_screen( engine );

    }
    else {

        FULL_DEST( sdl.screen );
        fill_surf( 0x0 );
        engine->theme->stan_font->align = ALIGN_X_CENTER | ALIGN_Y_CENTER;
        sprintf( text, "CPU thinks..." );
        write_text( engine->theme->stan_font, sdl.screen, sdl.screen->w >> 1, sdl.screen->h >> 1, text, OPAQUE );
        sprintf( text, "( Enable option 'Show Cpu Turn' if you want to see what it is doing. )" );
        write_text( engine->theme->stan_font, sdl.screen, sdl.screen->w >> 1, ( sdl.screen->h >> 1 )+ 20, text, OPAQUE );
        refresh_screen( 0, 0, 0, 0 );

    }
}

/* get unit by map or mouse position */
Unit *get_unit( Engine *engine, int x, int y, int pos_type, int unit_type )
{
    int map_x, map_y;

    switch ( pos_type ) {

        case MOUSE_POS:
            get_map_pos( engine, x, y, &map_x, &map_y );
            x = map_x; y = map_y;
            /* no break; use the MAP_POS stuff */

        case MAP_POS:
            if ( unit_type == GROUND_UNIT )
                return map_tile( x, y )->unit;
            else
                return map_tile( x, y )->air_unit;

    }

    return 0;
}

/* get absolute map position from mouse position */
int get_map_pos( Engine *engine, int mouse_x, int mouse_y, int *map_x, int *map_y )
{
    int x = 0, y = 0;
    int screen_x, screen_y;
    int tile_x, tile_y;
    int total_y_offset;

    /* check if outside of map */
    if ( mouse_y < engine->upper_bar->height || mouse_y >= sdl.screen->h - engine->lower_bar->height )
        return 0;

    /* get the map offset in screen from mouse position */
    x = ( mouse_x - engine->start_x ) / engine->x_offset;
    /* y value computes the same like the x value but their may be an offset of engine::y_offset */
    total_y_offset = 0;
    if ( EVEN( engine->map_x ) && ODD( x ) )
        total_y_offset = engine->y_offset;
    /* if engine::map_x is odd there must be an offset of engine::y_offset for the start of first tile */
    /* and all odd tiles receive an offset of -engine::y_offset so the result is:
        odd: offset = 0 even: offset = engine::y_offset it's best to draw this ;-D
    */
    if ( ODD( engine->map_x ) && EVEN( x ) )
        total_y_offset = engine->y_offset;
    y = ( mouse_y - total_y_offset - engine->start_y ) / map.def->tile_height;

    /* compute screen position */
    get_screen_pos( engine, x + engine->map_x, y + engine->map_y, &screen_x, &screen_y );

    /* test mask with  mouse_x - screen_x, mouse_y - screen_y */
    tile_x = mouse_x - screen_x;
    tile_y = mouse_y - screen_y;

    if ( !engine->tile_mask[tile_y * map.def->tile_width + tile_x] ) {
        if ( EVEN( engine->map_x ) ) {
            if ( tile_y < engine->y_offset && EVEN( x ) ) y--;
            if ( tile_y >= engine->y_offset && ODD( x ) ) y++;
        }
        else {
            if ( tile_y < engine->y_offset && ODD( x ) ) y--;
            if ( tile_y >= engine->y_offset && EVEN( x ) ) y++;
        }
        x--;
    }

    /* add engine map offset and assign */
    x += engine->map_x;
    y += engine->map_y;
    *map_x = x;
    *map_y = y;

    /* check range */
    if ( x >= map.width || y >= map.height ) return 0;

    /* ok, tile exists */
    return 1;
}

/* get screen position from absolute map position */
int get_screen_pos( Engine *engine, int map_x, int map_y, int *screen_x, int *screen_y )
{
    int x = engine->start_x, y = engine->start_y;

    /* this is the starting position if x-pos of first tile on screen is not odd */
    /* if it is odd we must add the y_offset to the starting position */
    if ( ODD( engine->map_x ) )
        y += engine->y_offset;

    /* reduce to visible map tiles */
    map_x -= engine->map_x;
    map_y -= engine->map_y;

    /* check range */
    if ( map_x < 0 || map_y < 0)
        return 0;

    /* compute pos */
    x += map_x * engine->x_offset;
    y += map_y * map.def->tile_height;

    /* if x_pos of first tile is even we must add y_offset to the odd tiles in screen */
    if ( EVEN( engine->map_x ) ) {
        if ( ODD( map_x ) )
            y += engine->y_offset;
    }
    else {
        /* we must substract y_offset from even tiles */
        if ( ODD( map_x ) )
            y -= engine->y_offset;
    }

    /* check range */
    if ( x > sdl.screen->w || y > sdl.screen->h - engine->lower_bar->height )
        return 0;

    /* assign */
    *screen_x = x;
    *screen_y = y;
    return 1;
}

/* select unit by mouse */
int select_unit_by_mouse( Engine *engine, int mouse_x, int mouse_y )
{
    Unit *unit = 0;
    int map_x, map_y;

    /* get map position */
    if ( !get_map_pos( engine, mouse_x, mouse_y, &map_x, &map_y ) ) return 0;

    /* get unit pointer */
    if ( engine->air_mode ) {

        unit = map_tile( map_x, map_y )->air_unit;
        if ( unit == 0 )
            unit = map_tile( map_x, map_y )->unit;

    }
    else {

        unit = map_tile( map_x, map_y )->unit;
        if ( unit == 0 )
            unit = map_tile( map_x, map_y )->air_unit;

    }

    /* no unit found? get back! */
    if ( !unit ) return 0;

    /* select */
    if ( select_unit( engine, unit ) ) {
        /* set action */
        engine->action = SELECT;
        engine->check_unit_buttons = 1;
        return 1;
    }
    return 0;
}

/* initiate movement to position x,y */
int start_unit_move( Engine *engine, int x, int y )
{
    int dest_x, dest_y;

    /* clear ambushing unit */
    engine->ambush_unit = 0; /* if this value is set an ambush occurs at end of movement! */

    /* check if this tile is in range and not the one you are currently on */
    if ( !engine->sel_unit ) return 0 ;
    if ( mask_tile( x, y )->in_range == -1 ) return 0;
    if ( engine->sel_unit->x == x && engine->sel_unit->y == y ) return 0;
    if ( mask_tile( x, y )->blocked ) return 0;

    /* get way points */
    FREE( engine->way );
    engine->way = get_way_points( engine->sel_unit, x, y, &engine->way_length, &engine->ambush_unit );
    /* does a way exist? */
    if ( engine->way == 0 ) return 0;
    /* get_way_points already checked influence; that means x,y may not be the last
    way point so use dest_X, dest_y instead */
    dest_x = engine->way[engine->way_length - 1].x;
    dest_y = engine->way[engine->way_length - 1].y;

    /* assume that undo move is ok; will be reset if an enemy is spotted */
    engine->undo_ok = 1;
    /* backup stuff mask for undo */
    backup_unit( engine->sel_unit, &engine->backup );
    backup_spot_mask( engine );
    /* add last flag_id/player_id of destination to unit backup */
    engine->backup.dest_player_id = map_tile( dest_x, dest_y )->player_id;
    engine->backup.dest_nation_id = map_tile( dest_x, dest_y )->nation_id;
    engine->backup.flag_saved = 1;

    /* if ground transporter needed mount unit */
    if ( mask_tile( x, y )->mount )
        mount_unit( engine->sel_unit );

    /* start at first pos */
    engine->way_pos = 0;
    /* set status and action */
    engine->action = START_MOVE;
    engine->status = MOVING;
    /* action taken */
    engine->sel_unit->no_action_yet = 0;
    /* create background buffer */
    engine->back_buffer = create_surf( engine->sel_unit->sel_prop->width, engine->sel_unit->sel_prop->width, SDL_SWSURFACE );

    /* flying unit? then remaining mov points must be refueled, too */
    if ( engine->sel_unit->sel_prop->flags & FLYING )
        engine->sel_unit->add_fuel_cost = mask_tile( dest_x, dest_y )->in_range;

    /* decrease moving points */
    if ( ( engine->sel_unit->sel_prop->flags & RECON ) && !engine->ambush_unit )
        engine->sel_unit->cur_prop.mov = mask_tile( dest_x, dest_y )->in_range;
    else
        engine->sel_unit->cur_prop.mov = 0;
    if ( engine->sel_unit->cur_prop.mov < 0 )
        engine->sel_unit->cur_prop.mov = 0;

    /* decrease fuel */
    if ( use_fuel( engine->sel_unit ) && config.supply )
        engine->sel_unit->cur_prop.fuel -= engine->way_length - 1;

    /* no entrenchment */
    engine->sel_unit->entr = 0;

    /* update values */
    update_cur_unit_prop( engine->sel_unit );

    engine->check_unit_buttons = 1;

    return 1;
}

/* get primary target; return 0 if there is none */
Unit* get_prim_target( Engine *engine, int x, int y )
{
    Unit *unit = 0;
    /* get primary unit */
    unit = get_prim_unit( engine, x, y );
    if ( unit == 0 ) return 0;
    if ( is_target( engine->sel_unit, unit ) ) return unit;
    /* if this is no valid target check secondary unit */
    if ( unit->sel_prop->flags & FLYING ) {
        unit = map_tile( x, y )->unit;
        if ( is_target( engine->sel_unit, unit ) ) return unit;
    }
    else {
        unit = map_tile( x, y )->air_unit;
        if ( is_target( engine->sel_unit, unit ) ) return unit;
    }
    return 0;
}

/* get primary unit by checking engine->air_mode */
inline Unit* get_prim_unit( Engine *engine, int x, int y )
{
    Map_Tile *tile = map_tile( x, y );
    if ( tile->unit && tile->air_unit ) {

        if ( engine->air_mode )
            return tile->air_unit;
        else
            return tile->unit;

    }
    else
        if ( tile->unit )
            return tile->unit;
        else
            if ( tile->air_unit )
                return tile->air_unit;
            else
                return 0;
}

/* get primary deploy unit by checking engine->air_mode */
inline Unit* get_prim_deploy_unit( Engine *engine, int x, int y )
{
    Map_Tile *tile = map_tile( x, y );
    if ( tile->unit && tile->unit->delay > 0 && tile->air_unit && tile->air_unit->delay > 0 ) {
        if ( engine->air_mode )
            return tile->air_unit;
        else
            return tile->unit;
    }
    else
        if ( tile->unit && tile->unit->delay > 0 )
            return tile->unit;
        else
            if ( tile->air_unit && tile->air_unit->delay > 0 )
                return tile->air_unit;
            else
                return 0;
}

/* deactivate unit buttons */
void clear_unit_buttons( Engine *engine )
{
    get_child( engine->lower_bar, ID_SUPPLY_BUTTON )->active = 0;
    get_child( engine->lower_bar, ID_EMBARK_BUTTON )->active = 0;
    get_child( engine->lower_bar, ID_RENAME_BUTTON )->active = 0;
    get_child( engine->lower_bar, ID_UNDO_BUTTON )->active = 0;
    get_child( engine->lower_bar, ID_MERGE_BUTTON )->active = 0;
}

/* check unit buttons status */
void check_unit_buttons( Engine *engine )
{
    char str[128];

    /* don't even redraw if cpu turn and not show cpu turn */
    if ( engine->cont_status == CPU && !config.show_cpu_turn )
        return;

    get_child( engine->lower_bar, ID_SUPPLY_BUTTON )->ref = 1;
    get_child( engine->lower_bar, ID_EMBARK_BUTTON )->ref = 1;
    get_child( engine->lower_bar, ID_RENAME_BUTTON )->ref = 1;
    get_child( engine->lower_bar, ID_UNDO_BUTTON )->ref = 1;
    get_child( engine->lower_bar, ID_MERGE_BUTTON )->ref = 1;

    if ( !engine->sel_unit ) {

        get_child( engine->lower_bar, ID_SUPPLY_BUTTON )->active = 0;
        get_child( engine->lower_bar, ID_EMBARK_BUTTON )->active = 0;
        get_child( engine->lower_bar, ID_RENAME_BUTTON )->active = 0;
        get_child( engine->lower_bar, ID_UNDO_BUTTON )->active = 0;
        get_child( engine->lower_bar, ID_MERGE_BUTTON )->active = 0;
        return;

    }

    /* only update these buttons if human player is in control */
    if ( engine->cont_status != HUMAN ) return;

    /* supply ? */
    if ( unit_can_supply( engine->sel_unit, ANYTHING ) ) {
        get_child( engine->lower_bar, ID_SUPPLY_BUTTON )->active = 1;
        /* display the supplylevel in brackets, too */
        sprintf( str, "Supply Unit (%i%c)", engine->sel_unit->supply_level, '%' );
        set_simple_window_text( get_child( engine->lower_bar, ID_SUPPLY_BUTTON ), str, NO_REFRESH );
    }
    else
        get_child( engine->lower_bar, ID_SUPPLY_BUTTON )->active = 0;

    /* check if air_embark */
    get_child( engine->lower_bar, ID_EMBARK_BUTTON )->active = 0;
    if ( engine->sel_unit->embark == NO_EMBARK )
        if ( unit_can_embark( engine, engine->sel_unit, engine->sel_unit->x, engine->sel_unit->y, AIR_EMBARK ) )
            get_child( engine->lower_bar, ID_EMBARK_BUTTON )->active = 1;
    if ( engine->sel_unit->embark == AIR_EMBARK )
        if ( unit_can_debark( engine, engine->sel_unit, engine->sel_unit->x, engine->sel_unit->y, AIR_EMBARK ) )
            get_child( engine->lower_bar, ID_EMBARK_BUTTON )->active = 1;

    /* can always rename */
    get_child( engine->lower_bar, ID_RENAME_BUTTON )->active = 1;

    /* undo unit move ? */
    get_child( engine->lower_bar, ID_UNDO_BUTTON )->active = 0;
    if ( engine->undo_ok )
        get_child( engine->lower_bar, ID_UNDO_BUTTON )->active = 1;

    /* may merge with another unit? */
    if ( engine->merge_unit_count > 0 && engine->sel_unit->no_action_yet )
        get_child( engine->lower_bar, ID_MERGE_BUTTON )->active = 1;
    else
        get_child( engine->lower_bar, ID_MERGE_BUTTON )->active = 0;
}

/* while engine->sel_pred->support_count > 0 initiate fight between att and supporter else between
attacker and defender */
int init_next_fight( Engine *engine )
{
    int fight = 0;
    char str[128];

    if ( !engine->sel_pred ) return 0;

    engine->combat_type = NORMAL;

    /* check if there are supporting units; if so initate fight between attacker and these units;
    attacker can'T shoot back and supporter does fire with half its power */
    if ( engine->sel_pred->support_count > 0 ) {
        engine->sel_unit = engine->sel_pred->support[--engine->sel_pred->support_count];
        engine->sel_target = engine->sel_pred->att;
        /* attacker doesn't shoot back */
        engine->sel_target->cur_prop.attack[engine->sel_unit->sel_prop->type] = 0;
        fight = 1;
        /* set message if seen */
        if ( engine->cont_status != CPU || config.show_cpu_turn ) {
            /* defensive fire? */
            if ( engine->sel_unit->sel_prop->flags & ARTILLERY )
                sprintf( str, "Defensive Fire" );
            else
                if ( engine->sel_unit->sel_prop->flags & AIR_DEFENCE )
                    sprintf( str, "Air-Defence" );
                else
                    sprintf( str, "Interceptors" );
            /* display message */
            set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ), str, REFRESH );
        }
        /* is defensive fire */
        engine->combat_type = DEF_FIRE;
    }
    else {
        /* clear info */
        if ( engine->cont_status != CPU || config.show_cpu_turn )
            set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ), "", REFRESH );
        /* normal attack */
        engine->sel_unit = engine->sel_pred->att;
        engine->sel_target = engine->sel_pred->def;
        update_cur_unit_prop( engine->sel_unit );
        update_cur_unit_prop( engine->sel_target );
        /* stop attacks */
        engine->sel_pred = 0;
        /* the attacker is completely suppressed broke up attack */
        if ( engine->sel_unit->damage + engine->sel_unit->sup >= MAX_DAMAGE ) {
            if ( engine->cont_status != CPU || config.show_cpu_turn ) {
                set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ),
                                        "Attack broken up!", REFRESH );
                engine->block_time = 1000;
            }
            /* attack was used anyway */
            engine->sel_unit->cur_prop.attack_count--;
        }
        else
            fight = 1;
    }

    if ( fight ) {
        reset_delay( &engine->cross_delay );
        engine->status = ATTACKING;
        engine->action = SHOW_ATT_CROSS;
        return 1;
    }
    return 0;
}

/* start sel_unit attack */
int start_unit_attack( Engine *engine, int map_x, int map_y )
{
    int i;

    if ( !engine->sel_unit ) return 0;
    if ( engine->sel_unit->cur_prop.ammo <= 0 ) return 0;
    if ( !engine->ambush && engine->sel_unit->cur_prop.attack_count <= 0 ) return 0;

    /* get target from pred list */
    for ( i = 0; i < engine->pred_count; i++ ) {
        if ( get_prim_target( engine, map_x, map_y ) == engine->pred[i].def )
            break;
        if ( engine->pred[i].def->x == map_x && engine->pred[i].def->y == map_y )
            if ( engine->sel_unit->x == map_x && engine->sel_unit->y == map_y )
                break;
    }
    if ( i == engine->pred_count ) return 0;
    engine->sel_pred = &engine->pred[i];

    /* gets and initiates next fight either between att, def or att and units giving supporting fire */
    init_next_fight( engine );
    /* can't undo turn now */
    if ( engine->undo_ok ) {
        engine->check_unit_buttons = 1;
        deny_undo( engine );
    }
    return 1;
}


/* ambush selected unit with unit */
int start_unit_ambush( Engine *engine )
{
    /* ambush possible? */
    if ( !can_attack_unit( engine->ambush_unit, engine->sel_unit, DEFEND ) ) return 0;
    if ( scen.players[engine->ambush_unit->player_id]->dipl[engine->sel_unit->player_id] != ENEMIES )
        return 0;

    /* show ambushing unit */
    update_full_map( engine, CLIP );

    /* declare as ambush */
    engine->ambush = 1;

    /* set rugged defence flag */
    engine->ambush_unit->rugged_def = 1;
    /* select as target */
    engine->sel_target = engine->ambush_unit;

    /* start attack */
    reset_delay( &engine->cross_delay );
    engine->status = ATTACKING;
    engine->action = SHOW_ATT_CROSS;

    /* no movement */
    engine->sel_unit->cur_prop.mov = 0;
    /* no attacks */
    engine->sel_unit->cur_prop.attack_count = 0;

    /* is ambush */
    engine->combat_type = AMBUSH;

    return 1;
}

/* remove unit completely */
void remove_unit( Engine *engine, Unit *unit )
{
    /* check if it's an enemy to the current player; if so the influence must be removed */
    if ( engine->player->dipl[unit->player_id] == ENEMIES ) {
        remove_infl( unit );
        remove_vis_infl( unit );
    }

    /* from map */
    if ( unit->sel_prop->flags & FLYING )
        map_tile(unit->x,unit->y)->air_unit = 0;
    else
        map_tile(unit->x,unit->y)->unit = 0;

    /* from unit list */
    dl_delete_poi( &scen.units, unit );
}

/* confirm that ground transporter should be abandoned */
int aban_ground_tran( Engine *engine, Unit *unit )
{
    if ( unit->tran_prop_set ) {

        /* run confirm window, if canceled, cancel embarkment */
        set_simple_window_text( get_child( engine->conf_window, ID_UPPER_LABEL ), "Do you really want to",
                     NO_REFRESH );
        set_simple_window_text( get_child( engine->conf_window, ID_LOWER_LABEL ), "abandon ground transporter?",
                     NO_REFRESH );
        if ( run_stan_window( engine->conf_window ) == ID_CANCEL )
            return 0;

    }

    return 1;
}

/* draw pre-turn info: turn count, weather, current player */
void draw_turn_info( Engine *engine )
{
    int text_x, text_y;
    char text[400];

    FULL_DEST( sdl.screen );
    fill_surf( 0x0 );
    refresh_screen( 0, 0, 0, 0 );

    text_x = sdl.screen->w >> 1;
    text_y = ( sdl.screen->h - 3 * engine->theme->stan_font->height ) >> 1;
    engine->theme->stan_font->align = ALIGN_X_CENTER | ALIGN_Y_TOP;
    sprintf( text, "Next Player: %s", engine->player->name );
    write_text( engine->theme->stan_font, sdl.screen, text_x, text_y, text, OPAQUE );
    if ( scen.rem_turns > 1 )
        sprintf( text, "Remaining Turns: %i", scen.rem_turns );
    else
        sprintf( text, "Remaining Turns: Last Turn" );
    write_text( engine->theme->stan_font, sdl.screen, text_x, text_y + engine->theme->stan_font->height, text, OPAQUE );
    sprintf( text, "Weather: %s", map.def->weather_names[scen.weather[scen.cur_turn]] );
    write_text( engine->theme->stan_font, sdl.screen, text_x, text_y + 2 * engine->theme->stan_font->height, text, OPAQUE );
    refresh_screen( 0, 0, 0, 0 );
    wait_for_click();
}

/* update map + upper/lower bar */
void update_full_screen( Engine *engine )
{
    set_simple_window_text( get_child( engine->upper_bar, ID_MIDDLE_LABEL ), "", NO_REFRESH );
    open_window( engine->upper_bar, REFRESH  );
    open_window( engine->lower_bar, REFRESH  );
    update_full_map( engine, CLIP );
    refresh_rects();
}

/*
reini_engine() must be run after loading a scenario; it updates all variables
according to engine::player_id and draws a turn startup team */
void reinit_engine( Engine *engine )
{
    /* update status */
    engine->status = WAITING;
    engine->action = NONE;
    engine->player = scen.players[engine->player_id];
    engine->cont_status = engine->player->cont;
    SDL_SetCursor( engine->stan_cursor );
    engine->window_id = 0;
    engine->scen_loaded = 0;
    engine->sel_unit = 0;
    engine->cpu_actions_count = 0;

    /* update masks */
    set_spot_mask( engine );
    /* show turn info */
    draw_turn_info( engine );
    /* show screen */
    update_full_screen( engine );
}

/* reload scenario and restart */
void restart_scen( Engine *engine )
{
    char *scen_name;
    int *player_cont = 0;
    int i;

    /* save scenario name */
    scen_name = strdup( scen.file_name );

    /* save controls */
    player_cont = calloc( scen.player_count, sizeof( int ) );
    for ( i = 0; i < scen.player_count; i++ )
        player_cont[i] = scen.players[i]->cont;

    /* reinit engine */
    clear_engine( engine );
    init_engine( engine, scen_name, player_cont );

    /* set first player */
    engine->player_id = -1;
    end_turn( engine );

    engine->restart_scen = 0;
    FREE( scen_name );
}

/* update and run briefing window */
void run_brief_window( Engine *engine, char *text_str )
{
    set_window_text( get_child( engine->brief_window, 101 ),
                      create_text( text_str, get_child( engine->brief_window, 101 )->char_limit ),
                      NO_REFRESH );
    run_stan_window( engine->brief_window );
}

/*
====================================================================
This function handles the AI turn. It calls ai_get_action() or
restores a stored action and initate the engine to execute this
action.
====================================================================
*/
void ai_handle( Engine *engine )
{
    AI_Action ai_action;

    /* ready for next action? */
    if ( engine->cont_status != CPU ) {
        fprintf( stderr, "ai_handle: error: human is in control\n" );
        return;
    }
    if ( engine->status != THINKING ) {
        fprintf( stderr, "ai_handle: error: engine's still busy with executing last action\n" );
        return;
    }
    /* single-actions are executed immediately; in case of double-action (MOVE_ATTACK ...) */
    /* the second action is stored */
    if ( !ai_get_stored_action( &ai_action ) )
        ai_action = ai_get_action( engine );
    /* check and initate action */
    switch ( ai_action.type ) {
        /* no action */
        case AI_NO_ACTION:
#ifdef AI_DEBUG
            printf( "%s: no_action\n", ai_action.unit->name );
#endif
            break;
        /* end ai turn */
        case AI_END_TURN:
            ai_end_turn( engine );
            end_turn( engine );
            break;
        /* attack! */
        case AI_ATTACK:
#ifdef AI_DEBUG
            printf( "%s: attacks %s\n", ai_action.unit->name, ai_action.target->name );
#endif
            select_unit( engine, ai_action.unit );
            if ( !start_unit_attack( engine, ai_action.target->x, ai_action.target->y ) )
                fprintf( stderr, "ai_handle: warning: cannot execute attack of unit %s on %i,%i: "
                                 "start_unit_attack(...) failed\n",
                                 ai_action.unit->name,
                                 ai_action.target->x, ai_action.target->y );
            break;
        case AI_MOVE_ATTACK:
        case AI_MOVE:
#ifdef AI_DEBUG
            printf( "%s: moves to %i,%i\n", ai_action.unit->name, ai_action.x, ai_action.y );
#endif
            select_unit( engine, ai_action.unit );
            if ( !start_unit_move( engine, ai_action.x, ai_action.y ) )
                fprintf( stderr, "ai_handle: warning: cannot move unit %s to %i,%i: "
                                 "start_unit_move(...) failed\n",
                                 ai_action.unit->name, ai_action.x, ai_action.y );
            /* check if there is an additional action */
            if ( ai_action.type == AI_MOVE_ATTACK ) {
                ai_store_action( ai_attack_action( ai_action.target ) );
#ifdef AI_DEBUG
                printf( "%s: stores attack on %i,%i\n", ai_action.unit->name,
                        ai_action.target->x, ai_action.target->y );
#endif
                break;
            }
            break;

        case AI_SUPPLY:
            supply_unit( ai_action.unit );
#ifdef AI_DEBUG
            printf( "%s: supplies\n", ai_action.unit->name );
#endif
            break;

        default:
            fprintf( stderr, "ai_handle: error: unknown AI action: %i\n", ai_action.type );
            break;

    }
}

/*
====================================================================
Apply nescessary changes when setting resolution.
====================================================================
*/
void apply_video_mode( Engine *engine )
{
    /* set new video mode saved in config */
    set_video_mode( config.video_mode );
    /* all windows must be recreated */
    delete_windows( engine );
    create_windows( engine );
    /* reset engine's map size (number of tiles on screen) */
    engine->map_width = ( sdl.screen->w + map.def->tile_x_offset - ( map.def->tile_width - engine->x_offset ) ) / engine->x_offset + 2;
    engine->map_height = ( sdl.screen->h + map.def->tile_y_offset - engine->lower_bar->height - engine->upper_bar->height ) / map.def->tile_height + 1;
    /* reset map pos if nescessary */
    if ( engine->map_x + engine->map_width >= map.width )
        engine->map_x = map.width - engine->map_width;
    if ( engine->map_y + engine->map_height >= map.height )
        engine->map_y = map.height - engine->map_height;
    if ( engine->map_x < 0 ) engine->map_x = 0;
    if ( engine->map_y < 0 ) engine->map_y = 0;
    /* recreate strategic map */
    strat_map_clear();
    strat_map_init( engine );
    /* recreate screen buffer */
    SDL_FreeSurface( engine->screen_buffer );
    engine->screen_buffer = create_surf( sdl.screen->w, sdl.screen->h, SDL_SWSURFACE );
    /* redraw map */
    if ( engine->status == SHOWING_STRAT_MAP ) {
        open_window( engine->upper_bar, REFRESH );
        open_window( engine->lower_bar, REFRESH );
        strat_map_update( engine, RECREATE_STRAT_MAP | RECREATE_UNIT_LAYER | RECREATE_FLAG_LAYER );
        refresh_screen( 0, 0, 0, 0 );
    }
    else
        update_full_screen( engine );
    engine->check_unit_buttons = 1;
}

/*
====================================================================
Activate button if reinfs left else deactivate.
====================================================================
*/
void check_deploy_button( Engine *engine )
{
    if ( engine->player->cont == HUMAN ) {
        /* check the status of the deploy button */
        get_child( engine->lower_bar, ID_DEPLOY_BUTTON )->ref = 1;
        if ( engine->player->avail_reinf.count > 0 )
            get_child( engine->lower_bar, ID_DEPLOY_BUTTON )->active = 1;
        else
            get_child( engine->lower_bar, ID_DEPLOY_BUTTON )->active = 0;
    }
}

/*
====================================================================
Run confirm window and return true if accepted else 0.
====================================================================
*/
int run_conf_window( Engine *engine, char *str1, char *str2 )
{
    if ( !str1 )
        set_simple_window_text( get_child( engine->conf_window, ID_UPPER_LABEL ), "",
                     NO_REFRESH );
    else
        set_simple_window_text( get_child( engine->conf_window, ID_UPPER_LABEL ), str1,
                     NO_REFRESH );
    if ( !str2 )
        set_simple_window_text( get_child( engine->conf_window, ID_LOWER_LABEL ), "",
                     NO_REFRESH );
    else
        set_simple_window_text( get_child( engine->conf_window, ID_LOWER_LABEL ), str2,
                     NO_REFRESH );
    return ( run_stan_window( engine->conf_window ) == ID_ACCEPT );
}
