/*
 * apply filter file to all files in a newsgroup
 *
 * Written by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
 * Copyright 1999.
 *
 * See file COPYING for restrictions on the use of this software.
 */
 
#include "leafnode.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <ctype.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <utime.h>

#define MAXHEADERSIZE 5000

int debug = 0;
int verbose;
int first, last;

int main( int argc, char * argv[] ) {
    char c[4] = "-\\|/" ; 
    int i, score, option, deleted, kept;
    unsigned int n;
    char *k, *l, *msgid;
    const char *msgidpath = "";
    FILE *f;
    DIR * d;
    struct dirent * de;
    struct stat st;
    struct utimbuf u;
    struct newsgroup * g;

    if ( !initvars( argv[0] ) )
	exit(1);

    if ( ( argc < 2 ) || ( argc > 3 ) ) {
	printf("Usage: %s newsgroups\n", argv[0] );
	exit(1);
    }

    while ( ( option = getopt( argc, argv, "v" ) ) != -1 ) {
	if ( option == 'v' )
	    verbose++;
	else {
	    printf("Usage: %s newsgroups\n", argv[0] );
	    exit( 1 );
	}
    }

#ifdef HAVE_OLD_SYSLOG
    openlog( "applyfilter", LOG_PID );
#else
    openlog( "applyfilter", LOG_PID|LOG_CONS, LOG_NEWS );
#endif

    if ( !readconfig() ) {
	printf( "Reading configuration failed, exiting "
		"(see syslog for more information).\n");
	exit( 2 );
    }
    if ( filterfile )
	readfilter( filterfile );
    else {
	printf("Nothing to filter -- no filterfile found.\n");
	exit(1);
    }
    if ( lockfile_exists() )
	exit( 1 );
    readactive();

    g = findgroup( argv[optind] );
    if ( !g ) {
	printf("Newsgroups %s not found in active file.\n", argv[optind] );
	unlink( lockfile );
	exit(1);
    }

    g->first = INT_MAX;
    g->last = 0;
    if ( !chdirgroup( g->name, FALSE )) {
	printf("No such newsgroup: %s\n", g->name );
	unlink( lockfile );
	exit(1);
    }
    if ( !(d = opendir( "." ) )) {
	printf("Unable to open directory for newsgroup %s\n", g->name );
	unlink( lockfile );
	exit(1);
    }

    i = 0;
    deleted = 0;
    kept = 0;
    l = critmalloc( MAXHEADERSIZE+1, "Space for article" );
    while ( ( de = readdir( d ) ) != NULL ) {
	if ( !isdigit((unsigned char) de->d_name[0] ) ) {
	    /* no need to stat file */
	    continue;
	}
	switch ( verbose ) {
	    case 1: {
		printf( "%c", c[i%4] );
		fflush( stdout );
		i++;
		break;
	    }
	    case 2: {
		printf( "%s\n", de->d_name );
	    }
	}
	stat( de->d_name, &st );
	if ( S_ISREG(st.st_mode) && ( f = fopen( de->d_name, "r" ) ) != NULL ) {
	    fread( l, sizeof(char), MAXHEADERSIZE, f );
	    if ( ferror(f) ) {
		printf("error reading %s\n", de->d_name );
		clearerr(f);
	    }
	    msgid = fgetheader( f, "Message-ID:" );
	    fclose( f );
	    if ( ( k = strstr( l, "\n\n" ) ) != NULL ) {
		*k = '\0';	/* cut off body */
		score = dofilter( l );
	    } else {
		/* article has no body - delete it */
		score = -1;
	    }
	    if ( score ) {
		unlink( de->d_name );
		/* delete stuff in message.id directory as well */
		if ( msgid ) {
		    msgidpath = lookup( msgid );
		    if ( ( stat( msgidpath, &st ) == 0 ) &&
			 ( st.st_nlink < 2 ) ) {
			if ( unlink( msgidpath ) == 0 )
			    deleted++;
		    }
		}
		if ( verbose )
		    printf( "%s %s deleted\n", de->d_name, msgidpath );
	    }
	    else {
		n = strtoul( de->d_name, NULL, 10 );
		if ( n ) {
		    if ( n < g->first )
			g->first = n;
		    if ( n > g->last )
			g->last = n;
		}
		u.actime = st.st_atime;
		u.modtime = st.st_mtime;
		utime( de->d_name, &u );
		kept++;
	    }
	} else
	    printf( "could not open %s\n", de->d_name );
    }
    closedir( d );
    free( l );
    if ( g->first <= g->last )
	writeactive();
    unlink( lockfile );
    printf( "%d articles deleted, %d kept.\n", deleted, kept );

#ifdef NOTYET
    if ( verbose )
	printf( "Updating .overview file\n" );
    fixxovergrp( g->name );
#endif
    exit(0);
}
