/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/JRV
 *
 *	$Id: reducescatter.c,v 6.4.2.1 2002/10/09 19:49:12 brbarret Exp $
 *
 *	Function:	- reduce and scatter data to all processes
 *	Accepts:	- send buffer
 *			- receive buffer
 *			- counts of elements to recv
 *			- type of elements
 *			- operation to perform
 *			- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>

#include <lam_config.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <terror.h>

/*@

MPI_Reduce_scatter - Combines values and scatters the results

Input Parameters:
+ sbuf - starting address of send buffer (choice) 
. rcounts - integer array specifying the 
number of elements in result distributed to each process.
Array must be identical on all calling processes. 
. dtype - data type of elements of input buffer (handle) 
. op - operation (handle) 
- comm - communicator (handle) 

Output Parameter:
. rbuf - starting address of receive buffer (choice) 

.N IMPI

.N fortran

.N collops

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_IMPI
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_BUFFER
.N MPI_ERR_OP
.N MPI_ERR_BUFFER
.N MPI_ERR_BUFFER_ALIAS

.N ACK
@*/
int MPI_Reduce_scatter(void *sbuf, void *rbuf, int *rcounts, 
		       MPI_Datatype dtype, MPI_Op op, MPI_Comm comm)
{
	int		i;
	int		size;			/* group size */
	int		rank;			/* my rank */
	int		count;			/* total count */
	int		err;			/* error code */
	int		*disps = 0;		/* displacement array */
	char		*buffer = 0;		/* reduce buffer */
	char		*origin = 0;		/* data origin in buffer */

	lam_initerr();
	lam_setfunc(BLKMPIREDUCESCATTER);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return (lam_errfunc(comm, BLKMPIREDUCESCATTER,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (dtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(comm, BLKMPIREDUCESCATTER,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (op == MPI_OP_NULL) {
		return(lam_errfunc(comm, BLKMPIREDUCESCATTER,
				lam_mkerr(MPI_ERR_OP, 0)));
	}

	if (rcounts == 0) {
		return (lam_errfunc(comm, BLKMPIREDUCESCATTER,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

#if LAM_WANT_IMPI

	/* Remove this when IMPI collectives are implemented */

        if (LAM_IS_IMPI(comm)) {
	  return lam_err_comm(comm, MPI_ERR_COMM, 0, 
			      "Collectives not yet implemented on IMPI communicators");
	}
#endif 


	LAM_TRACE(lam_tr_cffstart(BLKMPIREDUCESCATTER));
/*
 * Initialize reduce & scatterv info at the root (rank 0).
 */
	MPI_Comm_size(comm, &size);
	MPI_Comm_rank(comm, &rank);

	for (i = 0, count = 0; i < size; ++i) {
		if (rcounts[i] < 0) {
			return (lam_errfunc(comm, BLKMPIREDUCESCATTER,
						lam_mkerr(MPI_ERR_COUNT, 0)));
		}
		count += rcounts[i];
	}

	if (rank == 0) {
		disps = (int *) malloc((unsigned) size * sizeof(int));
		if (disps == 0) {
			free((char *) disps);
			return (lam_errfunc(comm, BLKMPIREDUCESCATTER,
					lam_mkerr(MPI_ERR_OTHER, errno)));
		}
		
		err = lam_dtbuffer(dtype, count, &buffer, &origin);
		if (err != MPI_SUCCESS) {
			free((char *) disps);
			return (lam_errfunc(comm, BLKMPIREDUCESCATTER, err));
		}

		disps[0] = 0;
		for (i = 0; i < (size - 1); ++i) {
			disps[i + 1] = disps[i] + rcounts[i];
		}
	}
/*
 * reduction
 */
	err = MPI_Reduce(sbuf, origin, count, dtype, op, 0, comm);
	if (err != MPI_SUCCESS) {
		if (disps) free((char *) disps);
		if (buffer) free(buffer);
		return (lam_errfunc(comm, BLKMPIREDUCESCATTER, err));
	}
/*
 * scatter
 */
	err = MPI_Scatterv(origin, rcounts, disps, dtype,
				rbuf, rcounts[rank], dtype, 0, comm);

	if (disps) free((char *) disps);
	if (buffer) free(buffer);

	if (err != MPI_SUCCESS) {
		return (lam_errfunc(comm, BLKMPIREDUCESCATTER, err));
	}

	LAM_TRACE(lam_tr_cffend(BLKMPIREDUCESCATTER,
					-1, comm, dtype, rcounts[rank]));

	lam_resetfunc(BLKMPIREDUCESCATTER);
	return(MPI_SUCCESS);
}
