/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN/RBD
 *
 *	$Id: lamupdown.c,v 6.12.2.1 2002/10/09 19:49:11 brbarret Exp $
 *
 *	Function:	- take top level info down
 *			- take error info up
 */

#include <stdlib.h>

#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#include <blktype.h>
#include <lam_config.h>
#if LAM_HAVE_THREADS
#include <lamthreads.h>
#endif

/*
 * global variables
 */
int			lam_topfunc = 0;
int			lam_toproot;
int			lam_toprootgps;

/*
 * private data
 */
static int              *funcs = 0;
static int              num_funcs = 0; /* points to next available slot */
static int              max_funcs = 0;
static int              inc_funcs = 1;

/*
 *	lam_setfunc
 *
 *	Function:	- set top level function
 *	Accepts:	- local function type
 */
void
lam_setfunc(locfunc)

int			locfunc;

{
#if LAM_HAVE_THREADS
/*
 * There are several places where MPI functions call each other -- we
 * want to allow this.  But we can't let other threads in until this
 * one finishes.  The lam_mpi_mutex will be released in
 * lam_reset_func().
 */
  if (lam_thread_level > MPI_THREAD_SINGLE) {
    if (!lam_mutex_is_owner(&lam_mpi_mutex)) {
      lam_mutex_lock(&lam_mpi_mutex);
    }
    lam_mutex_lock(&lam_updown_mutex);
  }
#endif

  /* Grow the stack if necessary */

  if (max_funcs <= num_funcs) {
    if (max_funcs == 0)
      funcs = malloc(sizeof(int) * inc_funcs);
    else
      funcs = realloc(funcs, sizeof(int) * (max_funcs + inc_funcs));

    max_funcs += inc_funcs;
  }

  /* Save the function in the next available slot */

  lam_topfunc = locfunc;
  funcs[num_funcs] = locfunc;
  num_funcs++;

#if LAM_HAVE_THREADS
  if (lam_thread_level > MPI_THREAD_SINGLE) {
    lam_mutex_unlock(&lam_updown_mutex);
  }
#endif
}

/*
 *	lam_resetfunc
 *
 *	Function:	- reset function at top level
 *	Accepts:	- local function type
 */
void
lam_resetfunc(locfunc)

int			locfunc;

{
  struct _fyiproc	*p;			/* favourite pointer */

#if LAM_HAVE_THREADS
  if (lam_thread_level > MPI_THREAD_SINGLE) {
    lam_mutex_lock(&lam_updown_mutex);
  }
#endif

  p = (struct _fyiproc *) _kio.ki_fyi;
  
  if (num_funcs > 0) {
    if (locfunc == -1 || funcs[num_funcs - 1] == locfunc) {
      num_funcs--;

      if (num_funcs > 0)
	lam_topfunc = funcs[num_funcs - 1];
      else
	lam_topfunc = p->fyp_func = 0;
    }
  }
#if LAM_HAVE_THREADS
/*
 * There are several places where MPI functions call each other -- we
 * want to allow this.  But we can't let other threads in until this
 * one finishes.  lam_mpi_mutex was locked in lam_setfunc().
 *
 * Special case: if we're resetting and we're in BLKMPIINITTHREAD and
 * the mutex owner is -1, don't try to unlock it.  See MPI_Init_thread
 * for the reason why.
 */
  if (lam_thread_level > MPI_THREAD_SINGLE) {
    if (lam_mutex_is_owner(&lam_mpi_mutex) && num_funcs == 0) {
      lam_mutex_unlock(&lam_mpi_mutex);
    }
    lam_mutex_unlock(&lam_updown_mutex);
  }
#endif
}

/*
 *	lam_numfuncs
 *
 *	Function:	- get the size of the stack
 *	Returns:	- size of the stack
 */
int
lam_numfuncs()
{
#if LAM_HAVE_THREADS
  int ret;

  if (lam_thread_level > MPI_THREAD_SINGLE) 
    lam_mutex_lock(&lam_updown_mutex);

  ret = num_funcs;

  if (lam_thread_level > MPI_THREAD_SINGLE) 
    lam_mutex_unlock(&lam_updown_mutex);

  return ret;
#else
  return num_funcs;
#endif
}


/*
 *	lam_getfunc
 *
 *	Function:	- get top level function
 *	Returns:	- top level function
 */
int
lam_getfunc()

{
  int ret;

#if LAM_HAVE_THREADS
  if (lam_thread_level > MPI_THREAD_SINGLE) 
    lam_mutex_lock(&lam_updown_mutex);
#endif

  if (num_funcs > 0)
    ret = funcs[num_funcs - 1];
  else
    ret = 0;

#if LAM_HAVE_THREADS
  if (lam_thread_level > MPI_THREAD_SINGLE) 
    lam_mutex_unlock(&lam_updown_mutex);
#endif

  return ret;
}

/*
 *	lam_setparam
 *
 *	Function:	- set top level parameters
 *	Accepts:	- local function type
 *			- root global/local rank (collective comm.)
 *			- root node/index GPS (collective comm.)
 */
void
lam_setparam(locfunc, root, rootgps)

int			locfunc;
int			root;
int			rootgps;

{
#if LAM_HAVE_THREADS
	if (lam_thread_level > MPI_THREAD_SINGLE) 
	  lam_mutex_lock(&lam_updown_mutex);
#endif

	if (lam_topfunc == locfunc) {
		lam_toproot = root;
		lam_toprootgps = rootgps;
	}

#if LAM_HAVE_THREADS
	if (lam_thread_level > MPI_THREAD_SINGLE) 
	  lam_mutex_unlock(&lam_updown_mutex);
#endif
}

/*
 *	lam_getparam
 *
 *	Function:	- get top level parameters
 *	Accepts:	- ptr root ranks
 *			- ptr root GPS
 */
void
lam_getparam(proot, prootgps)

int			*proot;
int			*prootgps;

{
#if LAM_HAVE_THREADS
	if (lam_thread_level > MPI_THREAD_SINGLE) 
	  lam_mutex_lock(&lam_updown_mutex);
#endif

	*proot = lam_toproot;
	*prootgps = lam_toprootgps;

#if LAM_HAVE_THREADS
	if (lam_thread_level > MPI_THREAD_SINGLE) 
	  lam_mutex_unlock(&lam_updown_mutex);
#endif
}

/*
 *	lam_mkerr
 *
 *	Function:	- form an error code
 *	Accepts:	- error class
 *			- error value
 *	Returns:	- error code
 */
int
lam_mkerr(class, error)

int			class;
int			error;

{
	int		errcode;

#if LAM_HAVE_THREADS
	if (lam_thread_level > MPI_THREAD_SINGLE) 
	  lam_mutex_lock(&lam_updown_mutex);
#endif

	errcode = ((error & 0xFFFF) << 8) | (lam_topfunc & 0xFF);
	errcode = (errcode << 8) | (class & 0xFF);

#if LAM_HAVE_THREADS
	if (lam_thread_level > MPI_THREAD_SINGLE) 
	  lam_mutex_unlock(&lam_updown_mutex);
#endif

	return(errcode);
}

/*
 *	lam_bkerr
 *
 *	Function:	- break error code into components
 *	Accepts:	- error code
 *			- ptr class (returned value)
 *			- ptr function (returned value)
 *			- ptr error (returned value)
 */
void
lam_bkerr(errcode, class, func, error)

int			errcode;
int			*class;
int			*func;
int			*error;

{
	*class = errcode & 0xFF;
	errcode >>= 8;
	*func = errcode & 0xFF;
	errcode >>= 8;
	*error = errcode & 0xFFFF;
}

/*
 *	lam_errfunc
 *
 *	Function:	- handle MPI errors according to error mode
 *			- pass error up to top level
 *			- call error handler at top level
 *	Accepts:	- communicator
 *			- local function type
 *			- error code
 *	Returns:	- error code
 */
int
lam_errfunc(errcomm, locfunc, errcode)

MPI_Comm		errcomm;
int			locfunc;
int			errcode;

{
	MPI_Comm	comm;			/* communicator */
	int		class;			/* error class */
	int		func;			/* function type */
	int		error;			/* errno value */
/*
 * This catches some cases where errors are returned in F77 wrappers
 * before the call is made to the C version of the MPI function.  
 */
	lam_initerr_m();

	lam_bkerr(errcode, &class, &func, &error);

	if (func == locfunc) {

		comm = (errcomm) ? errcomm : MPI_COMM_WORLD;

		if (comm->c_window) {
			return(lam_err_win(comm->c_window, class, error, ""));
		} else {
			return(lam_err_comm(comm, class, error, ""));
		}
	}

	return(errcode);
}

/*
 *	lam_printfunc
 *
 *	Function:	- print out the call stack without destroying it
 */
void
lam_printfunc()
{
  int i, myrank;

  if (num_funcs > 0) {
    myrank = lam_myproc->p_gps.gps_grank;
    printf("Rank (%d, MPI_COMM_WORLD): Call stack within LAM:\n", myrank);
    for (i = --num_funcs; i >= 0; i--)
      printf("Rank (%d, MPI_COMM_WORLD):  - %s()\n", myrank, 
	     blktype(funcs[i]));
    printf("Rank (%d, MPI_COMM_WORLD):  - main()\n", myrank);
  }
}


/*
 *	lam_nukefunc
 *
 *	Function:	- reset top function (cleanup)
 */
void
lam_nukefunc()

{
#if LAM_HAVE_THREADS
  if (lam_thread_level > MPI_THREAD_SINGLE) 
    lam_mutex_lock(&lam_updown_mutex);
#endif

  lam_topfunc = 0;
  if (max_funcs > 0) {
    max_funcs = 0;
    num_funcs = 0;
    free(funcs);
    funcs = (int*) 0;
  }

  lam_topfunc = 0;
  ((struct _fyiproc *) _kio.ki_fyi)->fyp_func = 0;

#if LAM_HAVE_THREADS
  if (lam_thread_level > MPI_THREAD_SINGLE) 
    lam_mutex_unlock(&lam_updown_mutex);
#endif
}

/*
 *	lam_err_comm
 *
 *	Function:	- handle MPI error on a communicator
 *	Accepts:	- communicator
 *			- error class
 *			- errno value
 *			- error message
 *	Returns:	- error class
 */
int
lam_err_comm(comm, errclass, error, errmsg)

MPI_Comm		comm;
int			errclass;
int			error;
char			*errmsg;

{
/*
 * Catch the uninitialized case when an error is returned in an F77
 * wrapper before the call is made to the C version of the function.  
 */
	lam_initerr_m();
/*
 * If the top function is zero then the error has already been handled.
 */
	if (lam_topfunc == 0) {
	    return(errclass);
	}

	if (comm == MPI_COMM_NULL) {
		comm = MPI_COMM_WORLD;
	}
/*
 * Compatibility with lam_errfunc() style error handling.  Check if the
 * errorclass has encoded information and if so break it down.
 */
	if (errclass & 0xFFFFFF00) {
		error = (errclass >> 16) & 0xFFFF;
		errclass &= 0xFF;
	}
/*
 * If this is a window communicator invoke the window error handler.
 */
	if (comm->c_window) {
		return(lam_err_win(comm->c_window, errclass, error, errmsg));
	}
/*
 * Invoke communicator error handler taking care with language calling
 * convention.
 */
	if (comm->c_errhdl->eh_flags & LAM_LANGF77) {
		(comm->c_errhdl->eh_func)(&comm->c_f77handle,
						&errclass, error, errmsg);
	} else if (comm->c_errhdl->eh_func) {
		(comm->c_errhdl->eh_func)(&comm, &errclass, error, errmsg);
	} else {
		lam_comm_errfatal(&comm, &errclass, error, errmsg);
	}

	lam_nukefunc();

	return(errclass);
}

/*
 *	lam_err_win
 *
 *	Function:	- handle MPI error on a window
 *	Accepts:	- window
 *			- error class
 *			- errno value
 *			- error message
 *	Returns:	- error class
 */
int
lam_err_win(win, errclass, error, errmsg)

MPI_Win			win;
int			errclass;
int			error;
char			*errmsg;

{
/*
 * Catch the uninitialized case when an error is returned in an F77
 * wrapper before the call is made to the C version of the function.  
 */
	lam_initerr_m();
/*
 * If the top function is zero then the error has already been handled.
 */
	if (lam_topfunc == 0) {
	    return(errclass);
	}
/*
 * Compatibility with lam_errfunc() style error handling.  Check if the
 * errorclass has encoded information and if so break it down.
 */
	if (errclass & 0xFFFFFF00) {
		error = (errclass >> 16) & 0xFFFF;
		errclass &= 0xFF;
	}

	if (win->w_errhdl->eh_flags & LAM_LANGF77) {
		(win->w_errhdl->eh_func)(&win->w_f77handle,
					&errclass, error, errmsg);
	} else if (win->w_errhdl->eh_func) {
		(win->w_errhdl->eh_func)(&win, &errclass, error, errmsg);
	} else {
		lam_win_errfatal(&win, &errclass, error, errmsg);
	}

	lam_nukefunc();
	
	return(errclass);
}
