/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *
 *	$Id: initthr.c,v 6.2.2.1 2002/10/09 19:49:09 brbarret Exp $
 *
 *	Function:	- initialize the MPI session for threads
 *			- then calls MPI_Init
 *	Accepts:	- ptr to argc
 *			- ptr to argv
 *			- desired thread level
 *			- provided thread level
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdio.h>

#include <mpi.h>
#include <mpisys.h>
#include <lamthreads.h>
#include <blktype.h>


/*
 * Create the whole MPI universe.  Setup thread support.
 */
/*@
   MPI_Init_thread - Initialize the MPI execution environment

Input Parameters:
+ pargc - Pointer to the number of arguments 
. pargv - Pointer to the argument vector
. requested - Desired level of thread support
- pprovided - Given level of thread support

Notes:

This function is mainly here for link-compatability.  It will
[currently] only ever return 'MPI_THREAD_SINGLE' in 'pprovided'.
Future versions of LAM/MPI will support multi-threaded user programs,
in which case this function must be used to initialize MPI.  Hence,
programmers can use this function now in order to program for future
flexibility.

For the moment, however, see 'MPI_Init'(3).

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_ARG
.N MPI_ERR_INIT
.N MPI_ERR_OTHER

.seealso MPI_Init, MPI_Finalize, lamboot, mpirun, lamhalt

.N ACK
@*/
int
MPI_Init_thread(int *pargc, char ***pargv, int requested, int *pprovided)
{
  int err;
  int fl_init;
  
  lam_setfunc(BLKMPIINITTHREAD);
/*
 * Check if we have been initialized or finalized.
 */
  MPI_Initialized(&fl_init);
  if (fl_init || lam_finalized()) {
    return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINITTHREAD,
		       lam_mkerr(MPI_ERR_ARG, EMPIINIT)));
  }
/*
 * Ensure we got a valid requested and pprovided
 */
  if (requested < MPI_THREAD_SINGLE || 
      requested > MPI_THREAD_MULTIPLE ||
      pprovided == NULL) {
    return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINITTHREAD,
		       lam_mkerr(MPI_ERR_ARG, EMPIINIT)));
  }
/*
 * Initialize the thread interface
 */
  lam_thread_init(requested, pprovided);
/*
 * Now that some global variables have been set, re-set the MPI stack
 * so that it gets locked properly.  This is ok to do because we know
 * that there is only one thread active in MPI right now (it is
 * erroneous to call MPI_Init_thread from more than one thread).
 */
  lam_resetfunc(BLKMPIINITTHREAD);
  lam_setfunc(BLKMPIINITTHREAD);
/*
 * Call MPI_Init
 */
  err = MPI_Init(pargc, pargv);
/*
 * All done
 */
  lam_resetfunc(BLKMPIINITTHREAD);
  return err;
}

