/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1993 The Ohio State University
 *	GDB
 *
 *	$Id: rflat.c,v 6.4 1999/06/16 05:54:53 jsquyres Exp $
 * 
 *	Function:	- runtime access to the flat server
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <args.h>
#include <events.h>
#include <flatreq.h>
#include <ksignal.h>
#include <net.h>
#include <preq.h>
#include <rreq.h>
#include <terror.h>
#include <typical.h>

/*
 *	Function:	- load a buffer into a node and assign it a tag

 *	Accepts:	- server's node ID
 *			- load buffer length
 *			- allocated buffer length
 *			- flat tag
 *			- buffer ptr
 */
int
rflat(nodeid, buffer, ldlength, malength, tag)

int4			nodeid;
char			*buffer;
int4			ldlength;
int4			malength;
int4			tag;

{
	struct flreq	*request;	/* flatd request structure */
	struct flreply	*reply;		/* flatd reply structure */
	struct nmsg	nhead;		/* message descriptor */
	int		mask;		/* signal mask */

	if (malength == 0) return(0);

	request = (struct flreq *) nhead.nh_data;
	reply = (struct flreply *) nhead.nh_data;

	request->flq_src_node =
		((nodeid == LOCAL) || (getrtype(nodeid) & NT_CAST)) ?
		nodeid : getnodeid();
	request->flq_src_event = -getpid();
	request->flq_req = FLQLOAD;
	request->flq_ldlength = ldlength;
	request->flq_malength = malength;
	request->flq_tag = tag;

	nhead.nh_node = nodeid;
	nhead.nh_event = EVFLATD;
	nhead.nh_type = 0;
	nhead.nh_flags = 0;
	nhead.nh_msg = 0;
	nhead.nh_length = 0;
	mask = ksigblock(sigmask(SIGUDIE) | sigmask(SIGARREST));

	if (nsend(&nhead)) {
		ksigsetmask(mask);
		return(ERROR);
	}

	nhead.nh_event = request->flq_src_event;
	
	if (nrecv(&nhead)) {
		ksigsetmask(mask);
		return(ERROR);
	}

	if (reply->flr_reply) {
		errno = reply->flr_reply;
		ksigsetmask(mask);
		return(ERROR);
	}
	else {
/*
 * Load buffer to destination node.
 */
		request->flq_src_node =
			((nodeid == LOCAL) || (getrtype(nodeid) & NT_CAST)) ?
				nodeid : getnodeid();
		request->flq_src_event = -getpid();
		request->flq_req = FLQLOAD;
		request->flq_ldlength = ldlength;
		request->flq_malength = malength;
		request->flq_tag = tag;

		nhead.nh_event = EVFLATD;
		nhead.nh_msg = buffer;
		nhead.nh_type = 0;

		while (ldlength > 0) {
			nhead.nh_length = (ldlength > MAXNMSGLEN)
						? MAXNMSGLEN : ldlength;
			if (nsend(&nhead)) {
				ksigsetmask(mask);
				return(ERROR);
			}

			ldlength -= nhead.nh_length;
			nhead.nh_msg += nhead.nh_length;
		}
	}
/*
 * Receive a reply that ensures that all data has been fully loaded.
 */
	nhead.nh_event = request->flq_src_event;
	nhead.nh_length = 0;
	nhead.nh_msg = 0;

	if (nrecv(&nhead)) {
		ksigsetmask(mask);
		return(ERROR);
	}

	ksigsetmask(mask);

	if (reply->flr_reply) {
		errno = reply->flr_reply;
		return(ERROR);
	}

	return(0);
}

/*
 *	Function:	- load argument and environment vectors
 *			  into a node and assign them a tag
 *	Accepts:	- server's node ID
 *			- argument vector
 *			- process environment
 *			- size of loaded environment (out)
 *			- tag (inout)
 */
int
rflatv(int4 nodeid, char **argv, struct penv *env, int4 *envsize, int4 *tag)
{
    char		*args;
    char		*envs;
    int			argsize;
    int			packsize;
    void		*p;

    if ((argv == 0) && (env->pe_envv == 0)) {
	*envsize = 0;
	return 0;
    }

    if (*tag == 0)
	*tag = (int4) getpid();

    if (argv) {
	args = argvglue(argv, 0xa, MAXNMSGLEN/2);
	if (args == 0)
	    return(LAMERROR);
	argsize = strlen(args);

	if (env->pe_envv) {
	    envs = argvglue(env->pe_envv, 0xa, MAXNMSGLEN/2);
	    if (envs == 0) {
		free(args);
		return(LAMERROR);
	    }

	    *envsize = (int4) strlen(envs);
	    packsize = argsize + *envsize + 1;

	    p = realloc(args, packsize);
	    if (p == 0) {
		free(args);
		free(envs);
		return(LAMERROR);
	    }

	    args = p;
	    memcpy(args + argsize, envs, *envsize + 1);
	    free(envs);
	} else {
	    *envsize = 0;
	    packsize = argsize + 1;
	}

	if (rflat(nodeid, args, (int4) packsize, (int4) packsize, *tag)) {
	    rflclean(nodeid, *tag);
	    free(args);
	    return(LAMERROR);
	}

	free(args);
    }
    else {
	envs = argvglue(env->pe_envv, 0xa, MAXNMSGLEN/2);
	if (envs == 0)
	    return(LAMERROR);

	*envsize = strlen(envs);
	packsize = *envsize + 1;

	if (rflat(nodeid, envs, (int4) packsize, (int4) packsize, *tag)) {
	    rflclean(nodeid, *tag);
	    free(envs);
	    return(LAMERROR);
	}

	free(envs);
    }

    return(0);
}
