/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/NJN
 *
 *	$Id: tfree.c,v 6.3 1999/12/31 21:27:40 jsquyres Exp $
 *
 *	Function:	- mark a datatype for deallocation
 *	Accepts:	- datatype (inout)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*
 * external functions
 */
extern int		lam_dtype_free();

/*@
    MPI_Type_free - Frees the datatype

Input/Output Parameter:
. dtype - datatype that is freed (handle) 

Predefined types:

The MPI standard states that (in Opaque Objects)

.vb
MPI provides certain predefined opaque objects and predefined, static
handles to these objects. Such objects may not be destroyed.
.ve

Thus, it is an error to free a predefined datatype.  The same section
makes it clear that it is an error to free a null datatype.

Upon successful completion, 'dtype' is set to 'MPI_DATATYPE_NULL'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TYPE
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Type_free(MPI_Datatype *dtype)
{
	int		err;			/* error code */

	lam_initerr();
	lam_setfunc(BLKMPITFREE);
/*
 * Check the arguments.
 */
	if (dtype == 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITFREE, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if (*dtype == MPI_DATATYPE_NULL || ((*dtype)->dt_flags & LAM_PREDEF)) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITFREE, lam_mkerr(MPI_ERR_TYPE, 0)));
	}
/*
 * If last reference really free the datatype otherwise just null the handle.
 */
	if (--((*dtype)->dt_refcount) == 0) {

		err = lam_type_free(*dtype);
		*dtype = MPI_DATATYPE_NULL;

		if (err != MPI_SUCCESS) {
			return(lam_errfunc(MPI_COMM_WORLD,
						BLKMPITFREE, err));
		}
	} else {
		*dtype = MPI_DATATYPE_NULL;
	}

	lam_resetfunc(BLKMPITFREE);
	return(MPI_SUCCESS);
}
