/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	$Id: lammutex.c,v 6.2 2000/10/17 12:00:25 jsquyres Exp $
 *
 *	Function:	- LAM interface for mutexes
 */

#include <stdio.h>

#include <mpi.h>
#include <lam_config.h>
#include <lamthreads.h>

/*
 * private variables
 */
#if LAM_HAVE_THREADS
static int first = 1;
static lam_mutex_t is_self_mutex;
#endif


/*
 *	lam_mutex_init
 *
 *	Function:	- initialize the LAM mutex interface
 *      Accepts:        - pointer to mutex
 */
void
lam_mutex_init(lam_mutex_t *mutex)
{
#if LAM_HAVE_THREADS
/*
 * If this is the first time, initialize the is_self_mutex (we know
 * that there won't be a race condition of multiple threads calling
 * this first because MPI_Init_thread() must be the first function
 * called (by itself).
 */
  if (first == 1) {
    first = 0;
    lam_mutex_init(&is_self_mutex);
  }

#if LAM_HAVE_SOL_THREADS
  mutex_init(&(mutex->mutex), 0, 0);
#elif LAM_HAVE_PTHREADS
  pthread_mutex_init(&(mutex->mutex), 0);
#endif
  mutex->thread = (lam_thread_t) -1;
#endif
}


/*
 *	lam_mutex_lock
 *
 *	Function:	- lock a LAM mutex
 *      Accepts:        - pointer to mutex
 */
void
lam_mutex_lock(lam_mutex_t *mutex)
{
#if LAM_HAVE_THREADS
  if (mutex->thread != (lam_thread_t) -1 &&
      mutex->thread == lam_thread_self()) {
    fprintf(stderr, "LAM/MPI WARNING: Some thread attempting to lock a mutex that it already has!\n");
  }

#if LAM_HAVE_SOL_THREADS
  mutex_lock(&(mutex->mutex));
#elif LAM_HAVE_PTHREADS
  pthread_mutex_lock(&(mutex->mutex));
#endif
  mutex->thread = lam_thread_self();
#endif
}


/*
 *	lam_mutex_is_owner
 *
 *	Function:	- determine if this thread owns the mutex
 *      Accepts:        - pointer to mutex
 *      Returns:        - 1 if this thread owns the mutex, 0 otherwise
 */
int
lam_mutex_is_owner(lam_mutex_t *mutex)
{
#if LAM_HAVE_THREADS
  int ret;

  lam_mutex_lock(&is_self_mutex);
  ret = lam_thread_compare(mutex->thread, lam_thread_self());
  lam_mutex_unlock(&is_self_mutex);

  return ret;
#else
  return 1;
#endif
}


/*
 *	lam_mutex_unlock
 *
 *	Function:	- unlock a LAM mutex
 *      Accepts:        - pointer to mutex
 */
void
lam_mutex_unlock(lam_mutex_t *mutex)
{
#if LAM_HAVE_THREADS
  if (mutex->thread != lam_thread_self()) {
    fprintf(stderr, "LAM/MPI WARNING: Some thread unlocking a mutex that it doesn't own\n");
  } else {
    mutex->thread = (lam_thread_t) -1;
  }

#if LAM_HAVE_SOL_THREADS
  mutex_unlock(&(mutex->mutex));
#elif LAM_HAVE_PTHREADS
  pthread_mutex_unlock(&(mutex->mutex));
#endif
#endif
}


/*
 *	lam_mutex_finalize
 *
 *	Function:	- destroy a LAM mutex
 *      Accepts:        - pointer to mutex
 */
void
lam_mutex_finalize(lam_mutex_t *mutex)
{
#if LAM_HAVE_THREADS
  if (mutex->thread != (lam_thread_t) -1) {
    fprintf(stderr, "LAM/MPI WARNING: Some thread destroying a mutex that is in use!\n");
  }

#if LAM_HAVE_SOL_THREADS
  mutex_destroy(&(mutex->mutex));
#elif LAM_HAVE_PTHREADS
  pthread_mutex_destroy(&(mutex->mutex));
#endif
#endif
}
