/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: infogetvlen.c,v 6.5 2001/02/03 20:24:34 jsquyres Exp $
 *
 *	Function:	- get length of info value associated with key
 *	Accepts:	- info
 *			- key
 *			- value length (out)
 *			- key found flag (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <string.h>

#include <all_list.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <terror.h>


/*@
  MPI_Info_get_valuelen - Get the length of a value for a given key in an 'MPI_Info' object

Input Parameters:
+ info - info object (handle)
- key - null-terminated character string of the index key

Output Parameters:
+ valuelen - length of the value associated with 'key' (integer)
- flag - true (1) if 'key' defined on 'info', false (0) if not
(logical)

Notes:

The length returned in C and C++ does not include the end-of-string
character.  If the 'key' is not found on 'info', 'valuelen' is set to 
'MPI_UNDEFINED'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_ARG
.N MPI_ERR_INFO_KEY

.N WEB
@*/
int MPI_Info_get_valuelen(MPI_Info info, char *key, 
			  int *valuelen, int *flag)
{
	struct _infoent	search;			/* search element */
	struct _infoent	*found;			/* found key/value pair */
	int		keylen;			/* length of key */
	
	lam_initerr_m();
	lam_setfunc_m(BLKMPIINFOGETLEN);

	if (info == 0 || valuelen == 0 || flag == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINFOGETLEN,
					lam_mkerr(MPI_ERR_ARG, 0)));
	}

	keylen = (key) ? strlen(key) : 0;
	if (keylen == 0 || keylen > MPI_MAX_INFO_KEY) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINFOGETLEN,
					lam_mkerr(MPI_ERR_INFO_KEY, 0)));
	}

	strcpy(search.ie_key, key);

	if ((found = al_find(info->info_list, &search))) {
		*flag = 1;
		*valuelen = strlen(found->ie_value);
	} else {
		*flag = 0;
		*valuelen = MPI_UNDEFINED;
	}

	lam_resetfunc_m(BLKMPIINFOGETLEN);
	return(MPI_SUCCESS);
}
