/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD
 *
 *	$Id: gtranks.c,v 6.4 1999/12/31 21:27:28 jsquyres Exp $
 *
 *	Function:	- get process ranks in a second group
 *	Accepts:	- first group
 *			- # of processes
 *			- array of ranks in first group
 *			- second group
 *			- array of ranks in second group (returned values)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Group_translate_ranks - Translates the ranks of processes in one
group to those in another group

Input Parameters:
+ g1 - group1 (handle) 
. n - number of ranks in 'r1' and 'r2' arrays (integer)
. r1 - array of zero or more valid ranks in 'g1' 
- g2 - group2 (handle) 

Output Parameter:
. r2 - array of corresponding ranks in 'g2', 'MPI_UNDEFINED' when no
correspondence exists.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_GROUP
.N MPI_ERR_ARG
.N MPI_ERR_RANK

.N ACK
@*/
int MPI_Group_translate_ranks(MPI_Group g1, int n, 
			      int *r1, MPI_Group g2, int *r2)
{
	int		i, j;			/* favourite indices */
	int		rank;			/* process rank */
	struct _proc	**p1, **p2;		/* gps pointers */

	lam_initerr();
	lam_setfunc(BLKMPIGROUPTRANKS);
/*
 * Check given arguments.
 */
	if ((g1 == MPI_GROUP_NULL) || (g2 == MPI_GROUP_NULL)) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGROUPTRANKS, lam_mkerr(MPI_ERR_GROUP, 0)));
	}

	if (n == 0) {
		lam_resetfunc(BLKMPIGROUPTRANKS);
		return(MPI_SUCCESS);
	}

	if ((n < 0) || (r1 == 0) || (r2 == 0)) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGROUPTRANKS, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Loop over the processes, translating their ranks.
 */
	for (i = 0; i < n; ++i) {
		rank = r1[i];

		if ((rank >= 0) && (rank < g1->g_nprocs)) {

			p1 = g1->g_procs + rank;
			p2 = g2->g_procs;

			for (j = 0; j < g2->g_nprocs; ++j, ++p2) {
				if (*p1 == *p2) break;
			}

			r2[i] = (j < g2->g_nprocs) ? j : MPI_UNDEFINED;
		}
		else {
			return(lam_errfunc(MPI_COMM_WORLD,
				BLKMPIGROUPTRANKS, lam_mkerr(MPI_ERR_RANK, 0)));
		}
	}

	lam_resetfunc(BLKMPIGROUPTRANKS);
	return(MPI_SUCCESS);
}
