/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/JRV
 *
 *	$Id: gatherv.c,v 6.4 1999/12/31 21:27:24 jsquyres Exp $
 *
 *	Function:	- gather vectored buffers at root in process rank order
 *	Accepts:	- send buffer
 *			- send count
 *			- send datatype
 *			- recv buffer
 *			- recv counts
 *			- displacements
 *			- recv datatype
 *			- root
 *			- communicator
 *	Returns:	- MPI_SUCCESS or an MPI error code
 */

#include <lam_config.h>
#include <app_mgmt.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>

/*@

MPI_Gatherv - Gathers into specified locations from all processes in a group

Input Parameters:
+ sendbuf - starting address of send buffer (choice) 
. sendcount - number of elements in send buffer (integer) 
. sendtype - data type of send buffer elements (handle) 
. recvcounts - integer array (of length group size) 
containing the number of elements that are received from each process
(significant only at 'root') 
. displs - integer array (of length group size). Entry 
 'i'  specifies the displacement relative to recvbuf  at
which to place the incoming data from process  'i'  (significant only
at root) 
. recvtype - data type of recv buffer elements 
(significant only at 'root') (handle) 
. root - rank of receiving process (integer) 
- comm - communicator (handle) 

Output Parameter:
. recvbuf - address of receive buffer (choice, significant only at 'root') 

.N IMPI

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_INTERCOMM
.N MPI_ERR_IMPI
.N MPI_ERR_TYPE
.N MPI_ERR_BUFFER
.N MPI_ERR_ROOT
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Gatherv(void *sbuf, int scount, MPI_Datatype sdtype,
		void *rbuf, int *rcounts, int *disps, 
		MPI_Datatype rdtype, int root, MPI_Comm comm)
{
	int		i;			/* favourite index */
	int		rank;			/* my rank */
	int		size;			/* group size */
	int		err;			/* error code */
	char		*ptmp;			/* temp. buffer */
	MPI_Aint	extent;			/* datatype extent */
	MPI_Status	stat;			/* receive status */
	struct _gps	*p;			/* favourite pointer */

	lam_initerr();
	lam_setfunc(BLKMPIGATHERV);
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm, BLKMPIGATHERV,
					lam_mkerr(MPI_ERR_COMM, 0)));
	}

	MPI_Comm_rank(comm, &rank);
	MPI_Comm_size(comm, &size);

	if ((root >= size) || (root < 0)) {
		return(lam_errfunc(comm, BLKMPIGATHERV,
					lam_mkerr(MPI_ERR_ROOT, 0)));
	}

	if ((sdtype == MPI_DATATYPE_NULL)
			|| (rank == root && rdtype == MPI_DATATYPE_NULL)) {
		return(lam_errfunc(comm, BLKMPIGATHERV,
					lam_mkerr(MPI_ERR_TYPE, 0)));
	}


	if ((rank == root) && (disps == 0)) {
		return(lam_errfunc(comm, BLKMPIGATHERV,
					lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((scount < 0) || (rank == root && rcounts == 0)) {
		return(lam_errfunc(comm, BLKMPIGATHERV,
					lam_mkerr(MPI_ERR_COUNT, 0)));
	}


#if LAM_WANT_IMPI
	/* Remove this when IMPI collectives are implemented */

        if (LAM_IS_IMPI(comm)) {
	  return lam_err_comm(comm, MPI_ERR_COMM, 0, 
			      "Collectives not yet implemented on IMPI communicators");
	}    
#endif

	LAM_TRACE(lam_tr_cffstart(BLKMPIGATHERV));
/*
 * Remember required parameters.
 */
	p = &(comm->c_group->g_procs[root]->p_gps);

	lam_setparam(BLKMPIGATHERV, root | (p->gps_grank << 16),
				(p->gps_node << 16) | p->gps_idx);
/*
 * Switch to collective communicator.
 */
	lam_mkcoll(comm);
/*
 * Everyone but root sends data and returns.
 */
	if (rank != root) {

		err = MPI_Send(sbuf, scount, sdtype,
					root, BLKMPIGATHERV, comm);
		lam_mkpt(comm);

		if (err != MPI_SUCCESS) {
			return(lam_errfunc(comm, BLKMPIGATHERV, err));
		}

		LAM_TRACE(lam_tr_cffend(BLKMPIGATHERV,
						root, comm, sdtype, scount));

		lam_resetfunc(BLKMPIGATHERV);
		return(MPI_SUCCESS);
	}
/*
 * I am the root, loop receiving data.
 */
	MPI_Type_extent(rdtype, &extent);

	for (i = 0; i < size; ++i) {

		ptmp = ((char *) rbuf) + (extent * disps[i]);
/*
 * simple optimization
 */
		if (i == rank) {
			err = lam_dtsndrcv(sbuf, scount, sdtype,
						ptmp, rcounts[i], rdtype,
						BLKMPIGATHERV, comm);
		} else {
			err = MPI_Recv(ptmp, rcounts[i], rdtype, i,
						BLKMPIGATHERV, comm, &stat);
		}

		if (err != MPI_SUCCESS) {
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPIGATHERV, err));
		}
	}

	lam_mkpt(comm);

	LAM_TRACE(lam_tr_cffend(BLKMPIGATHERV, root, comm, sdtype, scount));

	lam_resetfunc(BLKMPIGATHERV);
	return(MPI_SUCCESS);
}
