/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: cdisconnect.c,v 6.4 2001/01/30 18:26:42 jsquyres Exp $
 *
 *	Function:	- completes all pending communication on
 *			  a communicator and frees it
 *	Accepts:	- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>
#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>


/*@
   MPI_Comm_disconnect - Disconnect a client and server

Input Parameter:

. comm - communicator

.N fortran

.N IMPI_MPI2

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_OTHER

.seealso MPI_Close_port(3), MPI_Comm_accept(3), MPI_Comm_connect(3), MPI_Open_port(3)

.N WEB
@*/
int
MPI_Comm_disconnect(MPI_Comm *comm)
{
	MPI_Request 	*reqs;			/* array of pending requests */
	struct _req	*r;			/* request */
	int4		ccid;			/* collective context ID */
	int		npend = 0;		/* number of pending requests */
	int		err;			/* error return */
	int		i;
	
	lam_initerr();
	lam_setfunc(BLKMPICOMMDISCONNECT);

	LAM_TRACE(lam_tr_cffstart(BLKMPICOMMDISCONNECT));
/*
 * Find number of pending requests on the communicator.
 */
	ccid = lam_pt2coll((*comm)->c_contextid);
	
	for (r = lam_rq_top; r; r = r->rq_next) {
		if (r->rq_state != LAM_RQSDONE && (r->rq_cid == ccid
				|| r->rq_cid == (*comm)->c_contextid)) {
			npend++;
		}
	}
/*
 * Create array of pending requests to wait on.
 */
	reqs = (MPI_Request *) malloc((unsigned) npend * sizeof(MPI_Request));
	if (reqs == 0) {
		return(lam_errfunc(*comm, BLKMPICOMMDISCONNECT,
				lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	for (i = 0, r = lam_rq_top; r; r = r->rq_next) {
		if (r->rq_state != LAM_RQSDONE && (r->rq_cid == ccid
				|| r->rq_cid == (*comm)->c_contextid)) {
			reqs[i++] = r;
		}
	}

	err = MPI_Waitall(npend, reqs, MPI_STATUSES_IGNORE);

	free((char *) reqs);
	if (err) {
		return(lam_errfunc(*comm, BLKMPICOMMDISCONNECT, err));
	}
	
	LAM_TRACE(lam_tr_cffend(BLKMPICOMMDISCONNECT, -1, *comm, 0, 0));

	if ((err = MPI_Comm_free(comm)) != MPI_SUCCESS) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPICOMMDISCONNECT, err));
	}
	*comm = MPI_COMM_NULL;

        lam_resetfunc(BLKMPICOMMDISCONNECT);
	return(MPI_SUCCESS);
}
