/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: cartrank.c,v 6.3 1999/12/31 21:27:18 jsquyres Exp $
 *
 *	Function:	- translate coordinates to rank
 *	Accepts:	- communicator
 *			- coordinates array
 *			- ptr rank (returned value)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Cart_rank - Determines process rank in communicator given Cartesian
                location

Input Parameters:
+ comm - communicator with cartesian structure (handle) 
- coords - integer array (of size  'ndims') specifying the cartesian 
  coordinates of a process 

Output Parameter:
. prank - rank of specified process (integer) 

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_TOPOLOGY
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Cart_rank(MPI_Comm comm, int *coords, int *prank)
{
	int		rank;
	int		dim;
	int		ord;
	int		factor;
	int		i;
	int		*d;
	int		*c;

	lam_initerr();
	lam_setfunc(BLKMPICARTRANK);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICARTRANK, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPICARTRANK, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (!LAM_IS_CART(comm)) {
		return(lam_errfunc(comm, BLKMPICARTRANK,
				lam_mkerr(MPI_ERR_TOPOLOGY, 0)));
	}

	if ((coords == 0) || (prank == 0)) {
		return(lam_errfunc(comm,
			BLKMPICARTRANK, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Loop over coordinates computing the rank.
 */
	factor = 1;
	rank = 0;
	i = comm->c_topo_ndims - 1;
	d = comm->c_topo_dims + i;
	c = coords + i;

	for (; i >= 0; --i, --c, --d) {
		dim = (*d > 0) ? *d : -(*d);
		ord = *c;
		if ((ord < 0) || (ord >= dim)) {
			if (*d > 0) {
				return(lam_errfunc(comm, BLKMPICARTRANK,
					lam_mkerr(MPI_ERR_ARG, 0)));
			}

			ord %= dim;
			if (ord < 0) {
				ord += dim;
			}
		}

		rank += factor * ord;
		factor *= dim;
	}

	*prank = rank;

	lam_resetfunc(BLKMPICARTRANK);
	return(MPI_SUCCESS);
}
