/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: cartget.c,v 6.3 1999/12/31 21:27:18 jsquyres Exp $
 *
 *	Function:	- get info on cartesian topology
 *	Accepts:	- communicator
 *			- size of dimensions array
 *			- dimensions array
 *			- periodicity array
 *			- coordinates array
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Cart_get - Retrieves Cartesian topology information associated with a 
               communicator

Input Parameters:
+ comm - communicator with cartesian structure (handle) 
- maxndims - length of vectors  'dims', 'periods', and 'coords'
in the calling program (integer) 

Output Parameters:
+ dims - number of processes for each cartesian dimension (array of integer) 
. periods - periodicity (true/false) for each cartesian dimension 
(array of logical) 
- coords - coordinates of calling process in cartesian structure 
(array of integer) 

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TOPOLOGY
.N MPI_ERR_COMM
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Cart_get(MPI_Comm comm, int maxndims, int *dims, 
		 int *periods, int *coords)
{
	int		i;
	int		*d;
	int		*c;

	lam_initerr();
	lam_setfunc(BLKMPICARTGET);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICARTGET, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPICARTGET, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (!LAM_IS_CART(comm)) {
		return(lam_errfunc(comm, BLKMPICARTGET,
				lam_mkerr(MPI_ERR_TOPOLOGY, 0)));
	}

	if ((maxndims < 0) || ((maxndims > 0) &&
			((dims == 0) || (periods == 0) || (coords == 0)))) {
		return(lam_errfunc(comm,
			BLKMPICARTGET, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Fill the dimension, period, and coordinate arrays.
 */
	d = comm->c_topo_dims;
	c = comm->c_topo_coords;
	for (i = 0; (i < comm->c_topo_ndims) && (i < maxndims); ++i) {
		if (*d > 0) {
			*dims++ = *d++;
			*periods++ = 0;
		} else {
			*dims++ = -(*d++);
			*periods++ = 1;
		}

		*coords++ = *c++;
	}

	lam_resetfunc(BLKMPICARTGET);
	return(MPI_SUCCESS);
}
