/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD
 *
 *	$Id: cartcoords.c,v 6.4 1999/12/31 21:27:17 jsquyres Exp $
 *
 *	Function:	- translate rank to coordinates
 *	Accepts:	- communicator
 *			- rank
 *			- max # dimensions to fill
 *			- coordinates array (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Cart_coords - Determines process coords in cartesian topology given
                  rank in group

Input Parameters:
+ comm - communicator with cartesian structure (handle) 
. rank - rank of a process within group of 'comm' (integer) 
- maxdims - length of vector 'coords' in the calling program (integer) 

Output Parameter:
. coords - integer array (of size 'ndims') containing the cartesian
coordinates of specified process (integer)

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_TOPOLOGY
.N MPI_ERR_RANK
.N MPI_ERR_DIMS
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Cart_coords(MPI_Comm comm, int rank, int maxdims, 
		    int *coords)
{
	int		dim;
	int		remprocs;
	int		i;
	int		*d;

	lam_initerr();
	lam_setfunc(BLKMPICARTCOORDS);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICARTCOORDS, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPICARTCOORDS, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (!LAM_IS_CART(comm)) {
		return(lam_errfunc(comm, BLKMPICARTCOORDS,
				lam_mkerr(MPI_ERR_TOPOLOGY, 0)));
	}

	if ((rank < 0) || (rank >= comm->c_group->g_nprocs)) {
		return(lam_errfunc(comm,
			BLKMPICARTCOORDS, lam_mkerr(MPI_ERR_RANK, 0)));
	}

	if ((maxdims < 0) || ((maxdims > 0) && (coords == 0))) {
		return(lam_errfunc(comm,
			BLKMPICARTCOORDS, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Loop computing the coordinates.
 */
	d = comm->c_topo_dims;
	remprocs = comm->c_topo_nprocs;
	for (i = 0; (i < comm->c_topo_ndims) && (i < maxdims); ++i, ++d) {
		dim = (*d > 0) ? *d : -(*d);
		remprocs /= dim;
		*coords++ = rank / remprocs;
		rank %= remprocs;
	}

	lam_resetfunc(BLKMPICARTCOORDS);
	return(MPI_SUCCESS);
}
