/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB
 *
 *	$Id: kexit.c,v 6.3 1999/05/25 17:50:36 kmeyer1 Exp $
 *
 *	Function:	- exit package
 *			- performs all tasks required when a process exits
 */

#include <stdio.h>
#include <unistd.h>

#include <kio.h>
#include <preq.h>
#include <terror.h>
#include <typical.h>
#include <laminternal.h>

/*
 * global functions
 */
void			_kexit();
void			kexit();
void			lam_kexit();
int			atkexit();

/*
 * external variables
 */
extern struct kio_t	_kio;			/* Kernel Input/Output block */

/*
 *	atkexit
 *
 *	Function:	- add function to cleanup table
 *	Accepts:	- pointer to exit function
 *			- argument to exit function
 *			- does not store a combination of exit function
 *			  and argument more than once
 *	Returns:	- 0 or ERROR
 */
int
atkexit(exitfunc, arg)

void			(*exitfunc)();
char			*arg;

{
	int		i;		/* favourite index variable */

	for (i = 0; (i < TNEXIT) && (_kio.ki_exit[i]) &&
			!((_kio.ki_exit[i] == exitfunc) &&
			(_kio.ki_arg[i] == arg)); ++i);

	if (i < TNEXIT) {

		if (!_kio.ki_exit[i]) {
			_kio.ki_exit[i] = exitfunc;
			_kio.ki_arg[i] = arg;
		}

		return(0);
	}

	else {
		errno = ENOEXDESCRIPTORS;
		return(LAMERROR);
	}
}

/*
 *	lam_kexit
 *
 *	Function:	- terminates Trollius runtime system
 *			- calls cleanup routines
 *			- detaches from kenyad and the kernel
 *	Accepts:	- exit status
 */
void
lam_kexit(status)

int			status;

{
	int		i;

	if (_kio.ki_pid == getpid()) {

		for (i = TNEXIT - 1; i >= 0; --i) {

			if (_kio.ki_exit[i]) {
				(*(_kio.ki_exit[i]))(_kio.ki_arg[i], status);
			}
		}

		if (_kio.ki_rtf & RTF_IO) {
			fflush(stdout);
			fflush(stderr);
		}
		
		if (_kio.ki_rtf & RTF_KENYA) {
			lpdetach(0, status);
		}

		kdetach(0);
	}
/*
 * Destroy underlying communication structures and free related resources.
 */
	_cio_cleanup();
}

/*
 *	kexit
 *
 *	Function:	- exits from Trollius
 *			- calls cleanup routines
 *			- does not return
 *	Accepts:	- exit status
 */
void
kexit(status)

int			status;

{
	lam_kexit(status);
/*
 * _cio_bail does not return.
 */
	_cio_bail(status);
}

/*
 *	_kexit
 *
 *	Function:	- exits from Trollius
 *			- does not call cleanup routines
 *			- does not return
 *	Accepts:	- exit status
 */
void
_kexit(status)

int			status;

{
	if (_kio.ki_pid == getpid()) {

		if (_kio.ki_rtf & RTF_KENYA) {
			lpdetach(0, status);
		}

		kdetach(0);
	}
/*
 * Destroy underlying communication structures and free related resources.
 */
	_cio_cleanup();
/*
 * _cio_bail does not return.
 */
	_cio_bail(status);
}
