/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres and Arun Rodrigues with Brian Barrett,
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 * 
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB/NJN
 *
 *	$Id: kill.c,v 6.11 2001/02/16 17:10:35 jsquyres Exp $
 *
 *	Function:	- kill file useful routines
 */

#include <stdio.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netdb.h>			/* MAXHOSTNAMELEN in Solaris */
#include <unistd.h>
#include <errno.h>
   
#include <laminternal.h>
#include <terror.h>
#include <typical.h>

#define KILLDIR		"/tmp/lam-"


/*
 * local variables
 */
static char *preset_killname = NULL;

/*
 * local variables
 */
static int check_dir_perms(char* dirname);


/*
 *	get_batchid
 *
 *	Function:	- get batch job ID
 *			- only checks for PBS at the moment
 *			- this prevents LAM from using the same socket
 *			  name on multiple jobs in a batch environment,
 *			  where the scheduler may put multiple LAM/MPI
 *			  jobs on the same host
 *
 *	Returns:	- job ID in a string, or "" 
 */
char*
get_batchid(void)
{
  static char ret[128];
  static int set = 0;

  if (set == 0) {
    set = 1;
    ret[0] = ret[127] = '\0';

    /*
     * We only check for PBS at the moment 
     */    
    if (getenv("LAM_MPI_SOCKET_SUFFIX") != NULL) {
      strncpy(ret, getenv("LAM_MPI_SOCKET_SUFFIX"), 127);
    } else if (getenv("PBS_ENVIRONMENT") != NULL && 
	       getenv("PBS_JOBID") != NULL) {
      strcpy(ret, "pbs");
      strncat(ret, getenv("PBS_JOBID"), 124);
    }
    /*
     * Add more batch scheduler ID checks here...
     */    
  }

  return ret;
}


/*
 *	set_killname
 *
 *	Function:	- pre-set the killname
 *			- used to override whatever the environment may say
 */
void
set_killname(char *name)
{
  /*
   * ditch the last preset killname
   */
  if (preset_killname != NULL) {
    free(preset_killname);
    preset_killname = NULL;
  }

  /*
   * set the next preset killname, or NULL it out
   */
  if (name != NULL)
    preset_killname = strdup(name);
  else
    preset_killname = NULL;
}


/*
 *	killname
 *
 *	Function:	- forms kill filename
 *			- allocates space
 *	Returns:	- kill filename
 */
char *
killname(void)
{
	char		*f_kill;		/* kill file */
	char		*user;			/* user name */
	char		*batchid;		/* batch job id */
	struct passwd	*pwdent;		/* user's passwd file entry */
	char		host[MAXHOSTNAMELEN + 1];

	if ((pwdent = getpwuid(getuid())) == 0) {
	  return(0);
	}
	
	user = pwdent->pw_name;
/*
 * See if we've got a preset_killname.  If so, use that, and don't
 * check for anything else.
 */
	if (preset_killname != NULL) {
/*
 * a few extra spaces for good measure
 */
	  f_kill = malloc((unsigned) (strlen(KILLDIR) + strlen(user) +
				      strlen(preset_killname) + 64));
	  
	  if (!f_kill) {
	    return(0);
	  }
	  
	  strcpy(f_kill, KILLDIR);
	  strcat(f_kill, user);
	  strcat(f_kill, "@");
	  strcat(f_kill, preset_killname);
	}
/*
 * We're not using a preset killname, so make one up
 */
	else {
	  if (gethostname(host, sizeof(host))) {
	    return(0);
	  }
/*
 * Check for running under a batch system.  
 */
	  batchid = get_batchid();
/*
 * 2 spaces for '@' and '\0', and 64 expansion spaces
 */
	  f_kill = malloc((unsigned) (strlen(KILLDIR) + strlen(host) +
				      strlen(user) + strlen(batchid) + 
				      2 + 64));
	  
	  if (!f_kill) {
	    return(0);
	  }
	  
	  strcpy(f_kill, KILLDIR);
	  strcat(f_kill, user);
	  strcat(f_kill, "@");
	  if (strlen(batchid) > 0)
	    strcat(f_kill, batchid);
	  else
	    strcat(f_kill, host);
	}

/*
 * Because Solaris won't open a socket with secure permissions; we
 * have to make this in a subdirectory
*/
	if (check_dir_perms(f_kill) == (LAMERROR))
	  return 0;
	strcat(f_kill, "/lam");

	return(f_kill);
}

/*
 *	sockname
 *
 *	Function:	- forms process socket filename 
 *	Returns:	- socket filename
 */
char *
sockname(void)
{
	char		*f_sock;		/* socket filename */

	f_sock = killname();
	if (f_sock == 0) return(0);
		
	strcat(f_sock, "-s");
	return(f_sock);
}

/*
 *	iosockname
 *
 *	Function:	- forms IO daemon server socket filename 
 *	Returns:	- socket filename
 */
char *
iosockname(void)
{
	char		*f_sock;		/* socket filename */

	f_sock = killname();
	if (f_sock == 0) return(0);
		
	strcat(f_sock, "-sio");
	return(f_sock);
}

/*
 *	lam_mktmpid
 *
 *	Function:	- create temporary file name based on an id
 *			- returns error if insufficient space
 *	Accepts:	- id number to make name unique
 *			- temporary file name (out)
 *			- size of storage for temporary name
 *	Returns:	- 0 or LAMERROR
 */
int
lam_mktmpid(int id, char *name, unint n)
{
	char		*user;			/* user name */
	struct passwd	*pwdent;		/* user's passwd file entry */
	char		idbuf[8];
	char		*batchid;		/* batch job id */
	char		host[MAXHOSTNAMELEN + 1];

	  if ((pwdent = getpwuid(getuid())) == 0) {
	    return(LAMERROR);
	  }
	  
	  user = pwdent->pw_name;
/*
 * See if we've got a preset_killname.  If so, use that, and don't
 * check for anything else.
 */
	if (preset_killname != NULL) {
/*
 * 3 spaces for '@', '-' and '\0', 64 for id
 */
	  if ((strlen(KILLDIR) + strlen(user) + 
	       strlen(preset_killname) + 3 + 64) > n) {
	    LAMSetLastError(LAM_ERR_INVAL);
	    return(LAMERROR);
	  }
	  
	  strcpy(name, KILLDIR);
	  strcat(name, user);
	  strcat(name, "@");
	  strcat(name, preset_killname);
	}
/*
 * We're not using a preset killname, so make one up
 */
	else {
	  if (gethostname(host, sizeof(host))) {
	    return(LAMERROR);
	  }
/*
 * Check for running under a batch system.  
 */
	  batchid = get_batchid();
/*
 * 3 spaces for '@', '-' and '\0', 64 for id
 */
	  if ((strlen(KILLDIR) + strlen(user) + strlen(host) + 
	       strlen(batchid) + 3 + 64) > n) {
	    LAMSetLastError(LAM_ERR_INVAL);
	    return(LAMERROR);
	  }
	  
	  strcpy(name, KILLDIR);
	  strcat(name, user);
	  strcat(name, "@");
	  if (strlen(batchid) > 0)
	    strcat(name, batchid);
	  else
	    strcat(name, host);
	}
/*
 * append id
 */
	if (check_dir_perms(name) == LAMERROR) {
	  memset(name, 0, n);
	  return LAMERROR;
	}

	strcat(name, "/lam-");
	sprintf(idbuf, "%d", id);
	strncat(name, idbuf, 8);
	
	return(0);
}


/*
 * make the directory if it doesn't exist, ensure the perms are secure
 */
static int
check_dir_perms(char* dirname)
{
  int ret, statret, mode;
  int badfile = 0;
  int baddir = 0;
  struct stat s;

  do {
    statret = stat(dirname, &s);
    if (statret == -1) {
      if (errno == EINTR)
	continue;
      else if (errno == ENOENT)
	break;
      else if (errno == EACCES)
	return (LAMERROR);
    }
  } while (statret != 0);

  /* If there is some file by this name, ensure that it is a
     directory, has the right permissions, and has the right owner */

  if (statret == 0) {

    /* Check that it's a directory */

    if ((s.st_mode & S_IFDIR) != S_IFDIR)
      badfile = 1;
    else {

    /* Check permissions */

      mode = (S_IRWXU | S_IRWXG | S_IRWXO);
      if ((s.st_mode & mode) != S_IRWXU)
	baddir = 1;

      /* Check owner */

      if (s.st_uid != getuid())
	baddir = 1;
    }

    /* If we need to remove something, do it */

    if (badfile == 1 || baddir == 1) {
      do {
	if (badfile == 1)
	  ret = unlink(dirname);
	else
	  ret = rmdir(dirname);

	if (ret == -1) {
	  if (errno == EINTR)
	    continue;
	  else if (errno == ENOENT)
	    /* shouldn't happen, but... */
	    break;
	  else
	    /* If we couldn't remove it, badness */
	    return (LAMERROR);
	}
      } while (ret != 0);
    } else
      /* Otherwise, we have goodness */
      return 0;
  }

  /* Make the directory */

  mode = umask(077);
  ret = mkdir(dirname, 0700);
  umask(mode);

  if (ret == 0)
    return 0;
  else
    return (LAMERROR);
}
