/* This file is part of the KDE libraries
    Copyright (C) 1997 Torben Weis (weis@kde.org)

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#ifndef KxSOCK_H
#define KxSOCK_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef NULL
#undef NULL
#endif
#define NULL 0

#include <sys/types.h>
#include <sys/socket.h> 
#include <sys/un.h>
#include <netinet/in.h>

#include <qsocketnotifier.h>

int gethostbyname_r_portable(const char *szHostName, struct hostent *h);

/** 
 * A TCP/IP client socket. You can connect this socket to any internet address. 
 *
 * The socket gives you three signals: When ready for reading/writing or 
 * if the connection is broken.  Using socket() you get a file descriptor
 * which you can use with usual unix commands like write(..) or read(...). 
 * If you have already such a socket identifier you can construct a kxSocket
 * on this identifier. 
 *  
 * If socket() delivers a value of -1 or less, the connection 
 * had no success.
 *
 * @author Torben Weis <weis@uni-frankfurt.de>
 * @version $Id: kxsock.h,v 1.1.1.1 2000/10/09 17:19:18 herwinjs Exp $
 * @short a TCP/IP client socket.
 */
class kxSocket : public QObject
{
    Q_OBJECT
public:
    /** 
     * Create a kxSocket with the provided file descriptor.
     * @param _sock	the file descriptor to use.
     */
    kxSocket( int _sock )
  	: sock(_sock), domain(0), 
	    readNotifier(0), writeNotifier(0) {}
    
    /** 
     * Create a socket and connect to a host.
     * @param _host	the remote host to which to connect.
     * @param _port	the port on the remote host.
     */
    kxSocket( const char *_host, unsigned short int _port );
    kxSocket( Q_UINT32 _host, unsigned short int _port );
    kxSocket( const char *_host, unsigned short int _port, int timeOut);
    
    /** 
     * Connects to a UNIX domain socket.
     * @param _path    the filename of the socket
     */
    kxSocket( const char *_path );

    /** 
     * Destructor. Closes the socket if it is still open.
     */
    ~kxSocket();
    
    /** 
     * Returns a file descriptor for this socket.
     */
    int socket() const { return sock; }
    
    /** 
     * Enable the socket for reading.
     *
     * If you enable read mode, the socket will emit the signal
     * readEvent whenever there is something to read out of this
     * socket.
     */
    void enableRead( bool );
    
    /** 
     * Enable the socket for writing.
     *
     * If you enable write mode, the socket will emit the signal
     * writeEvent whenever the socket is ready for writing.
     */
    void enableWrite( bool );
    
    /**
     * Return address.
     */
    unsigned long getAddr();
    
signals:
    /** 
     * Data has arrived for reading.
     *
     * This signal will only be raised if enableRead( TRUE ) was called
     * first.
     */
    void readEvent( kxSocket * );
    
    /** 
     * Socket is ready for writing.
     *
     * This signal will only be raised if enableWrite( TRUE ) was called
     * first.
     */
    void writeEvent( kxSocket * );
    
    /** 
     * Raised when the connection is broken.
     */
    void closeEvent( kxSocket * );
    
public slots:
    /** 
     * Connected to the writeNotifier.
     */
    void slotWrite( int );
    
    /** 
     * Connected to the readNotifier.
     */
    void slotRead( int );
    
protected:
    bool connect( const char *_host, unsigned short int _port );
    bool connect( Q_UINT32 _host, unsigned short int _port );
    bool connect( const char *_path );
  
    bool init_sockaddr( const char *hostname, unsigned short int port );
    bool init_sockaddr( Q_UINT32 hostname, unsigned short int port );
    
    struct sockaddr_in server_name;
    struct sockaddr_un unix_addr;

    /******************************************************
     * The file descriptor for this socket. sock may be -1.
     * This indicates that it is not connected.
     */
    int sock;

    int domain;
  
    QSocketNotifier *readNotifier;
    QSocketNotifier *writeNotifier;

private:
    int timeOut;
    
    kxSocket(const kxSocket&);
    kxSocket& operator=(const kxSocket&);
};

/**
 * Monitor a port for incoming TCP/IP connections.
 *
 * You can use a kxServerSocket to listen on a port for incoming
 * connections. When a connection arrived in the port, a kxSocket
 * is created and the signal accepted is raised. Make sure you
 * always connect to this signal. If you dont the ServerSocket will
 * create new kxSocket's and no one will delete them!
 *
 * If socket() is -1 or less the socket was not created properly.
 *
 * @author Torben Weis <weis@stud.uni-frankfurt.de>
 * @version $Id: kxsock.h,v 1.1.1.1 2000/10/09 17:19:18 herwinjs Exp $
 * @short Monitor a port for incoming TCP/IP connections.
*/
class kxServerSocket : public QObject
{
    Q_OBJECT
public:
    /**
     * Constructor.
     * @param _port	the port number to monitor for incoming connections.
     */
    kxServerSocket( int _port );

    /**
     * Creates a UNIX domain server socket.
     */
    kxServerSocket( const char *_path );
  
    /** 
     * Destructor. Closes the socket if it was not already closed.
     */
    ~kxServerSocket();
    
    /** 
     * Get the file descriptor assoziated with the socket.
     */
    int socket() const { return sock; }

    /** 
     * Returns the port number which is being monitored.
     */
    unsigned short getPort();

    /** 
     * The address.
     */
    unsigned long getAddr();

public slots: 
    /** 
     * Called when someone connected to our port.
     */
    virtual void slotAccept( int );

signals:
    /**
     * A connection has been accepted.
     * It is your task to delete the kxSocket if it is no longer needed.
     */
    void accepted( kxSocket* );

protected:
    bool init( short unsigned int );
    bool init( const char* _path );
  
    /** 
     * Notifies us when there is something to read on the port.
     */
    QSocketNotifier *notifier;
    
    /** 
     * The file descriptor for this socket. sock may be -1.
     * This indicates that it is not connected.
     */    
    int sock;  

    int domain;

private:
    Q_UINT32 our_ip;

    kxServerSocket(const kxServerSocket&);
    kxServerSocket& operator=(const kxServerSocket&);
};


#endif


