/***************************************************************************
                          kstarsdata.cpp  -  K Desktop Planetarium
                             -------------------
    begin                : Sun Jul 29 2001
    copyright            : (C) 2001 by Heiko Evermann
    email                : heiko@evermann.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
//JH 25.08.2001: added i18n() to strings

#include <kapp.h>
#include <kmessagebox.h>
#include <qtextstream.h>

#include "dms.h"
#include "skypoint.h"
#include "kstarsdata.h"
#include "klocale.h"

KStarsData::KStarsData() {
	stdDirs = new KStandardDirs();
	options = new KStarsOptions();
	locale = new KLocale( "kstars" );
	oldOptions = 0;
	
	objList = new QList < SkyObject >;
	ObjNames = new QList < SkyObjectName >;

	geoList.setAutoDelete( TRUE );
	objList->setAutoDelete( FALSE );//sky objects in this list are deleted elsewhere
	starList.setAutoDelete( TRUE );
	messList.setAutoDelete( TRUE );
	ngcList.setAutoDelete( TRUE );
	icList.setAutoDelete( TRUE );
	clineList.setAutoDelete( TRUE );
	clineModeList.setAutoDelete( TRUE );
	cnameList.setAutoDelete( TRUE );
	Equator.setAutoDelete( TRUE );
	Horizon.setAutoDelete( TRUE );
}

KStarsData::~KStarsData() {
	if ( 0 != oldOptions ) {
  	delete oldOptions;
		oldOptions = 0;
	}
	if ( 0 != options ) {
		delete options;
		options = 0;
	}
	// the list items do not need to be removed by hand.
	// all lists are set to AutoDelete = true
	delete ObjNames;
	delete stdDirs;
	delete Sun;
	delete Mercury;
	delete Venus;
	delete Mars;
	delete Jupiter;
	delete Saturn;
	delete Uranus;
	delete Neptune;
}

//Functions for reading data files
bool KStarsData::openDataFile( QFile &file, QString s, bool doWarn, bool required ) {
	bool result;
	KStandardDirs *stdDirs = new KStandardDirs();
	QString ResName = "kstars/" + s;
	QString FileName= stdDirs->findResource( "data", ResName );
	
	if ( FileName != QString::null ) {
		file.setName( FileName );
		if ( !file.open( IO_ReadOnly ) ) {
			delete stdDirs;
			result = false;
		}
		delete stdDirs;
		result = true;
  } else { //File not found, try the local option
		file.setName( "data/" + s );

		if (!file.open( IO_ReadOnly ) ) { //File not found.  return false.
			delete stdDirs;
			result = false;
		}	else {
			delete stdDirs;
			result = true;	
		}									
  }

/*
	if ( result==false && doWarn ) {
		QString caption;
		QString message;

		if (required) {
			message = i18n( "The file " ) + s + i18n( " could not be found." )+ "\n" +
				i18n( "Shutting down, because KStars cannot run properly without this file.") + "\n" +
				i18n( "Place it in one of the following locations, then try again:" ) + "\n\n" +
				(QString)"\t$(KDEDIR)/share/apps/kstars/" + s +
				(QString)"\n\t~/.kde/share/apps/kstars/" + s +
				(QString)"\n\t$(PATH_TO_KSTARS)/data/" + s;
			caption = i18n( "Critical file not found: " ) + s;
		} else {
			message = i18n( "The file " ) + s + i18n( " could not be found." )+ "\n" +
				i18n( "KStars can still run without this file, so I will continue.") + "\n" +
				i18n( "However, to avoid seeing this message in the future, you may want to " ) + "\n" +
				i18n( "place the file in one of the following locations:" ) + "\n\n" +
				(QString)"\t$(KDEDIR)/share/apps/kstars/" + s +
				(QString)"\n\t~/.kde/share/apps/kstars/" + s +
				(QString)"\n\t$(PATH_TO_KSTARS)/data/" + s;
			caption = i18n( "Non-critical file not found: " ) + s;
		}

		KMessageBox::information( 0, message, caption );
	}
*/
	return result;
}

bool KStarsData::readCLineData( void ) {
	QFile file;
	if ( openDataFile( file, "clines.dat", false ) ) {
	  QTextStream stream( &file );

  	while ( !stream.eof() ) {
	  	QString line, name;
		  int rah, ram, ras, dd, dm, ds;
		  QChar sgn;
			QChar *mode;

		  line = stream.readLine();

		  rah = line.mid( 0, 2 ).toInt();
	  	ram = line.mid( 2, 2 ).toInt();
		  ras = line.mid( 4, 2 ).toInt();

		  sgn = line.at( 6 );
	  	dd = line.mid( 7, 2 ).toInt();
		  dm = line.mid( 9, 2 ).toInt();
		  ds = line.mid( 11, 2 ).toInt();
	    mode = new QChar( line.at( 13 ) );
		  dms r; r.setH( rah, ram, ras );
	  	dms d( dd, dm,  ds );

		  if ( sgn == "-" ) { d.setD( -1.0*d.getD() ); }

		  SkyPoint *o = new SkyPoint( r.getH(), d.getD() );
		  clineList.append( o );
			clineModeList.append( mode );
  	}
		file.close();

		return true;
	} else {
		return false;
	}
}

bool KStarsData::readCNameData( void ) {
  QFile file;
	cnameFile = "cnames.dat";
	if ( locale->language()=="fr" ) cnameFile = "fr_cnames.dat";

	if ( openDataFile( file, cnameFile, false ) ) {
	  QTextStream stream( &file );

  	while ( !stream.eof() ) {
	  	QString line, name;
		  int rah, ram, ras, dd, dm, ds;
		  QChar sgn;

	  	line = stream.readLine();

		  rah = line.mid( 0, 2 ).toInt();
		  ram = line.mid( 2, 2 ).toInt();
	  	ras = line.mid( 4, 2 ).toInt();

		  sgn = line.at( 6 );
		  dd = line.mid( 7, 2 ).toInt();
	  	dm = line.mid( 9, 2 ).toInt();
		  ds = line.mid( 11, 2 ).toInt();

		  name  = line.mid( 13 ).stripWhiteSpace();
	
		  dms r; r.setH( rah, ram, ras );
		  dms d( dd, dm,  ds );

	  	if ( sgn == "-" ) { d.setD( -1.0*d.getD() ); }

		  SkyObject *o = new SkyObject( -1, r, d, 0.0, name );
	  	cnameList.append( o );
			objList->append( o );
	  		ObjNames->append (new SkyObjectName (o->name, objList->last()));
	  }
		file.close();

		return true;
	} else {
		return false;
	}
}

bool KStarsData::readStarData( void ) {
	QFile file;
	if ( openDataFile( file, "sao.dat", false ) ) {
	  QTextStream stream( &file );

	  while ( !stream.eof() ) {
		  QString line, name;
	  	float mag;
		  int rah, ram, ras, dd, dm, ds;
		  QChar sgn, SpType;

	  	line = stream.readLine();

			name = "star";
		  rah = line.mid( 0, 2 ).toInt();
	  	ram = line.mid( 2, 2 ).toInt();
			ras = int( line.mid( 4, 6 ).toDouble() + 0.5 ); //add 0.5 to make int() pick nearest integer

		  sgn = line.at( 17 );
	  	dd = line.mid( 18, 2 ).toInt();
		  dm = line.mid( 20, 2 ).toInt();
		  ds = int( line.mid( 22, 5 ).toDouble() + 0.5 ); //add 0.5 to make int() pick nearest integer

	  	mag = line.mid( 33, 4 ).toFloat();
	  	SpType = line.at( 37 );
			name = line.mid( 38, 20 ).stripWhiteSpace();
			if ( name.isEmpty() ) name = "star";

		  dms r;
		  r.setH( rah, ram, ras );
		  dms d( dd, dm,  ds );
	
		  if ( sgn == "-" ) { d.setD( -1.0*d.getD() ); }

		  SkyObject *o = new SkyObject( 0, r, d, mag, name );
	  	starList.append( o );
			
			if ( o->name != "star" ) {
				objList->append( o );
	  			ObjNames->append (new SkyObjectName (o->name, objList->last()));
			}
						
			if (mag < 4.0) starCount0 = starList.count();
  	  if (mag < 6.0) starCount1 = starList.count();
			if (mag < 7.0) starCount2 = starList.count();
			if (mag < 7.5) starCount3 = starList.count();
  	}
		file.close();

		return true;
	} else {
		return false;
	}
}

bool KStarsData::readNGCData( void ) {
	QFile file;
	if ( openDataFile( file, "ngc2000.dat", false ) ) {
	  QTextStream stream( &file );

  	while ( !stream.eof() ) {
	  	QString line, smag, name, name2, longname;
		  float mag, ramf;
		  int type, ingc, imess, rah, ram, ras, dd, dm;
	  	QChar sgn, cat_id, cat2_id, mflag;

		  line = stream.readLine();

		  cat_id = line.at( 0 ); //check for IC catalog flag
	  	if ( cat_id == " " ) cat_id = 'N'; // if blank, set catalog to NGC
	
		  ingc = line.mid( 1, 4 ).toInt();  // NGC/IC catalog number
			type = line.mid( 6, 1 ).toInt();     // object type identifier
	  	rah = line.mid( 8, 2 ).toInt();
		  ramf = line.mid( 11, 4 ).toFloat();
		  sgn = line.at( 16 );
	  	dd = line.mid( 17, 2 ).toInt();
		  dm = line.mid( 20, 2 ).toInt();
		  smag = line.mid( 23, 4 );
		  if (smag == "    " ) { mag = 99.9; } else { mag = smag.toFloat(); }
			mflag = line.at( 28 );
			if ( mflag=='M' ) {
				cat2_id = cat_id;
				if ( ingc==0 ) cat2_id = ' ';
				cat_id = mflag;
				imess = line.mid( 30, 3 ).toInt();
			}
			longname = line.mid( 34, 30 ).stripWhiteSpace();
		
		  ram = int( ramf );
		  ras = int( 60.0*( ramf - (float)ram ) );
	
	  	dms r; r.setH( rah, ram, ras );
		  dms d( dd, dm, 0 );

		  if ( sgn == "-" ) { d.setD( -1.0*d.getD() ); }
	
			QString snum;
			if (cat_id=="I") {
				snum.setNum( ingc );
				name = "IC " + snum;
			} else if (cat_id=="N") {
				snum.setNum( ingc );
				name = "NGC " + snum;
			} else if (cat_id=="M") {
				snum.setNum( imess );
				name = "M " + snum;
				if (cat2_id=="N") {
					snum.setNum( ingc );
					name2 = "NGC " + snum;
				} else if (cat2_id=="I") {
					snum.setNum( ingc );
					name2 = "IC " + snum;
				} else {
					name2 = "";
				}
			}
			
	  	SkyObject *o = new SkyObject( type, r, d, mag, name, name2, longname );

	  	if ( !o->longname.isEmpty() ) {
	  		objList->append( o );
  			ObjNames->append (new SkyObjectName (o->longname, objList->last()));
	  	}
	  			
		  if (cat_id=="M") {
	  		messList.append( o );
				objList->append( o );
	  		ObjNames->append (new SkyObjectName (o->name, objList->last()));
		  } else if ( cat_id=="N" ) {
			  ngcList.append( o );
				objList->append( o );
	  		ObjNames->append (new SkyObjectName (o->name, objList->last()));
			} else if ( cat_id=="I" ) {
				icList.append( o );
				objList->append( o );
	  		ObjNames->append (new SkyObjectName (o->name, objList->last()));
			}  	
	  }
		file.close();

		return true;
	} else {
		return false;
	}
}

bool KStarsData::readURLData( QString urlfile ) {
	QFile file;
	if ( openDataFile( file, urlfile, true, false ) ) {
	  QTextStream stream( &file );

  	while ( !stream.eof() ) {
	  	QString line = stream.readLine();
			QString name = line.mid( 0, line.find(':') );
			QString url = line.mid( line.find(':')+1 );
		
			if ( urlfile == "hsturl.dat" ) {
				switch( name.at(0).unicode() ) {
					case 'M' : //Messier object
						for ( unsigned int i=0; i<messList.count(); ++i ) {
							SkyObject *o = messList.at(i);
							if ( o->name == name ) {
								o->hstURL = url;
								break;
							} 	
						}
						break;
					case 'N' : //NGC object
						for ( unsigned int i=0; i<ngcList.count(); ++i ) {
							SkyObject *o = ngcList.at(i);
							if ( o->name == name ) {
								o->hstURL = url;
								break;
							}
						}
						break;
					case 'I' : //IC object
						for ( unsigned int i=0; i<icList.count(); ++i ) {
							SkyObject *o = icList.at(i);
							if ( o->name == name ) {
								o->hstURL = url;
								break;
							}
						} 	
						break;
				}
			} else if ( urlfile == "messurl.dat" ) {
				for ( unsigned int i=0; i<messList.count(); ++i ) {
					SkyObject *o = messList.at(i);
					if ( o->name == name ) {
						o->infoURL = url;
						break;
					} 	
				}
			} else if ( urlfile == "planeturl.dat" ) {
      	if ( name == "Sun" ) {
					Sun->infoURL = url;
				} else if ( name == "Moon" ) {
					Moon->infoURL = url;
				} else if ( name == "Mercury" ) {
					Mercury->infoURL = url;
				} else if ( name == "Venus" ) {
					Venus->infoURL = url;
				} else if ( name == "Mars" ) {
					Mars->infoURL = url;
				} else if ( name == "Jupiter" ) {
					Jupiter->infoURL = url;
				} else if ( name == "Saturn" ) {
					Saturn->infoURL = url;
				} else if ( name == "Uranus" ) {
					Uranus->infoURL = url;
				} else if ( name == "Neptune" ) {
					Neptune->infoURL = url;
				}
    	}	
		}
		file.close();

		return true;
	} else {
		return false;
	}
}

bool KStarsData::readCityData( void ) {
	QFile file;
	if ( openDataFile( file, "Cities.dat", false ) ) {
		QTextStream stream( &file );

  	while ( !stream.eof() ) {
	  	QString line, name, state;
			char latsgn, lngsgn;
			int lngD, lngM, lngS;
	  	int latD, latM, latS;
			int TZ;
		  float lng, lat;

		 	line = stream.readLine();

		  name  = line.left( 32 );
	  	state = line.mid( 31, 17 );
		 	latD = line.mid( 49, 2 ).toInt();
		  latM = line.mid( 52, 2 ).toInt();
	  	latS = line.mid( 55, 2 ).toInt();
		 	latsgn = line.at( 58 ).latin1();
		  lngD = line.mid( 60, 3 ).toInt();
	  	lngM = line.mid( 64, 2 ).toInt();
		 	lngS = line.mid( 67, 2 ).toInt();
		  lngsgn = line.at( 70 ).latin1();

	  	lat = (float)latD + ((float)latM + (float)latS/60.0)/60.0;
			lng = (float)lngD + ((float)lngM + (float)lngS/60.0)/60.0;
  	
		 	if ( latsgn == 'S' ) lat *= -1.0;
			if ( lngsgn == 'W' ) lng *= -1.0;
  	
		  TZ = int(lng/-15.0);

			name = name.stripWhiteSpace();
			name = name.left(1).upper() + name.mid(1).lower();
			state = state.stripWhiteSpace();
			state = state.left(1).upper() + state.mid(1).lower();

			geoList.append (new GeoLocation( lng, lat, name, state, TZ ));
   	}
		file.close();

		return true;
	} else {
		return false;
	}
}

// Save and restore options
void KStarsData::saveOptions()
{
	if ( 0 == oldOptions) {
		oldOptions = new KStarsOptions();
	}
	oldOptions->copy( options );
}
	
void KStarsData::restoreOptions()
{
	if ( 0 == oldOptions ) {
		// this should not happen
		return;
	}
	options->copy( oldOptions );
}

long double KStarsData::getJD( QDateTime t ) {
  int year = t.date().year();
  int month = t.date().month();
  int day = t.date().day();
  int hour = t.time().hour();
  int minute = t.time().minute();
  double second = t.time().second() + 0.001*t.time().msec();
  int m, y, A, B, C, D;

  if (month > 2) {
	  m = month;
	  y = year;
  } else {
	  y = year - 1;
	  m = month + 12;
  }

 /*  If the date is after 10/15/1582, then take Pope Gregory's modification
	 to the Julian calendar into account */

	 if (( year  >1582 ) ||
		 ( year ==1582 && month  >9 ) ||
		 ( year ==1582 && month ==9 && day >15 ))
	 {
		 A = int(y/100);
		 B = 2 - A + int(A/4);
	 } else {
		 B = 0;
	 }

  if (y < 0) {
	  C = int((365.25*y) - 0.75);
  } else {
	  C = int(365.25*y);
  }

  D = int(30.6001*(m+1));

  long double d = double(day) + (double(hour) + (double(minute) + second/60.0)/60.0)/24.0;
  long double jd = B + C + D + d + 1720994.5;

  return jd;
}
