/***************************************************************************
                          ksplanet.cpp  -  K Desktop Planetarium
                             -------------------
    begin                : Sun Jul 22 2001
    copyright            : (C) 2001 by Jason Harris
    email                : jharris@30doradus.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qfile.h>
#include <qtextstream.h>
#include <math.h>
#include "kstarsdata.h"
#include "ksplanet.h"

KSPlanet::KSPlanet(){
	KSPlanet( "Unnamed" );
}

KSPlanet::KSPlanet( QString s, QImage im )
 : SkyObject( 2, 0.0, 0.0, 0.0, s, "", "", "" ) {
	EcLong.setD( 0.0 );
	EcLat.setD( 0.0 );
	image = im;
}

bool KSPlanet::findPosition( long double jd ) {
	QString fname, snum, line;
	QFile f;
	double sum[6], A, B, C, T, RE;
	double x, y, z;
	double sinL, sinL0, sinB, sinB0;
	double cosL, cosL0, cosB, cosB0;
	int nCount = 0;

	dms EcLong0, EcLat0; //heliocentric coords
	dms EarthLong, EarthLat; //heliocentric coords of Earth
	
	T = (jd - J2000)/365250.0; //Julian millenia since J2000

	//First, find heliocentric coordinates
	//Ecliptic Longitude
	for (int i=0; i<6; ++i) {
		sum[i] = 0.0;
		snum.setNum( i );
		fname = name.lower() + ".L" + snum + ".vsop";
		if ( KStarsData::openDataFile( f, fname, false ) ) {
			++nCount;
		  QTextStream stream( &f );
  		while ( !stream.eof() ) {
				line = stream.readLine();
				QTextIStream instream( &line );
  			instream >> A >> B >> C;
				sum[i] += A * cos( B + C*T );
	  	}
  		f.close();
  	}
  }

	if ( nCount==0 ) return false; //No longitude data found!

  EcLong.setRadians( sum[0] + sum[1]*T + sum[2]*T*T + sum[3]*T*T*T + sum[4]*T*T*T*T + sum[5]*T*T*T*T*T );
	EcLong.setD( EcLong.reduce().getD() );
	  	
	nCount = 0;
	//Compute Ecliptic Latitude
	for (int i=0; i<6; ++i) {
		sum[i] = 0.0;
		snum.setNum( i );
		fname = name.lower() + ".B" + snum + ".vsop";
		if ( KStarsData::openDataFile( f, fname, false ) ) {
			++nCount;
		  QTextStream stream( &f );
  		while ( !stream.eof() ) {
				line = stream.readLine();
				QTextIStream instream( &line );
  			instream >> A >> B >> C;
				sum[i] += A * cos( B + C*T );
	  	}
  		f.close();
  	}
  }

	if (nCount==0) return false; //no latitude data found!

  EcLat.setRadians( sum[0] + sum[1]*T + sum[2]*T*T + sum[3]*T*T*T + sum[4]*T*T*T*T + sum[5]*T*T*T*T*T );
  	
	//Compute Heliocentric Distance
	nCount = 0;
	for (int i=0; i<6; ++i) {
		sum[i] = 0.0;
		snum.setNum( i );
		fname = name.lower() + ".R" + snum + ".vsop";
		if ( KStarsData::openDataFile( f, fname, false ) ) {
			++nCount;
		  QTextStream stream( &f );
  		while ( !stream.eof() ) {
				line = stream.readLine();
				QTextIStream instream( &line );
  			instream >> A >> B >> C;
				sum[i] += A * cos( B + C*T );
	  	}
  		f.close();
  	}
  }

	if (nCount==0) return false; //no distance data found!

  Rsun = sum[0] + sum[1]*T + sum[2]*T*T + sum[3]*T*T*T + sum[4]*T*T*T*T + sum[5]*T*T*T*T*T;

  //Find Earth's heliocentric coords
	//Ecliptic Longitude
	nCount = 0;
	for (int i=0; i<6; ++i) {
		sum[i] = 0.0;
		snum.setNum( i );
		fname = "earth.L" + snum + ".vsop";
		if ( KStarsData::openDataFile( f, fname, false ) ) {
			++nCount;
		  QTextStream stream( &f );
  		while ( !stream.eof() ) {
				line = stream.readLine();
				QTextIStream instream( &line );
  			instream >> A >> B >> C;
				line = line.sprintf( "%f", C );
				sum[i] += A * cos( B + C*T );
	  	}
  		f.close();
  	}
  }

	if (nCount==0) return false; //no longitude data found!

  EarthLong.setRadians( sum[0] + sum[1]*T + sum[2]*T*T + sum[3]*T*T*T + sum[4]*T*T*T*T + sum[5]*T*T*T*T*T );
	EarthLong.setD( EarthLong.reduce().getD() );
	  	
	//Ecliptic Latitude
	nCount = 0;
	for (int i=0; i<6; ++i) {
		sum[i] = 0.0;
		snum.setNum( i );
		fname = "earth.B" + snum + ".vsop";
		if ( KStarsData::openDataFile( f, fname, false ) ) {
			++nCount;
		  QTextStream stream( &f );
  		while ( !stream.eof() ) {
				line = stream.readLine();
				QTextIStream instream( &line );
  			instream >> A >> B >> C;
				sum[i] += A * cos( B + C*T );
	  	}
  		f.close();
  	}
  }

	if (nCount==0) return false; //no latitude data found!

  EarthLat.setRadians( sum[0] + sum[1]*T + sum[2]*T*T + sum[3]*T*T*T + sum[4]*T*T*T*T + sum[5]*T*T*T*T*T );
  	
	//Compute Heliocentric Distance
	nCount = 0;
	for (int i=0; i<6; ++i) {
		sum[i] = 0.0;
		snum.setNum( i );
		fname = "earth.R" + snum + ".vsop";
		if ( KStarsData::openDataFile( f, fname, false ) ) {
			++nCount;
		  QTextStream stream( &f );
  		while ( !stream.eof() ) {
				line = stream.readLine();
				QTextIStream instream( &line );
  			instream >> A >> B >> C;
				sum[i] += A * cos( B + C*T );
	  	}
  		f.close();
  	}
  }

	if (nCount==0) return false; //no distance data found!

  RE = sum[0] + sum[1]*T + sum[2]*T*T + sum[3]*T*T*T + sum[4]*T*T*T*T + sum[5]*T*T*T*T*T;

	//find geometric geocentric coordinates
	EcLong.SinCos( sinL, cosL );
	EcLat.SinCos( sinB, cosB );
	EarthLong.SinCos( sinL0, cosL0 );
	EarthLat.SinCos( sinB0, cosB0 );
	x = Rsun*cosB*cosL - RE*cosB0*cosL0;
	y = Rsun*cosB*sinL - RE*cosB0*sinL0;
	z = Rsun*sinB - RE*sinB0;
	
	EcLong.setRadians( atan( y/x ) );
	if (x<0) EcLong.setD( EcLong.getD() + 180.0 ); //resolve atan ambiguity
	EcLat.setRadians( atan( z/( sqrt( x*x + y*y ) ) ) );
	
  EclipticToEquatorial( jd );

	return true;
}

KSPlanet::~KSPlanet(){
}

void KSPlanet::EclipticToEquatorial( long double jd ) {
	//compute obliquity of ecliptic for date jd
	double T = (jd - 2451545.0)/36525.0; //number of centuries since Jan 1, 2000
	double DeltaObliq = 46.815*T + 0.0006*T*T - 0.00181*T*T*T; //change in Obliquity, in arcseconds
	dms Obliquity;
	Obliquity.setD( 23.439292 - DeltaObliq/3600.0 );
	
	double sinLong, cosLong, sinLat, cosLat, sinObliq, cosObliq;
	EcLong.SinCos( sinLong, cosLong );
	EcLat.SinCos( sinLat, cosLat );
	Obliquity.SinCos( sinObliq, cosObliq );
	
	double sinDec = sinLat*cosObliq + cosLat*sinObliq*sinLong;
	
	double y = sinLong*cosObliq - (sinLat/cosLat)*sinObliq;
	double RARad =  atan( y / cosLong );
	
	//resolve ambiguity of atan:
	if ( cosLong < 0 ) RARad += PI();
	if ( cosLong > 0 && y < 0 ) RARad += 2.0*PI();
	
	dms newRA, newDec;
	newRA.setRadians( RARad );
	newDec.setRadians( asin( sinDec ) );
	pos()->setRA( newRA );
	pos()->setDec( newDec );
}

