/***************************************************************************
                          timer.cpp  -  Counts down to selected date
                             -------------------
    begin                : Sat Feb 16 2002
    copyright            : (C) 2002 by Erik Johansson
    email                : erre@telia.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <qdatetime.h>

#include "timer.h"
#include "global.h"

Timer::Timer( const QDateTime *finish, short int format = 15, short int round = 0,
						  const QDateTime *halfWay = 0, short int halfFormat = -1, short int halfRound = 0)
 : useHalfWay(false), timeLeft(true), stringFormat(format), roundOff( round ), halfRoundOff (halfRound )
{
	targetDate = new QDateTime( *finish );
	oldDate = new QDateTime();
	tempDate = new QDateTime();

	if ( stringFormat < 0 || stringFormat > 15 )
	{
		stringFormat = 15;
		returnTime.error = true;
	}

	if ( halfWay != 0 )
	{
		halfDate = new QDateTime( *halfWay );
		useHalfWay = true;
		if ( halfFormat == -1 )
			halfStringFormat = stringFormat;
		else
		{
			if ( halfFormat < 0 || halfFormat > 15 )
			{
				halfStringFormat = 15;
				returnTime.error = true;
			} else {
				halfStringFormat =  halfFormat;
			}
		}
	}

	returnTime.setValue( 0 );
	time.setValue( 0, false );
}

Timer::~Timer()
{
	delete targetDate;
	delete halfDate;
	delete oldDate;
	delete tempDate;
}

int Timer::countdown()
{
	// Make sure we use the same currentDateTime through the whole func
	*tempDate = QDateTime::currentDateTime();

	// Only calc if time has changed and there is time left
	if ( (*tempDate != *oldDate) && timeLeft)
	{
		// Opps, no time left...
		if ( *targetDate <= *tempDate )
		{
			timeLeft = false;
			emit changeSettings(2);
			return 2; // No more time left
		}

		// Secs to finish
		unsigned long secs = tempDate->secsTo( *targetDate );	

		if ( useHalfWay )
		{
			if ( *halfDate <= *tempDate )			
			{
				useHalfWay = false;
				time.half = true;
				emit changeSettings(1);
			}
		}

		// Whole days left
		time.days = (secs - (secs % (24 * 3600))) / (24 * 3600);
		secs = secs - (time.days * 24 * 3600);

		// Whole hours, not counting days
		time.hours = (secs - (secs % 3600)) / 3600;
		secs = secs - (time.hours * 3600);

		// Whole mins, not counting hours
		time.mins = (secs - (secs % 60)) / 60;

		// Whole secs, not counting mins
		time.secs = secs - (time.mins * 60);

		*oldDate = *tempDate;
		return 1;  // Time has changed
	}
	return 0;  // Nothing has changed
}

STimeLeft Timer::getTime()
{
	if ( countdown() == 0 )   // Nothing has changed, return old values
		return returnTime;	
	else
	{
		short int format = stringFormat;
		short int round = roundOff;
		if ( time.half )
		{
			format = halfStringFormat;
			round = roundOff;
		}

		// Init to 0 and calc it according to current round off method
		returnTime.setValue( 0 );
		switch ( round )
		{
		case 0:
			countNoRound( format );
			break;
		case 1:
			countUpRound ( format );
			break;
		case 2:
			countMathRound ( format );
			break;
		}
	}
	return returnTime;
}

/***
*	No round off == round down
* Don't need to do any thing special, it is already done
***/
void Timer::countNoRound( short int format )
{
	if ( format >= 8 ) // Calc days
	{
		returnTime.days = time.days;
		format -= 8;
	}
	else
		time.hours = time.days * 24 + time.hours;
	
	if ( format >= 4 ) // Calc hours
	{
		returnTime.hours = time.hours;
		format -= 4;
	}
	else
		time.mins = time.hours * 60 + time.mins;

	if ( format >= 2 ) // Calc mins
	{
		returnTime.mins = time.mins;
		format -= 2;
	}
	else
		time.secs = time.mins * 60 + time.secs;

	if ( format == 1 ) // Calc secs
		returnTime.secs = time.secs;
}
/***
*	Round up
*	Add one to the lowest time as long as there is some
***/
void Timer::countUpRound( short int format )
{
	if ( format == 8 ) // Only days
	{
		returnTime.days = time.days;
		if ( time.hours > 0 || time.mins > 0 || time.secs > 0 )
			returnTime.days += 1;
		format -= 8;
	}
	else if ( format > 8 ) // Calc days
	{
		returnTime.days = time.days;
		format -= 8;
	}
	else
		time.hours = time.days * 24 + time.hours;

	if ( format == 4 )
	{
		returnTime.hours = time.hours;
		if ( time.mins > 0 || time.secs > 0 )
			returnTime.hours += 1;
		format -= 4;
	}
	else if ( format > 4 ) // Calc hours
	{
		returnTime.hours = time.hours;
		format -= 4;
	}
	else
		time.mins = time.hours * 60 + time.mins;

	if ( format == 2 )
	{
		returnTime.mins = time.mins;
		if ( time.secs > 0 )
			returnTime.mins += 1;
		format -= 2;
	}
	else if ( format > 2 ) // Calc mins
	{
		returnTime.mins = time.mins;
		format -= 2;
	}
	else
		time.secs = time.mins * 60 + time.secs;

	if ( format == 1 ) // Calc secs
		returnTime.secs = time.secs;
}
/***
*	Math Round off
* Add one to the lowest if there is something below and it is >= MAX/2 and < MAX
*	if it's not, round it off down == do nothing
***/
void Timer::countMathRound(  short int format )
{
	if ( format == 8 ) // Only days
	{
		returnTime.days = time.days;
		if ( 	( time.hours >= 12 && time.hours < 24 ) ||
				( time.mins >= 30 && time.mins < 60 ) ||
				( time.secs >= 30 && time.secs < 60 ) )
			returnTime.days += 1;
		format -= 8;
	}
	else if ( format > 8 ) // Calc days
	{
		returnTime.days = time.days;
		format -= 8;
	}
	else
		time.hours = time.days * 24 + time.hours;

	if ( format == 4 )
	{
		returnTime.hours = time.hours;
		if ( ( time.mins >= 30 && time.mins < 60 ) || ( time.secs >= 30 && time.secs < 60 ) )
			returnTime.hours += 1;
		format -= 4;
	}
	else if ( format > 4 ) // Calc hours
	{
		returnTime.hours = time.hours;
		format -= 4;
	}
	else
		time.mins = time.hours * 60 + time.mins;

	if ( format == 2 )
	{
		returnTime.mins = time.mins;
		if ( ( time.secs >= 30 && time.secs < 60 ) )
			returnTime.mins += 1;
		format -= 2;
	}
	else if ( format > 2 ) // Calc mins
	{
		returnTime.mins = time.mins;
		format -= 2;
	}
	else
		time.secs = time.mins * 60 + time.secs;

	if ( format == 1 ) // Calc secs
		returnTime.secs = time.secs;
}


