/*  This file is part of the KDE project
    Copyright (C) 2001 Simon Hausmann <hausmann@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception this program may be linked with Qt non-commercial 
    edition, the resulting executable be distributed, without including the 
    source code for the Qt non-commercial edition in the source distribution.

*/


#include "htmlview.h"

#include <qtimer.h>

#include <kglobal.h>
#include <kio/jobclasses.h>
#include <kdatastream.h>

#include <khtmlview.h>

// ###
#include <xml/dom_docimpl.h>
#include <rendering/render_object.h>
#include <misc/loader.h>

#include <assert.h>

HTMLView::HTMLView( QWidget *parentWidget, const char *widgetName,
                    QObject *parent, const char *name )
    : KHTMLPart( parentWidget, widgetName, parent, name )
{
    m_imageMode = false;
}

bool HTMLView::openURL( const KURL &url )
{
    static const QString &html = KGlobal::staticQString( "<html><body><img src=\"%1\"></body></html>" );

    QString serviceType = browserExtension()->urlArgs().serviceType;
    if ( !serviceType.isEmpty() )
        m_imageMode = serviceType == "image/*";

    if ( !m_imageMode )
        return KHTMLPart::openURL( url );

    m_url = url;

    KParts::URLArgs args = browserExtension()->urlArgs();

    begin( m_url, args.xOffset, args.yOffset );

    DOM::DocumentImpl *impl = static_cast<DOM::DocumentImpl *>( document().handle() ); // ### hack ;-)
    if ( impl && browserExtension()->urlArgs().reload )
        impl->docLoader()->setReloading( true );

    write( html.arg( m_url.url() ) );
    end();

    KIO::Job *job = khtml::Cache::loader()->jobForRequest( m_url.url() );

    if ( job )
    {
        emit started( job );

        connect( job, SIGNAL( result( KIO::Job * ) ),
                 this, SLOT( slotImageJobFinished( KIO::Job * ) ) );
    }
    else
    {
        emit started( 0 );
        emit completed();
    }

    return true;
}

void HTMLView::saveState( QDataStream &stream )
{
    stream << m_imageMode;
    KHTMLPart::saveState( stream );
}

void HTMLView::restoreState( QDataStream &stream )
{
    stream >> m_imageMode;
    KHTMLPart::restoreState( stream );
}

void HTMLView::slotImageJobFinished( KIO::Job *job )
{
    assert( m_imageMode );

    if ( job->error() )
    {
        job->showErrorDialog();
        emit canceled( job->errorString() );
    }
    else
    {
        if ( view()->contentsY() == 0 )
        {
            KParts::URLArgs args = browserExtension()->urlArgs();
            view()->setContentsPos( args.xOffset, args.yOffset );
        }

        emit completed();
    }
}

KParts::ReadOnlyPart *HTMLView::createPart( QWidget *parentWidget, const char *widgetName,
                                            QObject *parent, const char *name,
                                            const QString &mimetype, QString &serviceName,
                                            QStringList &serviceTypes, const QStringList &/*params*/ )
{
    assert( !m_imageMode );

    if ( mimetype.isEmpty() )
        return 0;

    HTMLView *res = new HTMLView( parentWidget, widgetName, parent, name );

    serviceTypes = QStringList() << mimetype;
    serviceName = QString::fromLatin1( "HTMLView" );

    return res;
}

#include "htmlview.moc"

