/*  This file is part of the KDE project
    Copyright (C) 2000,2001 Simon Hausmann <hausmann@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception this program may be linked with Qt non-commercial
    edition, the resulting executable be distributed, without including the
    source code for the Qt non-commercial edition in the source distribution.

*/

// !! The history code is (c) David Faure <faure@kde.org> !!

#include <config.h>

#include <klocale.h>
#include <kstddirs.h>

#include <qpopupmenu.h>
#include <qaction.h>
#include <qmenubar.h>
#include <qstatusbar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qlineedit.h>
#include <qtimer.h>
#include <qwidgetstack.h>
#include <qcombobox.h>
#include <qfile.h>

#ifdef _QT_QPE_
#include <qpe/global.h>
#endif

#include <kconfig.h>
#include <kmessagebox.h>
#include <kcharsets.h>
#include <dcopclient.h>

#include <assert.h>

#include <khtml_ext.h>
#include <khtml_factory.h>
#include <khtml_settings.h>

#include "bookmarks.h"
#include "xmltree.h"

#include "mainwindowbase.h"
#include "view.h"
#include "htmlview.h"
#include "preferencesimpl.h"
#include "defaults.h"


#include "pics/stop.xpm"
#include "pics/lock.xpm"
#include "pics/unlock.xpm"
#include "pics/back.xpm"
#include "pics/forward.xpm"
#include "pics/gohome.xpm"
#include "pics/reload.xpm"
#include "pics/close.xpm"
#include "pics/windows.xpm"
#include "pics/menu.xpm"
#include "pics/images_display.xpm"
#include "pics/viewmag.xpm"

KURL *MainWindowBase::s_homeURL = 0;

MainWindowBase::MainWindowBase( QWidget *parent, const char *name, WFlags flags )
    : QMainWindow( parent, name, flags )
{
}

void MainWindowBase::init()
{
    m_widgetStack = new QWidgetStack( this, "main widget stack" );
    m_widgetId = 0;

    m_mainMenu = new QPopupMenu( this );

    m_windowList = new QPopupMenu( this );
    connect( m_windowList, SIGNAL( aboutToShow() ),
             this, SLOT( prepareWindowListMenu() ) );
    connect( m_windowList, SIGNAL( activated( int ) ),
             this, SLOT( activateWindowFromPopup( int ) ) );

    setupZoomMenu();

    setCentralWidget( m_widgetStack );

    setToolBarsMovable( false );

    m_statusBarAutoHideTimer = new QTimer( this, "statusBarAutoHideTimer" );
    connect( m_statusBarAutoHideTimer, SIGNAL( timeout() ),
             statusBar(), SLOT( hide() ) );

    statusBar()->hide();

    m_locationCombo = 0;

    m_bookmarks = new BookmarkManager( this );

    initActions();
    initGUI();

    if ( m_locationCombo )
        connect( m_locationCombo, SIGNAL( activated( const QString & ) ),
                 this, SLOT( urlEntered( const QString & ) ) );

    KConfig *cfg = KGlobal::config();
    cfg->setGroup( "General" );
    m_toggleLocationBar->setOn( cfg->readBoolEntry( "ShowLocationBar", true ) );

    // ### workaround
    (void)new KHTMLFactory( true /*clone*/ );
    KHTMLFactory::defaultHTMLSettings()->init();

    m_autoloadImages->setOn( KHTMLFactory::defaultHTMLSettings()->autoLoadImages() );
}

MainWindowBase::~MainWindowBase()
{
    QListIterator<View> it( m_views );
    for (; it.current(); ++it )
    {
        disconnect( it.current(), SIGNAL( destroyed() ),
                    this, SLOT( viewDestroyed() ) );
        if ( it.current()->document() )
            disconnect( it.current()->document()->widget(), SIGNAL( destroyed() ),
                        this, SLOT( viewWidgetDestroyed() ) );
    }

    KConfig *cfg = KGlobal::config();
    cfg->setGroup( "General" );
    cfg->writeEntry( "ShowLocationBar", m_toggleLocationBar->isOn() );
    cfg->sync();
}

void MainWindowBase::initActions()
{
    m_menu = new PopupAction( m_mainMenu, i18n( "Menu" ), loadPixmap( menu_xpm ),
                              QString::null, 0, this );

    m_close = new QAction( i18n( "Close View" ), QIconSet(), QString::null, CTRL + Key_W, this, "close" );
    connect( m_close, SIGNAL( activated() ),
             this, SLOT( closeView() ) );

    m_stop = new QAction( i18n( "Stop" ), loadPixmap( stop_xpm ), i18n( "Stop Loading" ), Key_Escape, this, "stop" );
    m_stop->setEnabled( false );

    m_back = new QAction( i18n( "Backward" ), loadPixmap( back_xpm ), QString::null, ALT+Key_Left, this, "back" );

    m_forward = new QAction( i18n( "Forward" ), loadPixmap( forward_xpm ), QString::null, ALT+Key_Right, this, "forward" );

    m_back->setEnabled( false );
    m_forward->setEnabled( false );

    m_home = new QAction( i18n( "Home" ), loadPixmap( gohome_xpm ), QString::null, CTRL+Key_Home, this );
    connect( m_home, SIGNAL( activated() ),
             this, SLOT( home() ) );

    m_reload = new QAction( i18n( "Reload" ), loadPixmap( reload_xpm ), QString::null, Key_F5, this );

    m_find = new QAction( i18n( "Find in Document" ), QIconSet(), QString::null, CTRL+Key_F, this );
    connect( m_find, SIGNAL( activated() ),
             this, SLOT( find() ) );

    m_closeFind = new QAction( i18n( "Close Find" ), loadPixmap( close_xpm ), QString::null, Key_Escape, this );
    connect( m_closeFind, SIGNAL( activated() ),
             this, SLOT( closeFind() ) );

    m_windows = new PopupAction( m_windowList, i18n( "Window List" ), loadPixmap( windows_xpm ),
                                 QString::null, 0, this );

    m_security = new QAction( i18n( "Show Security Information" ), loadPixmap( lock_xpm ),
                              QString::null, 0, this );

    m_newView = new QAction( i18n( "Open new View" ), QIconSet(), QString::null, 0, this );
    connect( m_newView, SIGNAL( activated() ),
             this, SLOT( newView() ) );

    m_preferences = new QAction( i18n( "Preferences..." ), QIconSet(), QString::null, 0, this );
    connect( m_preferences, SIGNAL( activated() ),
             this, SLOT( showPreferences() ) );

    m_toggleLocationBar = new QAction( i18n( "Show Location Bar" ), QIconSet(),
                                       QString::null, 0, this );
    m_toggleLocationBar->setToggleAction( true );
    connect( m_toggleLocationBar, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleLocationBar( bool ) ) );
    m_toggleLocationBar->setOn( true );

    m_loadImages = new QAction( i18n( "Load Images" ), loadPixmap( images_display_xpm ),
                                QString::null, 0, this );

    m_autoloadImages = new QAction( i18n( "Automatically Load Images" ), QIconSet(),
                                    QString::null, 0, this );
    m_autoloadImages->setToggleAction( true );
    connect( m_autoloadImages, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleAutoloadImages( bool ) ) ); 

    m_zoom = new PopupAction( m_zoomMenu, i18n( "Zoom" ), loadPixmap( viewmag_xpm ),
                              i18n( "Zoom Document" ), 0, this );

    /*
    m_encodingMenu = new QPopupMenu( this );

    m_encodingMenu->insertItem( QString::fromLatin1( "Auto" ) );

    connect( m_encodingMenu, SIGNAL( activated( const QString & ) ),
             this, SLOT( setEncoding( const QString & ) ) );

    QStringList encodings = KGlobal::charsets()->availableEncodingNames();

    QMap<QString, QStringList> encodingMapping; // languange -> encoding(s)
    QStringList languages;

    QStringList::ConstIterator it = encodings.begin();
    QStringList::ConstIterator end = encodings.end();
    for (; it != end; ++it )
    {
        QString lang = KGlobal::charsets()->languageForEncoding( *it );
        if ( !languages.contains( lang ) )
            languages.append( lang );

        QMap<QString, QStringList>::Iterator mapIt = encodingMapping.find( lang );
        if ( mapIt == encodingMapping.end() )
            mapIt = encodingMapping.insert( lang, QStringList() );

        mapIt.data().append( *it );
    }

    languages.sort();

    it = languages.begin();
    end = languages.end();
    for (; it != end; ++it )
    {
        QPopupMenu *menu = new QPopupMenu( m_encodingMenu );
        m_encodingMenu->insertItem( *it, menu );

        QMap<QString, QStringList>::ConstIterator mapIt = encodingMapping.find( *it );

        if ( mapIt == encodingMapping.end() )
            continue;

        QStringList encodings = mapIt.data();
        encodings.sort();
        QStringList::ConstIterator encodingIt = encodings.begin();
        QStringList::ConstIterator encodingEnd = encodings.end();
        for (; encodingIt != encodingEnd; ++encodingIt )
            menu->insertItem( *encodingIt );
    }

    m_encoding = new PopupAction( m_encodingMenu, QString::null, QIconSet(),
                                  i18n( "Set &Encoding" ), 0, this );
    */
}

QIconSet MainWindowBase::loadPixmap( const char *xpm[] )
{
    return QIconSet( QPixmap( xpm ) );
}

void MainWindowBase::urlEntered( const QString &_text )
{
    assert( m_locationCombo );
    QString text = _text;
    KURL u = text;

    if ( u.isMalformed() && QRegExp( "^www" ).match( text ) == 0 )
        u = KURL( text.prepend( "http://" ) );

    if ( !u.isMalformed() )
    {
        assert( m_currentView );
        m_currentView->openURL( u );
    }
    else
        KMessageBox::error( this, i18n( "Malformed URL\n%1" ).arg( u.prettyURL() ) );
}

void MainWindowBase::statusMessage( const QString &_msg )
{
    QString msg = _msg;
    msg.replace( QRegExp( "</?b>" ), QString::null );
    msg.replace( QRegExp( "<img.*>" ), QString::null );

#if defined(_QT_QPE_)
    Global::statusMessage( msg );
#else
    statusBar()->show();
    statusBar()->message( msg );

    m_statusBarAutoHideTimer->start( 5000, true );
#endif
}

void MainWindowBase::home()
{
    assert( m_currentView );
    m_currentView->openURL( homeURL() );
}

void MainWindowBase::find()
{
    assert( m_currentView );

    if (m_searchBar)
        m_searchBar->show();

    m_currentView->setFindEnabled( true );
    m_currentView->find();
}

void MainWindowBase::closeFind()
{
    assert( m_currentView );

    if (m_searchBar)
        m_searchBar->hide();
    m_currentView->setFindEnabled( false );
}

KURL MainWindowBase::homeURL()
{
    if ( !s_homeURL )
    {
        s_homeURL = new KURL;

        KConfig *cfg = KGlobal::config();
        cfg->setGroup( "General" );
        *s_homeURL = cfg->readEntry( "HomeURL", DEFAULT_HOMEPAGE );
    }

    return *s_homeURL;
}

void MainWindowBase::createNewView( const KURL &url, const KParts::URLArgs &args )
{
    createNewViewInternal( url, args );
}

View *MainWindowBase::createNewViewInternal( const KURL &url, const KParts::URLArgs &args )
{
    View *view = new View( args.frameName, m_widgetStack, "htmlview", this, "htmlpart" );

    // javascript stuff can do window.close()
    connect( view, SIGNAL( destroyed() ),
             this, SLOT( viewDestroyed() ) );

    connect( view->document()->widget(), SIGNAL( destroyed() ),
             this, SLOT( viewWidgetDestroyed() ) );

    int id = m_widgetId++;

    m_widgetStack->addWidget( view->document()->widget(), id );
    m_widgetStack->raiseWidget( id );

    m_views.append( view );

    setActiveView( view );

    connect( view->document()->browserExtension(), SIGNAL( createNewWindow( const KURL &, const KParts::URLArgs & ) ),
             this, SLOT( createNewView( const KURL &, const KParts::URLArgs & ) ) );
    connect( view->document()->browserExtension(), SIGNAL( createNewWindow( const KURL &,
                                                                            const KParts::URLArgs &,
                                                                            const KParts::WindowArgs &,
                                                                            KParts::ReadOnlyPart *& ) ),
             this, SLOT( createNewView( const KURL &, const KParts::URLArgs &,
                                        const KParts::WindowArgs &, KParts::ReadOnlyPart *& ) ) );

    connect( view, SIGNAL( openURLRequest( View *, const KURL &, const KParts::URLArgs & ) ),
             this, SLOT( openURL( View *, const KURL &, const KParts::URLArgs & ) ) );

    if ( !url.isEmpty() )
        view->openURL( url, args );

    m_windows->setEnabled( m_views.count() > 1 );

    return view;
}

void MainWindowBase::createNewView( const KURL &url, const KParts::URLArgs &args,
                                const KParts::WindowArgs &/*windowArgs*/, KParts::ReadOnlyPart *&part )
{
    View *view = 0;

    if ( !args.frameName.isEmpty() &&
         args.frameName.lower() != "_blank" ) // ### TODO _parent and _top?
    {
        KParts::BrowserHostExtension *hostExtension = 0;
        view = findChildView( args.frameName, &hostExtension );
    }

    if ( !view )
        view = createNewViewInternal( url, args );

    // ### obey windowArgs.lowerWindow

    part = view->document();
}

void MainWindowBase::setActiveView( View *view )
{
    if ( m_currentView )
        disconnectView( m_currentView );

    assert( m_views.findRef( view ) != -1 );

    m_currentView = view;

    m_widgetStack->raiseWidget( view->document()->widget() );

    connectView( m_currentView );

    if ( view->caption().isEmpty() )
        setCaption( "Konqueror" );
    else
        setCaption( view->caption() );
}

void MainWindowBase::disconnectView( View *view )
{
    disconnect( m_stop, SIGNAL( activated() ),
                view, SLOT( stop() ) );
    disconnect( view, SIGNAL( stopStatusChanged( bool ) ),
                m_stop, SLOT( setEnabled( bool ) ) );

    disconnect( m_back, SIGNAL( activated() ),
                view, SLOT( back() ) );
    disconnect( view, SIGNAL( backStatusChanged( bool ) ),
                m_back, SLOT( setEnabled( bool ) ) );

    disconnect( m_forward, SIGNAL( activated() ),
                view, SLOT( forward() ) );
    disconnect( view, SIGNAL( forwardStatusChanged( bool ) ),
                m_forward, SLOT( setEnabled( bool ) ) );

    disconnect( m_reload, SIGNAL( activated() ),
                view, SLOT( reload() ) );

    disconnect( view, SIGNAL( statusMessage( const QString & ) ),
                this, SLOT( statusMessage( const QString & ) ) );

    disconnect( view, SIGNAL( captionChanged( const QString & ) ),
                this, SLOT( setCaption( const QString & ) ) );

    if ( m_locationCombo )
        disconnect( view, SIGNAL( locationBarURLChanged( const QString & ) ),
                    this, SLOT( setLocationBarURL( const QString & ) ) );

    if (m_findInput)
	{
        disconnect( m_findInput, SIGNAL( textChanged( const QString & ) ),
                    view, SLOT( setFindText( const QString & ) ) );

        disconnect( m_findInput, SIGNAL( returnPressed() ),
                    view, SLOT( findNext() ) );
    }
    disconnect( m_security, SIGNAL( activated() ),
                view->document(), SLOT( slotSecurity() ) ); // ###

    disconnect( view->document(), SIGNAL( popupMenu( const QString &, const QPoint & ) ),
                this, SLOT( popupMenu( const QString &, const QPoint & ) ) );

    disconnect( m_loadImages, SIGNAL( activated() ),
                view->document(), SLOT( slotLoadImages() ) );
}

void MainWindowBase::connectView( View *view )
{
    connect( m_stop, SIGNAL( activated() ),
             view, SLOT( stop() ) );
    connect( view, SIGNAL( stopStatusChanged( bool ) ),
             m_stop, SLOT( setEnabled( bool ) ) );
    m_stop->setEnabled( view->isStopEnabled() );

    connect( m_back, SIGNAL( activated() ),
             view, SLOT( back() ) );
    connect( view, SIGNAL( backStatusChanged( bool ) ),
             m_back, SLOT( setEnabled( bool ) ) );
    m_back->setEnabled( view->canGoBack() );

    connect( m_forward, SIGNAL( activated() ),
             view, SLOT( forward() ) );
    connect( view, SIGNAL( forwardStatusChanged( bool ) ),
             m_forward, SLOT( setEnabled( bool ) ) );
    m_forward->setEnabled( view->canGoForward() );

    connect( m_reload, SIGNAL( activated() ),
             view, SLOT( reload() ) );

    connect( view, SIGNAL( statusMessage( const QString & ) ),
             this, SLOT( statusMessage( const QString & ) ) );

    connect( view, SIGNAL( captionChanged( const QString & ) ),
             this, SLOT( setCaption( const QString & ) ) );

    if ( m_locationCombo )
    {
        connect( view, SIGNAL( locationBarURLChanged( const QString & ) ),
                 this, SLOT( setLocationBarURL( const QString & ) ) );
        setLocationBarURL( view->locationBarURL() );
    }

    if ( m_findInput )
	{
        connect( m_findInput, SIGNAL( textChanged( const QString & ) ),
                 view, SLOT( setFindText( const QString & ) ) );

        connect( m_findInput, SIGNAL( returnPressed() ),
                 view, SLOT( findNext() ) );
    }

    if ( m_searchBar )
	{
        if ( view->isFindEnabled() )
            m_searchBar->show();
        else
            m_searchBar->hide();
    }
    connect( m_security, SIGNAL( activated() ),
             view->document(), SLOT( slotSecurity() ) ); // ###

    connect( view->document(), SIGNAL( popupMenu( const QString &, const QPoint & ) ),
             this, SLOT( popupMenu( const QString &, const QPoint & ) ) );

    connect( m_loadImages, SIGNAL( activated() ),
             view->document(), SLOT( slotLoadImages() ) );
}

void MainWindowBase::prepareWindowListMenu()
{
    m_windowList->clear();

    int i = 0;
    QListIterator<View> it( m_views );
    for (; it.current(); ++it, ++i )
    {
        QString text = it.current()->caption();

        if ( text.isEmpty() )
            text = it.current()->document()->url().prettyURL();

        if ( text.isEmpty() )
            text = i18n( "Untitled %1" ).arg( i + 1 );

        m_windowList->insertItem( text, i );
    }
}

void MainWindowBase::activateWindowFromPopup( int id )
{
    if ( id == -1 )
        return;

    setActiveView( m_views.at( id ) );
}

void MainWindowBase::viewDestroyed()
{
    View *view = const_cast<View *>( static_cast<const View *>( sender() ) );

    assert( view );

    bool current = static_cast<View *>( m_currentView ) == view;

    m_views.removeRef( view );

    m_windows->setEnabled( m_views.count() > 1 );

    // the widgetstack is cleaned up a few calls before

    if ( m_views.count() > 0 && current )
        setActiveView( m_views.first() ); // ### next()/prev()/first()
}

void MainWindowBase::viewWidgetDestroyed()
{
    // ### I still believe that QWidgetStack should connect to the destroyed signal
    QWidget *widget = const_cast<QWidget *>( static_cast<const QWidget *>( sender() ) );
    assert( widget );

    m_widgetStack->removeWidget( widget );
}

void MainWindowBase::closeView()
{
    if ( !m_currentView )
        return;

    if ( m_views.count() == 1 )
    {
        QTimer::singleShot( 0, this, SLOT( close() ) );
        return;
    }

    View *v = m_currentView;

    // ### hack
    m_views.findRef( m_currentView );
    View *next = m_views.next();
    if ( !next )
        next = m_views.first();

    setActiveView( next );

    delete v->document()->widget();
}

void MainWindowBase::openURL( View *view, const KURL &url, const KParts::URLArgs &_args )
{
    static QString _top = QString::fromLatin1( "_top" );
    static QString _self = QString::fromLatin1( "_self" );
    static QString _parent = QString::fromLatin1( "_parent" );
    static QString _blank = QString::fromLatin1( "_blank" );

    KParts::URLArgs args = _args;
    QString frameName = args.frameName;
    args.frameName = QString::null;

    if ( frameName.isEmpty() ||
         frameName == _top ||
         frameName == _self ||
         frameName == _parent )
    {
        view->openURL( url, args );
        return;
    }

    if ( frameName == _blank )
    {
        createNewView( url );
        return;
    }

    KParts::BrowserHostExtension *hostExtension = 0;

    if ( view->canHandleFrame( frameName, &hostExtension ) )
    {
        if ( hostExtension )
            hostExtension->openURLInFrame( url, args );
        else
            view->openURL( url, args );
    }
    else
    {
        View *v = findChildView( frameName, &hostExtension );

        if ( !v )
            createNewView( url, args );
        else if ( hostExtension )
            hostExtension->openURLInFrame( url, args );
        else
            v->openURL( url, args );
    }
}

void MainWindowBase::popupMenu( const QString &_url, const QPoint & )
{
    if ( _url.isEmpty() || !m_currentView )
        return;

    KURL url = m_currentView->document()->completeURL( _url );

    QPopupMenu *menu = new QPopupMenu( this );

    menu->insertItem( "Open link in new window", 0 );
    if ( m_bookmarks->isAvailable() )
        menu->insertItem( "Bookmark this link", 1 );

    int id = menu->exec( QCursor::pos() );

    if ( id == 0 )
       createNewView( url );
    else if ( id == 1 )
       m_bookmarks->addBookmark( url );

    delete menu;
}

View *MainWindowBase::findChildView( const QString &name, KParts::BrowserHostExtension **hostExtension )
{
    QListIterator<View> it( m_views );
    for (; it.current(); ++it )
    {
        if ( it.current()->canHandleFrame( name, hostExtension ) )
            return it.current();
    }

    return 0;
}

void MainWindowBase::newView()
{
    createNewView( KURL() );
}

void MainWindowBase::showPreferences()
{
    Preferences *pref = new Preferences( this );
    if ( pref->exec() == QDialog::Accepted )
    {
        pref->save();

        reparseConfiguration();

        KGlobal::config()->reparseConfiguration();

        if ( s_homeURL )
        {
            delete s_homeURL;
            s_homeURL = 0;
        }

        kapp->dcopClient()->send( "kcookiejar", "kcookiejar", "reloadPolicy()", QByteArray() );
    }
    delete pref;
}

void MainWindowBase::setLocationBarURL( const QString &url )
{
    if ( !m_locationCombo || url.isEmpty() )
        return;

    bool exists = false;
    int i = 0;
    for (; i < m_locationCombo->count(); ++i )
        if ( m_locationCombo->text( i ) == url )
        {
            exists = true;
            break;
        }

    if ( exists )
        m_locationCombo->setCurrentItem( i );
    else
    {
        m_locationCombo->insertItem( url, 0 );
        m_locationCombo->setCurrentItem( 0 );
    }
}

void MainWindowBase::toggleLocationBar( bool showLocationBar )
{
    QToolBar *tb = toolBar( "locationbar" );
    if ( !tb )
        return;

    if ( showLocationBar )
        tb->show();
    else
        tb->hide();
}

void MainWindowBase::toggleAutoloadImages( bool autoloadImages )
{
    // ###
    KConfig *config = KGlobal::config();
    config->setGroup( "HTML Settings" );
    config->writeEntry( "AutoLoadImages", autoloadImages );
    //config->sync();

    reparseConfiguration();

    showLoadImagesAction( !autoloadImages );
}

void MainWindowBase::showLoadImagesAction( bool /*show*/ )
{
}

void MainWindowBase::zoomValueSelected( int zoomValue )
{
    assert( m_currentView );
    m_currentView->document()->setScalingFactor( zoomValue );
}

void MainWindowBase::zoomMenuAboutToShow()
{
    // ### move all this into a QPopupMenu derived class!
    
    assert( m_currentView );

    int scalingFactor = m_currentView->document()->scalingFactor();

    for ( uint index = 0; index < m_zoomMenu->count(); ++index ) {
        int id = m_zoomMenu->idAt( index );
        m_zoomMenu->setItemChecked( id, id == scalingFactor );
    }
}

// ### should be const, but QObject::child isn't :(
QToolBar *MainWindowBase::toolBar( const char *name )
{
    return static_cast<QToolBar *>( child( name, "QToolBar" ) );
}

void MainWindowBase::reparseConfiguration()
{
    QListIterator<View> it( m_views );
    for (; it.current(); ++it )
        static_cast<KHTMLPartBrowserExtension *>( it.current()->document()->browserExtension() )
            ->reparseConfiguration();
}

void MainWindowBase::setupZoomMenu()
{
    m_zoomMenu = new QPopupMenu( this );

    for ( uint i = 20; i <= 150; i += 10 ) {
        QString text = QString::number( i ) + QString::fromLatin1( " %" );
        m_zoomMenu->insertItem( text, i );
    }

    connect( m_zoomMenu, SIGNAL( activated( int ) ),
             this, SLOT( zoomValueSelected( int ) ) );

    connect( m_zoomMenu, SIGNAL( aboutToShow() ),
             this, SLOT( zoomMenuAboutToShow() ) );
}

/*
void MainWindowBase::setEncoding( const QString &encoding )
{
}
*/

PopupAction::PopupAction( QPopupMenu *popup, const QString &text, const QIconSet &iconSet,
                          const QString &menuText, int accel, QObject *parent,
                          const char *name, bool toggle )
    : QAction( text, iconSet, menuText, accel, parent, name, toggle )
{
    m_popup = popup;
    m_toolButton = 0;
    m_popup->installEventFilter( this );
}

bool PopupAction::addTo( QWidget *widget )
{
    if ( widget->inherits( "QToolBar" ) )
    {
        QToolButton *button = new QToolButton( static_cast<QToolBar *>( widget ) );
        button->setToggleButton( isToggleAction() );

        QIconSet icon = iconSet();
        if ( !icon.isNull() )
            button->setIconSet( icon );

        button->setPopup( m_popup );
        button->setPopupDelay( 0 );

        m_toolButton = button;
        return true;
    }
    else if ( widget->inherits( "QPopupMenu" ) )
    {
        QPopupMenu *menu = static_cast<QPopupMenu *>( widget );
        menu->insertItem( menuText(), m_popup );
        return true;
    }
    else if ( widget->inherits( "QMenuBar" ) )
    {
        QMenuBar *menuBar = static_cast<QMenuBar *>( widget );
        menuBar->insertItem( menuText(), m_popup );
        return true;
    }
    return QAction::addTo( widget );
}

void PopupAction::setEnabled( bool enable )
{
    if ( m_toolButton )
        m_toolButton->setEnabled( enable );

    QAction::setEnabled( enable );
}

bool PopupAction::eventFilter( QObject *o, QEvent *e )
{
    if ( o != m_popup || !e || e->type() != QEvent::MouseButtonRelease )
        return false;

    QMouseEvent *mev = static_cast<QMouseEvent *>( e );
    if ( !m_popup->rect().contains( mev->pos() ) )
        return true;

    return false;
}

#include "mainwindow_qpe.h"
#include "mainwindow_x11.h"
#include "mainwindow_kiosk.h"

MainWindowBase *MainWindowBase::instantiate()
{
#if defined(KONQ_GUI_QPE)
    MainWindowBase *mw = new MainWindowQPE();
#elif defined(KONQ_GUI_X11)
    MainWindowBase *mw = new MainWindowX11();
#elif defined(KONQ_GUI_KIOSK)
    MainWindowBase *mw = new MainWindowKiosk();
#else
    return 0;
#endif
    mw->init();
    return mw;
}

#include "mainwindowbase.moc"

/*
 * vim:ts=4:et:sw=4
 */
