/***************************************************************************
                          ktransferwgetimpl.cpp  -  description
                             -------------------
    begin                : Thu Oct 19 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.23 $
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef ENABLE_WGET

#include <cstdlib>
#include <kdebug.h>
#include <qregexp.h>
#include <qtextstream.h>
#include <kapp.h>
#include "qdir.h"
#include "utils.h"
#include "krootcontainerimpl.h"
#include "ktmanagerimpl.h"
#include "ktransferwgetimpl.h"


const char KTransferWgetImpl::RE_STR[RE_NUM][300] = 
{
   // RE_INT
   "^--([[:digit:]]{2}:[[:digit:]]{2}:[[:digit:]]{2})--[[:space:]]+([^\n]+)\n[[:space:]]+(\\(try:[[:space:]]*([[:digit:]]+)[[:space:]]*\\)[[:space:]]+)?=>[[:space:]]+`([^\n]+)'\n",
   // RE_CNN
   "^Connecting to ([[:print:]]+):([[:digit:]]+)[[:space:]]*\\.{3}",
   // RE_FTP_LOGIN
   "^Logging in as ([[:print:]]+)[[:space:]]*\\.{3}",
   // RE_FTP_TYPE
   "^[[:space:]]*==>[[:space:]]+TYPE[[:space:]]+(I|A)[[:space:]]*\\.{3}",
   // RE_FTP_CWD
   "^[[:space:]]*==>[[:space:]]+CWD[[:space:]]+((not required\\.\n|not needed\\.\n)|([[:print:]]+ \\.{3}))",
   // RE_FTP_PORT
   "^[[:space:]]*==>[[:space:]]+PORT[[:space:]]+\\.{3}",
   // RE_FTP_PASV
   "^[[:space:]]*==>[[:space:]]+PASV[[:space:]]+\\.{3}",
   // RE_FTP_LIST
   "^[[:space:]]*==>[[:space:]]+LIST[[:space:]]+\\.{3}",
   // RE_FTP_REST
   "^[[:space:]]*==>[[:space:]]+REST[[:space:]]+([[:digit:]]+)[[:space:]]*\\.{3}",
   // RE_FTP_REST_ACK
   "^[[:space:]]*\nREST failed, starting from scratch.\n",
   // RE_FTP_RETR
   "^[[:space:]]*==>[[:space:]]+RETR[[:space:]]+([[:print:]]+) \\.{3}",
   // RE_FTP_LENGTH
   "^Length:[[:space:]]+([[:digit:],-]+)[[:space:]]+(\\[([[:digit:],-]+)[[:space:]]+to go\\][[:space:]]+)?(\\(unauthoritative\\))?\n",
   // RE_DWN
   "^\n*( *\\[ skipping +([[:digit:]]+)K \\]\n)? *([[:digit:]]+)K ->( ,+( ,+)*)?",
   // RE_DW1
   "^\n* *([[:digit:]]+)K -> ",
   // RE_DW2
   "^ ?\\.+( \\.+)*",
   // RE_DW3
   "^ *(\\[ *([[:digit:]-]+)%\\])?\n",
   // RE_FTP_DW4
   "^[[:print:]]+: [[:print:]]+, (closing [[:print:]]+\\.)\n+|"
   "^[[:print:]]+ \\([[:print:]]+\\) - ("
   "(Data c.+: [[:print:]]+; )\n*|"
   "(Control [[:print:]]+\\.|Data [[:print:]]+\\.)\n((Giving up|Retrying)\\.\n)?\n*|"
   "`[[:print:]]+' saved \\[([[:print:]]+)\\]\n+)",
   /* RE_FTP_ACK
    * 2: success message: 'done.', 'connected!', 'Logged in!'
    * 3: error message
    * 5: resuming from error: 'Retrying', 'Giving up'
    */
   "^[[:space:]]*(done\\.|connected!|Logged in!)[[:space:]]*\n?",
   /* RE_ACKERR
    * 1: error message
    * 3: retrying/giving up
    */
   "^[[:print:]]*\n([[:print:]]+)\n((Retrying|Giving up)\\.\n\n)?\n*",
   // RE_HTTP_REQUEST
   "^(HTTP|Proxy)[[:space:]]+request sent, awaiting response *\\.{3}",
   // RE_HTTP_AUTHFAIL
   "^(Authorization failed|Unknown authentication scheme)\\.\n",
   // _RE_HTTP_LOCATION
   "^Location: *(unspecified|[[:print:]]+)( +\\[following\\])? *\n",
   // RE_HTTP_LENGTH
   "^Length: +((([[:digit:],-]+) (\\(([[:digit:],-]+) to go\\))?)|(unspecified|ignored)) *(\\[[[:print:]]+\\])? *\n",
   // RE_HTTP_DW4 parse end string
   "^[[:print:]]+ \\([[:print:]]+\\) - `[[:print:]]+' saved \\[([[:digit:]]+)(/([[:digit:]]+))?\\] *\n+|"
   "^([[:print:]]+ \\([[:print:]]+\\) - [[:print:]]+)\n?(Giving up|Retrying)\\.\n+",
   // RE_HTTP_ACK
   // 1: success message: 'connected!', '200 OK.', '206 Partial Content'
   "^ *(connected!|[2-3][[:digit:]]+ [[:print:]]+)\n"
};

regex_t KTransferWgetImpl::_preg[RE_NUM];
bool KTransferWgetImpl::_reInit = false;
	

KTransferWgetImpl::KTransferWgetImpl(int type) 
   : KTransferImpl(type), _tryResuming(false), _localStatus(0), _protocol(0),
     _localError(0), _block(0), _processExit(false), _directory(false), 
     _cfg_passiveFtp(true), _cfg_HTTPCache(false), _cfg_proxy(true), 
     _cfg_readTimeout(0), _cfg_proxyHTTP("-"), _cfg_proxyFTP("-"), 
     _cfg_proxyPortHTTP(0), _cfg_proxyPortFTP(0), _cfg_speed(SPD_MICRO),
     _cfg_ignoreLen(true)
{
   initRE();
   connect(&process, SIGNAL(receivedStdout(KProcess*, char*, int)), 
	   SLOT(slotProcessOutput(KProcess*, char*, int)));
   connect(&process, SIGNAL(receivedStderr(KProcess*, char*, int)), 
	   SLOT(slotProcessOutput(KProcess*, char*, int)));
   connect(&process, SIGNAL(processExited(KProcess*)), 
	   SLOT(slotProcessExited(KProcess*)));
}

KObjectImpl * KTransferWgetImpl::clone() const 
{
   kdFatal(!isGlobal()) << name() << ": clone on a non global object" << endl;
   return new KTransferWgetImpl(type());
}

void KTransferWgetImpl::loadData() 
{
   KTransferImpl::loadData();
   kdDebug(D_INI) << name() << ": wget loadData" << endl;
   if (!dom().hasAttribute("CommandPath"))
   {
      setCmdPath("/usr/bin/wget");
      setPassiveFtp(false);
      setHTTPCache(true);
      setProxy(false);
      setReadTimeout(900);
      setProxyFTP("");
      setProxyPortFTP(8080);
      setProxyHTTP("");
      setProxyPortHTTP(8080);
      setNoProxyList(QStringList());
      setSpeed(SPD_DEFAULT);
      setIgnoreLength(false);
   }
   _cfg_cmdPath = dom().attribute("CommandPath", "/usr/bin/wget");
   _cfg_passiveFtp = dom().attribute("PassiveFtp", 
				     QString::number(false)).toInt();
   _cfg_HTTPCache = dom().attribute("HTTPCache", QString::number(true)).toInt();
   _cfg_proxy = dom().attribute("Proxy", QString::number(false)).toInt();
   _cfg_readTimeout = dom().attribute("ReadTimeout", 
				      QString::number(900)).toInt();
   _cfg_proxyFTP = dom().attribute("FTPProxy", "");
   _cfg_proxyPortFTP = dom().attribute("FTPProxyPort", 
				       QString::number(8080)).toInt();
   _cfg_proxyHTTP = dom().attribute("HTTPProxy", "");
   _cfg_proxyPortHTTP = dom().attribute("HTTPProxyPort", 
					QString::number(8080)).toInt();
   _cfg_noProxy = QStringList::split(",", dom().attribute("NoProxy", ""));
   _cfg_speed = static_cast<TransferSpeed>(
      dom().attribute("Speed", QString::number(SPD_DEFAULT)).toInt());
   _cfg_ignoreLen = dom().attribute("IgnoreLen", QString::number(false)).toInt();

   _protocol = 0;
   _tryResuming = false;
   _block = 0;
   _localStatus = 0;
   _localError = 0;
}

const KTransferWgetImpl * KTransferWgetImpl::global() const 
{
   return dynamic_cast<const KTransferWgetImpl*>(KTransferImpl::global());
}

bool KTransferWgetImpl::isProtocolSupported(const QString &proto) const 
{
   return (proto == "ftp" || proto == "http");
}

QString KTransferWgetImpl::getWgetVersion() 
{
   FILE *fd = popen(getCmdPath() + " --version", "r");
   if (fd == 0)
      return QString::null;
   QString line = QTextIStream(fd).readLine();
   pclose(fd);
   kdDebug(D_RUN) << name() << ": getWgetVersion " << line << endl;
   return line;
}

bool KTransferWgetImpl::isRunning() const 
{
   return KTransferImpl::isRunning() || process.isRunning();
}

KURL KTransferWgetImpl::getTempFile(const KURL &rmt, const KURL &) const 
{
   KURL tmp;
   tmp.setPath(QFileInfo(
		  dynamic_cast<KTManagerImpl*>(container())->getWorkingDir(), 
		  rmt.fileName()).absFilePath());
   return tmp;
}

void KTransferWgetImpl::initRE() 
{
   if (_reInit)
      return;
   int chk;
   char buf[100];
   for (int i = 0; i < RE_NUM; i++) 
   {
      chk = regcomp(&_preg[i], RE_STR[i], REG_EXTENDED | REG_ICASE);
      if (chk) 
      {
	 regerror(chk, &_preg[i], buf, 100);
	 kdFatal() << "RE " << i << ":" << RE_STR[i] << " ERROR:" 
		   << buf << endl;
      }
   }
   _reInit = true;
}

bool KTransferWgetImpl::processStart() 
{
   kdDebug(D_RUN) << name() << ": processStart" << endl;
   _directory = false;
   _processExit = false;
   _parsebuf = "";
   _tryResuming = false;
   _localStatus = 0;
   _localError = 0;
   QString tmpfile = tmp().path();
   _parseCmd = (remote().protocol() == QString("ftp") ? 
		&KTransferWgetImpl::parseFTP : &KTransferWgetImpl::parseHTTP);
   process.clearArguments();
   process << getCmdPath();
   if (QDir::current().exists(tmpfile)) 
   {
      if (rsmState() == RSM_NO)
	 QDir::current().remove(tmpfile);
      else
	 process << "-c";
   }
   setPartial(QFileInfo(tmpfile).size());
   if (getPassiveFtp())
      process << "--passive-ftp";
   process << (getHTTPCache() ? "--cache=on" : "--cache=off");
   QStringList lst = getNoProxyList();
   bool proxy = getProxy();
   for (QStringList::Iterator item = lst.begin(); item != lst.end() && proxy; 
	++item) 
   {
      QRegExp re(*item, false, true);
      proxy = re.match(remote().host(), 0, 0) == -1;
   }
   switch (getSpeed()) 
   {
   case SPD_DEFAULT:
      process << "--dot-style=default";
      _block = 1024;
      break;
   case SPD_BINARY:
      process << "--dot-style=binary";
      _block = 8192;
      break;
   case SPD_MEGA:
      process << "--dot-style=mega";
      _block = 65536;
      break;
   case SPD_MICRO:
      process << "--dot-style=micro";
      _block = 128;
      break;
   default:
      kdFatal() << name() << ": transfer speed unknown" << endl;
   }
   process << (proxy ? "--proxy=on" : "--proxy=off");
   if (proxy) 
   {
      // set environment variable
      char str[80];
      if (!getProxyHTTP().isEmpty()) 
      {
	 sprintf(str, "%s:%d", (const char*)getProxyHTTP(), getProxyPortHTTP());
	 int check = setenv("http_proxy", str, 1);
	 kdFatal(check != 0, D_RUN) << "SETENV ERROR" << endl;
      }
      else
	 unsetenv("http_proxy");
      if (!getProxyFTP().isEmpty()) 
      {
	 sprintf(str, "%s:%d", (const char*)getProxyFTP(), getProxyPortFTP());
	 int check = setenv("ftp_proxy", str, 1);
	 kdFatal(check != 0, D_RUN) << "SETENV ERROR" << endl;
      } 
      else
	 unsetenv("ftp_proxy");
   }
   // set locale to C
   unsetenv("LANGUAGE"); // thanks to Takumi ASAKI
   int check = setenv("LC_ALL", "C", 1);
   kdFatal(check != 0, D_RUN) << "SETENV ERROR" << endl;
   if (getIgnoreLength())
      process << "--ignore-length";
   process << "-t" << "1" // << QString().setNum(getMaxRetry())
      //<< "-w" << QString().setNum(getWaitRetry())
	   << "-P" << dynamic_cast<KTManagerImpl*>(container())->getWorkingDir().absPath()
	   << "-v"
	   << "-T" << QString().setNum(getReadTimeout())
	   << "-nr" // if you are downloading a directory from ftp
	   << "--htmlify=off" // don't htmlify directories
	   << "--glob=off" // don't use glob pattern in ftp url
	   << remote().url();
   //FIXME!! linee x debug
   QString cmdline;
   for (char *arg = process.args()->first(); arg != 0; 
	arg = process.args()->next())
      cmdline += arg + QString(" ");
   kdDebug(D_RUN) << name() << ": cmdline " << cmdline << endl;

   if (process.start(KProcess::NotifyOnExit, KProcess::AllOutput)) 
   {
      opProcessStart();
      return true;
   }
   return false;
}

bool KTransferWgetImpl::processKill() 
{
   return process.kill();
}

void KTransferWgetImpl::processCheckResumeBegin()
{
   _oldPartial = partial();
   QString tmpfile = tmp().path();
   QString tmpfileSave = tmpfile + ".tmp";
   if (QDir::current().exists(tmpfile))
      QDir::current().rename(tmpfile, tmpfileSave);
   QFile f(tmpfile);
   f.open(IO_WriteOnly);
   //FIXME!! gestione lunghezza, se > len ?
   char buffer[20];
   f.writeBlock(buffer, 20);
   f.close();
   setPartial(20);
}

void KTransferWgetImpl::processCheckResumeEnd()
{
   QString tmpfile = tmp().path();
   QString tmpfileSave = tmpfile + ".tmp";
   setPartial(_oldPartial);
   QDir::current().remove(tmpfile);
   if (QDir::current().exists(tmpfileSave))
      QDir::current().rename(tmpfileSave, tmpfile);
}

void KTransferWgetImpl::parseFTP() 
{
   Match res = MTC_OK;
   while (!_parsebuf.isEmpty() 
	  && (res == MTC_OK || (res != MTC_FAIL && _processExit))) 
   {
      kdDebug(D_PRS) << name() << "#" << _parsebuf << "#" << endl;
      switch (_localStatus) 
      {
      case FTP_DISCONNECT:
	 res = parseProlog(FTP_CONNECT_REQ);
	 // call opRunStart
	 break;
      case FTP_CONNECT_REQ:
	 //it can be a RE_CNN or a RE_CWD if the connection is still alive
	 res = parseToken(RE_CNN, FTP_CONNECT_ACK);
	 if (res != MTC_OK) {
	    Match res1 = parseCwd();
	    res = (res1 == MTC_OK ? MTC_OK : 
		   (res1 == MTC_CONT || res == MTC_CONT ? MTC_CONT : MTC_FAIL));
	 }
	 break;
      case FTP_CONNECT_ACK:
	 res = parseAck(RE_FTP_ACK, FTP_LOGIN_REQ);
	 if (res == MTC_OK)
	    opConnect();
	 break;
      case FTP_LOGIN_REQ:
	 res = parseToken(RE_FTP_LOGIN, FTP_LOGIN_ACK);
	 break;
      case FTP_LOGIN_ACK:
	 res = parseAck(RE_FTP_ACK, FTP_TYPE_REQ);
	 break;
      case FTP_TYPE_REQ:
	 res = parseToken(RE_FTP_TYPE, FTP_TYPE_ACK);
	 break;
      case FTP_TYPE_ACK:
	 res = parseAck(RE_FTP_ACK, FTP_CWD_REQ);
	 break;
      case FTP_CWD_REQ:
	 // if cwd not executed, goto FTP_(PORT/PASV)_REQ
	 res = parseCwd(); 
	 break;
      case FTP_CWD_ACK:
	 // if passive next is PASV else PORT
	 // warn: fatal error with one more \n
	 if (getPassiveFtp())
	    res = parseAck(RE_FTP_ACK, FTP_PASV_REQ);
	 else 
	    res = parseAck(RE_FTP_ACK, FTP_PORT_REQ);
	 break;
      case FTP_PORT_REQ:
	 res = parseToken(RE_FTP_PORT, FTP_PORT_ACK);
	 break;
      case FTP_PORT_ACK:
	 res = parseAck(RE_FTP_ACK, FTP_DUMMY1); // next DUMMY1
	 break;
      case FTP_PASV_REQ:
	 res = parseToken(RE_FTP_PASV, FTP_PASV_ACK);
	 break;
      case FTP_PASV_ACK:
	 res = parseAck(RE_FTP_ACK, FTP_DUMMY1); // next DUMMY1
	 break;
      case FTP_DUMMY1:
	 // here we can have LIST, REST or RETR
	 res = parseToken(RE_FTP_LIST, FTP_LIST_ACK);
	 if (res != MTC_OK)
	 {
	    Match res1 = parseToken(RE_FTP_RETR, FTP_RETR_ACK);
	    if (res1 != MTC_OK)
	    {
	       Match res2 = parseRest();
	       res = (res2 == MTC_OK ? MTC_OK :
		      (res2 == MTC_CONT || res1 == MTC_CONT 
		       || res == MTC_CONT ? MTC_CONT : MTC_FAIL));
	    }
	    else
	       res = MTC_OK;
	 }
	 break;
      case FTP_LIST_REQ:
	 // we never come here
	 res = parseToken(RE_FTP_LIST, FTP_LIST_ACK, FTP_REST_REQ);
	 break;
      case FTP_LIST_ACK:
	 // warn: fatal error with one more \n
	 res = parseAck(RE_FTP_ACK, FTP_LENGTH); // if match goto LENGTH
	 _directory = (res == MTC_OK);
	 if (res == MTC_OK) 
	 {
	    opCheckResume(RSM_NO);
	 }
	 break;
      case FTP_REST_REQ:
	 // we never come here
	 res = parseRest(); //if fail goto RETR, else setPartial && tryResuming
	 break;
      case FTP_REST_ACK:
	 // next is RETR
	 res = parseAck(RE_FTP_ACK, FTP_RETR_REQ);
	 break;
      case FTP_RETR_REQ:
	 res = parseToken(RE_FTP_RETR, FTP_RETR_ACK);
	 break;
      case FTP_RETR_ACK:
	 // next is LENGTH
	 // warn: fatal error witn one more \n
	 res = parseAck(RE_FTP_ACK, FTP_LENGTH);
	 break;
      case FTP_LENGTH:
	 res = parseFtpLength(); // setLen, goto DOWNLOAD_START
	 if (res == MTC_FAIL) 
	 {
	    // check for a fatal error
	    res = parseAck(RE_FTP_ACK, 0);
	    if (res == MTC_FAIL) 
	    {
	       // if the file transfer is complete, try to continue
	       setLen(partial());
	       res = MTC_CONT;
	       _localStatus = FTP_DOWNLOAD_START;
	    }
	 }
	 break;
      case FTP_DOWNLOAD_START:
	 res = parseDownloadStart(FTP_DOWNLOAD_RUN);
	 // call opDownload
	 break;
      case FTP_DOWNLOAD_RUN:
	 // call opDataRead()
	 res = parseDownloadRun();
	 if (res != MTC_OK) 
	 {
	    // check for end of transfer
	    // call opRunEnd if succesful
	    Match res1 = parseFtpDownloadEnd();
	    res = (res1 == MTC_OK ? MTC_OK : 
		   (res1 == MTC_CONT || res == MTC_CONT ? MTC_CONT : MTC_FAIL));
	 }
	 break;
      case FTP_DOWNLOAD_END:
	 //kdFatal(D_PRS) << name() << ": error reached FTP_END" << endl;
	 res = MTC_FAIL;
	 break;
      default:
	 res = MTC_FAIL;
	 //kdFatal(D_PRS) << name() << ":" << _localStatus << " unknown FTP state" << endl;
      }
      //kdFatal(res == MTC_FAIL) << name() << " FTP state " << _localStatus << " unable to parse #" << _parsebuf << "#" << endl;
      if (res == MTC_FAIL) 
      {
	 kdDebug(D_PRS) << name() << ": parse fail" << endl;
	 if (_processExit) 
	 {
	    // process finished, ignore remaining unparsable trash
	    res = MTC_OK;
	    _localError = ERR_OK;
	    _pmatch[0].rm_so = 0;
	    _pmatch[0].rm_eo = _parsebuf.length();
	 } 
	 else 
	 {
	    // raise a fatal error
	    opFatalError(QString("Wget Version: %1\nFTP state %2 : unable to parse #%3#").arg(getWgetVersion()).arg(_localStatus).arg(_parsebuf));
	    _localError = ERR_ABORT;
	 }
      }
      if (_localError == ERR_RETRY) 
      { 
	 // restart from beginning
	 kdDebug(D_PRS) << name() << ": retry" << endl;
	 _localStatus = FTP_DISCONNECT;
	 _localError = ERR_OK;
	 //opRunEnd();
      }
      // remove parsed string from buffer
      if (res == MTC_OK) 
      {
	 //kdDebug() << "!" << matchStr(0) << "!" << endl;
	 _parsebuf.remove(0, _pmatch[0].rm_eo - _pmatch[0].rm_so);
      }
      kdDebug(D_PRS) << name() << ": match " << res << " FTP state " << _localStatus << endl;
   } // end while
   parseExit();
   kdDebug(D_PRS) << name() << ":" << stateStr() << ": exit FTP parse" << endl;
}

QString KTransferWgetImpl::matchStr(int i) const 
{
   return _parsebuf.mid(_pmatch[i].rm_so, _pmatch[i].rm_eo - _pmatch[i].rm_so);
}

KTransferWgetImpl::Match 
KTransferWgetImpl::parseToken(REId token, int nextState, int failState) 
{
   int check = regexec(&_preg[token], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      // goto nextState and mark string as parsed
      _localStatus = nextState;
      return MTC_OK;
   }
   if (failState != 0 && _parsebuf.length() > 100 || _processExit) 
   { 
      // goto to failState, but not advance in string parsing
      _localStatus = failState;
      return MTC_CONT;
   }
   // fallback condition
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseProlog(int nextState) 
{
   int check = regexec(&_preg[RE_INT], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   { // ok
      QString str = matchStr(5);
      kdWarning(QFileInfo(str).absFilePath() != tmp().path(), D_PRS) << name() << ": file " << str << " != " << tmp().path() << endl;
      setTmp(QFileInfo(str).absFilePath());
      _localStatus = nextState;
      //opRunStart();
      return MTC_OK;
   }
   return (_parsebuf.length() > 200) ? MTC_FAIL : MTC_CONT;
}
  
KTransferWgetImpl::Match KTransferWgetImpl::parseAck(REId token, int nextState) 
{
   // check for download start, necessary in FTP_LENGTH
   int check = regexec(&_preg[RE_DWN], _parsebuf, 10, &_pmatch[0], 0);
   if (!check)
      return MTC_FAIL;
   // check for ack
   check = regexec(&_preg[token], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   { // ack ok
      _localStatus = nextState;
      return MTC_OK;
   }
   // be sure that check for ack is failed
   if (_parsebuf.length() < 40 && !_processExit)
      return MTC_CONT;
   // check for error ack
   check = regexec(&_preg[RE_ACKERR], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   { // error ack 
      QString str = matchStr(3);
      // retry, give up or abort
      if (str.isEmpty() && _processExit)
	 // fatal error
	 _localError = ERR_ABORT;
      else if (str == "Giving up")
	 _localError = ERR_GIVEUP;
      else if (str == "Retrying")
	 _localError = ERR_RETRY;
      else
	 goto fail;
      // error message
      str = matchStr(1);
      kdDebug(D_PRS) << name() << ": FTP state " << _localStatus << " error :" << str << endl;
      opError(str);
      kdDebug(D_PRS) << name() << ": error action " << matchStr(3) << endl;
      return MTC_OK;
   }
fail:
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}
  
KTransferWgetImpl::Match KTransferWgetImpl::parseCwd() 
{
   int check = regexec(&_preg[RE_FTP_CWD], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      if (_pmatch[2].rm_so != -1) // cwd not required/not needed, goto PORT/PASV
	 _localStatus = getPassiveFtp() ? FTP_PASV_REQ : FTP_PORT_REQ;
      else
	 _localStatus = FTP_CWD_ACK;
      return MTC_OK;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseRest() 
{
   int check = regexec(&_preg[RE_FTP_REST], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      QString str = matchStr(1);
      setPartial(str.toInt());
      _tryResuming = true;
      _localStatus = FTP_REST_ACK;
      return MTC_OK;
   }
   if (_parsebuf.length() > 40 || _processExit) 
   { // REST not present
      _localStatus = FTP_RETR_REQ;
      kdError(partial() > 0, D_PRS) << name() << " REST not present and partial != 0" << endl;
      setPartial(0);
      _tryResuming = false;
      return MTC_CONT;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseRestAck() 
{
   int check = regexec(&_preg[RE_FTP_REST_ACK], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      setPartial(0);
      _localStatus = FTP_RETR_REQ;
      if (_tryResuming) 
      {
	 opCheckResume(RSM_NO);
      }
      return MTC_OK;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseFtpLength() 
{
   int check = regexec(&_preg[RE_FTP_LENGTH], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      QString str = matchStr(1);
      setLen(parseNumber(str) + partial());
      _localStatus = FTP_DOWNLOAD_START;
      return MTC_OK;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseDownloadStart(int nextState) 
{
   int check = regexec(&_preg[RE_DWN], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      if (_tryResuming) 
      {
	 opCheckResume(RSM_YES);
      }
      opDownload();
      _localStatus = nextState;
      return MTC_OK;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}
  
KTransferWgetImpl::Match KTransferWgetImpl::parseDownloadRun() 
{
   int check = regexec(&_preg[RE_DW1], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      // parse: 120K ->
      return MTC_OK;
   }
   check = regexec(&_preg[RE_DW2], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      // parse:  .... ....
      QString str = matchStr(0);
      int datalen = str.contains('.') * _block;
      opDataRead(datalen);
      return MTC_OK;
   }
   check = regexec(&_preg[RE_DW3], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      // parse: [ 33%]\n"
      return MTC_OK;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseFtpDownloadEnd() 
{
   int check = regexec(&_preg[RE_FTP_DW4], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      QString str;
      if (_pmatch[7].rm_so != -1) 
      {
	 // transfer succesfully completed
	 str = matchStr(7);
	 int l = str.toInt();
	 kdWarning(l != len(), D_PRS) << name() << " length " << l << "!=" << len() << endl;
	 if (l != len())
	    setLen(l);
	 if (partial() < len())
	    opDataRead(len() - partial());
	 _localStatus = FTP_DOWNLOAD_END;
	 //opRunEnd();
	 return MTC_OK;
      }
      // errors
      if (_pmatch[1].rm_so != -1) 
      {
	 // match 1: fatal error
	 str = matchStr(1);
	 _localError = ERR_ABORT;
      } 
      else if (_pmatch[3].rm_so != -1) 
      {
	 // match 3: ???, continue
	 str = matchStr(3);
      } 
      else if (_pmatch[4].rm_so != -1) 
      {
	 // match 4: error: retry, giveup or ok
	 str = matchStr(6);
	 if (str.isEmpty())
	    ; //ok
	 else if (str == "Giving up")
	    // give up 
	    _localError = ERR_GIVEUP;
	 else
	    // retry
	    _localError = ERR_RETRY;
	 // error message
	 str = matchStr(4);
      }
      // output error message
      opError(str);
      return MTC_OK;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

void KTransferWgetImpl::parseExit() 
{
   // set resume if we are checking for it
   if (rsmState() == RSM_CHECKING && _processExit) 
   {
      if (_localError != ERR_OK) 
      {
	 // if we are checking for resuming and it give a recoverable error after RETR
	 //setRsmState((_localStatus == FTP_RETR_ACK && _localError == ERR_GIVEUP) ? RSM_NO : RSM_CHECKFAIL);
	 opCheckResume((_localStatus == FTP_RETR_ACK && _localError == ERR_GIVEUP) ? RSM_NO : RSM_CHECKFAIL);
      }
      else
      {
	 //setRsmState(RSM_CHECKFAIL);
	 opCheckResume(RSM_CHECKFAIL);
      }
   }
   switch (_localError) 
   {
   case ERR_OK:
      if (_processExit)
      {
	 if (killed())
	    setState(TRN_END_KILL);
	 else if (partial() == len())
	    setState(TRN_END_OK);
	 else 
	    setState(TRN_END_ERROR);
      }
      break;
   case ERR_RETRY:
      kdFatal(D_PRS) << name() << " exiting with ERR_RETRY" << endl;
      break;
   case ERR_GIVEUP:
      setState(TRN_END_ERROR);
      break;
   case ERR_ABORT:
      //opFatalError(QString("Wget version:%1\nHTTP state %2 : abort on #%3#").arg(getWgetVersion()).arg(_localStatus).arg(_parsebuf));
      kdWarning(D_PRS) << name() << ": ERR_ABORT " << stateStr() << endl;
      setState(TRN_END_FATAL); //FIXME!!
      break;
   default:
      kdFatal(D_PRS) << name() << " : unknown localError " << _localError << endl;
      //emitMod(); //FIXME??
   }
}

void KTransferWgetImpl::parseHTTP() 
{
   Match res = MTC_OK;
   while (!_parsebuf.isEmpty() 
	  && (res == MTC_OK || (res != MTC_FAIL && _processExit))) 
   {
      kdDebug(D_PRS) << name() << "#" << _parsebuf << "#" << endl;
      switch (_localStatus) 
      {
      case HTTP_DISCONNECT:
	 res = parseProlog(HTTP_CONNECT_REQ);
	 // call opRunStart
	 break;
      case HTTP_CONNECT_REQ:
	 res = parseToken(RE_CNN, HTTP_CONNECT_ACK);
	 break;
      case HTTP_CONNECT_ACK:
	 res = parseAck(RE_HTTP_ACK, HTTP_REQUEST_REQ);
	 if (res == MTC_OK)
	    opConnect();
	 break;
      case HTTP_REQUEST_REQ:
	 res = parseToken(RE_HTTP_REQUEST, HTTP_REQUEST_ACK);
	 break;
      case HTTP_REQUEST_ACK:
	 res = parseAck(RE_HTTP_ACK, HTTP_LENGTH);
	 break;
      case HTTP_LENGTH:
	 res = parseHttpLength();
	 if (res != MTC_OK) 
	 {
	    // if parse correctly then raise a FATALERROR
	    Match res1 = parseToken(RE_HTTP_AUTHFAIL, 0);
	    if (res1 == MTC_OK)
	       _localError = ERR_ABORT;
	    else
	    {
	       // if LOCATION is found, it may redirect to an HTTP or an FTP
	       // transfer, so go out and restart, else goto LENGTH
	       Match res2 = parseLocation();
	       res = (res2 == MTC_OK ? MTC_OK :
		      (res1 == MTC_CONT 
		       || res2 == MTC_CONT ? MTC_CONT : MTC_FAIL));
	    }
	 }
	 break;
      case HTTP_DOWNLOAD_START:
	 res = parseDownloadStart(HTTP_DOWNLOAD_RUN);
	 // call opDownloadStart();
	 break;
      case HTTP_DOWNLOAD_RUN:
	 // call opDataRead()
	 res = parseDownloadRun();
	 if (res != MTC_OK)
	 {
	    Match res1 = parseHttpDownloadEnd();
	    res = (res1 == MTC_OK ? MTC_OK :
		   (res1 == MTC_CONT || res == MTC_CONT ? MTC_CONT : MTC_FAIL));
	 }
	 break;
      case HTTP_DOWNLOAD_END:
	 //kdFatal(D_PRS) << name() << ": error reached HTTP_END" << endl;
	 res = MTC_FAIL;
	 break;
      default:
	 res = MTC_FAIL;
	 //kdFatal(D_PRS) << name() << ":" << _localStatus << " unknown HTTP state" << endl;
      }
      //kdFatal(res == MTC_FAIL) << name() << " HTTP state " << _localStatus << " unable to parse #" << _parsebuf << "#" << endl;
      if (res == MTC_FAIL) 
      {
	 if (_processExit) 
	 {
	    // process finished, ignore remaining unparsable trash
	    res = MTC_OK;
	    _localError = ERR_OK;
	    _pmatch[0].rm_so = 0;
	    _pmatch[0].rm_eo = _parsebuf.length();
	 } 
	 else 
	 {
	    // raise a fatal error
	    opFatalError(QString("Wget version:%1\nHTTP state %2 : unable to parse #%3#").arg(getWgetVersion()).arg(_localStatus).arg(_parsebuf));
	    _localError = ERR_ABORT;
	 }
      }
      if (_localError == ERR_RETRY) 
      { 
	 // restart from beginning
	 kdDebug(D_PRS) << name() << ": retry" << endl;
	 _localStatus = HTTP_DISCONNECT;
	 _localError = ERR_OK;
	 //opRunEnd();
      }
      // remove parsed string from buffer
      if (res == MTC_OK) 
      {
	 _parsebuf.remove(0, _pmatch[0].rm_eo - _pmatch[0].rm_so);
      }
      kdDebug(D_PRS) << name() << " match " << res << " HTTP state " << _localStatus << endl;
   }
   parseExit();
   kdDebug(D_PRS) << name() << ":" << stateStr() << ": exit HTTP parse" << endl;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseLocation() 
{
   int check = regexec(&_preg[RE_HTTP_LOCATION], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      QString str = matchStr(1);
      if (str == QString("unspecified")) 
      {
	 //FIXME try to restart with http
	 _localStatus = HTTP_DISCONNECT;
	 return MTC_OK;
      } 
      else 
      {
	 // redefine parse command in response to new address
	 KURL newurl(str);
	 _parseCmd = (newurl.protocol() == QString("ftp") ? 
		      &KTransferWgetImpl::parseFTP : &KTransferWgetImpl::parseHTTP);
	 _localStatus = HTTP_DISCONNECT; // == FTP_DISCONNECT;
	 // update parse buffer
	 _parsebuf.remove(0, _pmatch[0].rm_eo - _pmatch[0].rm_so);
	 // return CONT, exit the http loop and enter the new one
	 return MTC_CONT;
      }
   }
   // goto HTTP_LENGTH
   _localStatus = HTTP_LENGTH;
   return MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseHttpLength() 
{
   int check = regexec(&_preg[RE_HTTP_LENGTH], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      if (_pmatch[3].rm_so != -1) 
      {
	 setLen(parseNumber(matchStr(3)));
	 QString str = matchStr(5);
	 if (str.isEmpty())
	    setPartial(0);
	 else
	    setPartial(len() - parseNumber(str));
	 _tryResuming = partial() > 0;
      } 
      else 
      {
	 setLen(0);
	 setPartial(0);
	 _tryResuming = false;
	 if (rsmState() == RSM_CHECKING) 
	 {
	    opCheckResume(RSM_NO);
	 }
      }
      _localStatus = HTTP_DOWNLOAD_START;
      return MTC_OK;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseHttpDownloadEnd() 
{
   int check = regexec(&_preg[RE_HTTP_DW4], _parsebuf, 10, &_pmatch[0], 0);
   if (!check) 
   {
      QString str;
      if (_pmatch[1].rm_so != -1) 
      {
	 // match 1: transfer successfully completed
	 int l = matchStr(1).toInt();
	 kdWarning(l != len(), D_PRS) << name() << " length " << l << "!=" << len() << endl;
	 if (l != len())
	    setLen(l);
	 if (partial() < len())
	    opDataRead(len() - partial());
	 _localStatus = HTTP_DOWNLOAD_END;
	 //opRunEnd();
      } 
      else if (_pmatch[4].rm_so != -1) 
      {
	 // match 5: error: retry or giveup
	 opError(matchStr(4));
	 str = matchStr(5);
	 if (str == "Giving up")
	    // give up 
	    _localError = ERR_GIVEUP;
	 else 
	    // retry
	    _localError = ERR_RETRY;
      }
      return MTC_OK;
   }
   return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

/////////////// config entries

bool KTransferWgetImpl::getPassiveFtp() const 
{
   return useGlobal() ? global()->getPassiveFtp() : _cfg_passiveFtp;
}

void KTransferWgetImpl::setPassiveFtp(bool v)
{
   if (_cfg_passiveFtp != v)
   {
      _cfg_passiveFtp = v;
      dom().setAttribute("PassiveFtp", _cfg_passiveFtp);
      setMod(MDI_TRW_CFG_PASSIVE);
      root()->setModified();
   }
}

QString KTransferWgetImpl::getCmdPath() const 
{
   return useGlobal() ? global()->getCmdPath() : _cfg_cmdPath;
}

void KTransferWgetImpl::setCmdPath(const QString &v)
{
   if (_cfg_cmdPath != v)
   {
      _cfg_cmdPath = v;
      dom().setAttribute("CommandPath", _cfg_cmdPath);
      setMod(MDI_TRW_CFG_CMDPATH);
      root()->setModified();
   }
}

bool KTransferWgetImpl::getHTTPCache() const 
{
   return useGlobal() ? global()->getHTTPCache() : _cfg_HTTPCache;
}

void KTransferWgetImpl::setHTTPCache(bool v)
{
   if (_cfg_HTTPCache != v)
   {
      _cfg_HTTPCache = v;
      dom().setAttribute("HTTPCache", _cfg_HTTPCache);
      setMod(MDI_TRW_CFG_HTTPCACHE);
      root()->setModified();
   }
}

int KTransferWgetImpl::getReadTimeout() const 
{
   return useGlobal() ? global()->getReadTimeout() : _cfg_readTimeout;
}

void KTransferWgetImpl::setReadTimeout(int v)
{
   if (_cfg_readTimeout != v)
   {
      _cfg_readTimeout = v;
      dom().setAttribute("ReadTimeout", _cfg_readTimeout);
      setMod(MDI_TRW_CFG_RDTIMEOUT);
      root()->setModified();
   }
}

bool KTransferWgetImpl::getProxy() const 
{
   return useGlobal() ? global()->getProxy() : _cfg_proxy;
}

void KTransferWgetImpl::setProxy(bool v)
{
   if (_cfg_proxy != v)
   {
      _cfg_proxy = v;
      dom().setAttribute("Proxy", _cfg_proxy);
      setMod(MDI_TRW_CFG_PROXY);
      root()->setModified();
   }
}

QString KTransferWgetImpl::getProxyFTP() const 
{
   return useGlobal() ? global()->getProxyFTP() : _cfg_proxyFTP;
}

void KTransferWgetImpl::setProxyFTP(const QString &v)
{
   if (_cfg_proxyFTP != v)
   {
      _cfg_proxyFTP = v;
      dom().setAttribute("FTPProxy", _cfg_proxyFTP);
      setMod(MDI_TRW_CFG_PROXYFTP);
      root()->setModified();
   }
}

int KTransferWgetImpl::getProxyPortFTP() const 
{
   return useGlobal() ? global()->getProxyPortFTP() : _cfg_proxyPortFTP;
}

void KTransferWgetImpl::setProxyPortFTP(int v)
{
   if (_cfg_proxyPortFTP != v)
   {
      _cfg_proxyPortFTP = v;
      dom().setAttribute("FTPProxyPort", _cfg_proxyPortFTP);
      setMod(MDI_TRW_CFG_PROXYPORTFTP);
      root()->setModified();
   }
}

QString KTransferWgetImpl::getProxyHTTP() const 
{
   return useGlobal() ? global()->getProxyHTTP() : _cfg_proxyHTTP;
}

void KTransferWgetImpl::setProxyHTTP(const QString &v)
{
   if (_cfg_proxyHTTP != v)
   {
      _cfg_proxyHTTP = v;
      dom().setAttribute("HTTPProxy", _cfg_proxyHTTP);
      setMod(MDI_TRW_CFG_PROXYHTTP);
      root()->setModified();
   }
}

int KTransferWgetImpl::getProxyPortHTTP() const 
{
   return useGlobal() ? global()->getProxyPortHTTP() : _cfg_proxyPortHTTP;
}

void KTransferWgetImpl::setProxyPortHTTP(int v)
{
   if (_cfg_proxyPortHTTP != v)
   {
      _cfg_proxyPortHTTP = v;
      dom().setAttribute("HTTPProxyPort", _cfg_proxyPortHTTP);
      setMod(MDI_TRW_CFG_PROXYPORTHTTP);
      root()->setModified();
   }
}

const QStringList & KTransferWgetImpl::getNoProxyList() const 
{
   return useGlobal() ? global()->getNoProxyList() : _cfg_noProxy;
}

void KTransferWgetImpl::setNoProxyList(const QStringList &v)
{
   if (_cfg_noProxy != v)
   {
      _cfg_noProxy = v;
      dom().setAttribute("NoProxy", _cfg_noProxy.join(","));
      setMod(MDI_TRW_CFG_NOPROXYLST);
      root()->setModified();
   }
}

TransferSpeed KTransferWgetImpl::getSpeed() const 
{
   return useGlobal() ? global()->getSpeed() : _cfg_speed;
}

void KTransferWgetImpl::setSpeed(TransferSpeed v)
{
   if (_cfg_speed != v)
   {
      _cfg_speed = v;
      dom().setAttribute("Speed", _cfg_speed);
      setMod(MDI_TRW_CFG_SPEED);
      root()->setModified();
   }
}

bool KTransferWgetImpl::getIgnoreLength() const 
{
   return useGlobal() ? global()->getIgnoreLength() : _cfg_ignoreLen;
}

void KTransferWgetImpl::setIgnoreLength(bool v)
{
   if (_cfg_ignoreLen != v)
   {
      _cfg_ignoreLen = v;
      dom().setAttribute("IgnoreLen", _cfg_ignoreLen);
      setMod(MDI_TRW_CFG_IGNLEN);
      root()->setModified();
   }
}

/////////////// SLOTS

void KTransferWgetImpl::slotProcessOutput(KProcess *, char *buffer, int buflen) 
{
   QCString cdata(buffer, buflen+1);
   _parsebuf += cdata;
   opProcessOutput(cdata);
   (this->*_parseCmd)();
}

void KTransferWgetImpl::slotProcessExited(KProcess *) 
{
   _processExit = true;
   (this->*_parseCmd)();
   opProcessEnd(process.exitStatus());
} 

#include "ktransferwgetimpl.moc"

#endif
