/***************************************************************************
			  ktransferimpl.h  -  description
                             -------------------
    begin                : Mon Oct 16 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.17 $
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KTRANSFERIMPL_H
#define KTRANSFERIMPL_H

#include <qdatetime.h>
#include <kurl.h>
#include <qcstring.h>
#include <qfile.h>
#include <qtimer.h>
//#include "ktransfer.h"
#include "kobjectimpl.h"

class KTManagerImpl;
class KTransfer;

/**
 * @short Abstract base class for transfer implementations.
 *
 * @author Sergio Moretti
 *
 * @version $Id
 */

class KTransferImpl : public KObjectImpl
{
   Q_OBJECT

   friend class KTransfer;

public:
   static const char DOCID[];
   static const char DOCTYPE[];

   virtual ~KTransferImpl();

   /** document name */
   virtual const char * docId() const { return DOCID; }

   /** dom document type */
   virtual const char * docType() const { return DOCTYPE; }

   // references to other objects

   /** reference to global instance */
   const KTransferImpl * global() const;

   /** length of remote file, it can be 0 if length is unknown or not yet
    * retrieved
    */
   int len() const { return _len; }

   /** length of partial retrieved data */
   int partial() const { return _partial; }

   /** percent */
   int percent() const { return (len() != 0 ? int(100.0 * partial() / len()) : 0); }

   /** number of retries done */
   int retry() const { return _retry; }

   /** get the state of the transfer
    * @li TRN_READY	ready to run
    * @li TRN_START	transfer is starting
    * @li TRN_CONNECT	connection is established
    * @li TRN_DOWNLOAD	downloading data
    * @li TRN_END_OK	transfer is finished without error, this doesn't mean
    *			that download is complete
    * @li TRN_END_ERROR	transfer is finished due to an error
    * @li TRN_END_FATAL	a fatal error has occurred
    * @li TRN_END_KILL	transfer has been killed by user
    * @li TRN_END_READY	transfer ready to resume / retry
    */
   TrnState state() const { return _state; }

   /** set transfer state */
   void setState(TrnState s);

   /** get the resume state of the transfer
    * @li RMS_UNKNOWN	resume state is unknown
    * @li RSM_CHECKING	resume state is being checked
    * @li RSM_CHECKFAIL	resume state check is failed, so the state is unknown
    * @li RSM_YES	transfer resumable
    * @li RSM_NO	transfer unresumable
    */
   RsmState rsmState() const { return _rsmState; }

   /** get the istantaneaous bandwidth */
   int bandwidth() const { return _bandwidth; }

   /** get the medium bandwidth */
   int mediumBandwidth() const { return _mediumBandwidth; }

   /** get start time */
   QTime startTime() const { return _timeStart; }

   /* get state converted in string form */
   QString stateStr() const { return stateStr(state()); }
   /** state string */
   static QString stateStr(TrnState s);

   /** get transfer progress converted in string form */
   QString progressStr() const;

   /** get connection state converted in string form */
   QString connectionStr() const;

   /** get the resumable state string: Yes, No, ??? */
   QString resumeStr() const { return resumeStr(rsmState()); }
   /** resume state string short */
   static QString resumeStr(RsmState s);
   /** resume state string long */
   static QString resumeString(RsmState s);

   /** get the local file */
   const KURL & local() const { return _local; }

   /** get the remote url */
   const KURL & remote() const { return _remote; }

   /** get the temporary file */
   const KURL & tmp() const { return _tmp; }

   /** get number of resumes */
   int resumed() const { return _resumed; }

   /** set number of resumes */
   void setResumed(int r);

   // OPERATIONS on transfer

   /** check if protocol is supported */
   virtual bool isProtocolSupported(const QString &proto) const = 0;

   /** true if transfer is complete */
   bool isComplete() const;

   /** check if transfer is active */
   virtual bool isRunning() const;

   /** check if transfer is finished */
   virtual bool isFinished() const;

   /** check if transfer is ready */
   virtual bool isReady() const
      { return (state() == TRN_READY || state() == TRN_END_READY); }

   /** check if the transfer can be started */
   virtual bool isRunnable(bool automatic) const;

   /** check if the transfer can be started now or in the future */
   virtual bool isRunnableInFuture(bool dLock, bool automatic) const;

   /** check for transfer resume capability */
   bool isResumable() const { return rsmState() == RSM_YES; }

   /** download estimated time */
   QTime estTime();

   /** guess the name of the temporary file based on rmt or lcl urls */
   virtual KURL getTempFile(const KURL &rmt, const KURL &lcl) const;

   /** start the transfer
    * @returns false on errors
    */
   virtual bool start();

   /** start the transfer doing the check for resuming
    * @returns false on errors
    */
   virtual bool startCheckResume();

   /**
    * kill the transfer
    * @param wait	wait for transfer to finish
    * @returns false on errors
    */
   virtual bool kill(bool wait = false);

   /**
    * kill the transfer, set state to TRN_END_OK, decrease priority
    */
   virtual bool stop(bool wait = false);

   /** clear transfer state: erase tmpfile */
   void clear();

   /** reset transfer state to ready */
   void resetState();

   virtual void runPeriodically();

   // CONFIG

   bool getCheckResume() const;
   void setCheckResume(bool c);
   int getMaxResume() const;
   void setMaxResume(int r);
   bool getAutoResume() const;
   void setAutoResume(bool m);
   int getWaitResume() const;
   void setWaitResume(int w);
   int getMaxRetry() const;
   void setMaxRetry(int m);
   int getWaitRetry() const;
   void setWaitRetry(int w);
   bool getLogProcess() const;
   void setLogProcess(bool l);

protected:
   KTransferImpl(int type);

   /** load state from dom */
   virtual void loadData();

   /** check and update state on loading */
   virtual void checkState();

   /** set transfer status */
   void setRsmState(RsmState s);

   void setLocal(const KURL &l);

   void setRemote(const KURL &r);

   void setTmp(const KURL &t);

   void setLen(int l);

   void setPartial(int p);

   void setRetry(int r);

   /** calculate bandwidth */
   void calcBandwidth();

   /** raise a fatal error */
   void raiseFatalError();

   /** move temporay file to final file */
   void moveTmp();

   /** process is being killed */
   bool killed() const { return _killed; }

   // abstract operations for derived classes

   /** send a start request to the transfer process */
   virtual bool processStart() = 0;

   /** send a stop request to transfer process */
   virtual bool processKill() = 0;

   /** prepare for running a check resume */
   virtual void processCheckResumeBegin() = 0;

   /** restore state from a check resume run */
   virtual void processCheckResumeEnd() = 0;

   // operations that derived class have to call to maintain the state

   /** called when process start */
   virtual void opProcessStart();

   /** called when process end */
   virtual void opProcessEnd(int exitStatus);

   /** called when connection is up */
   virtual void opConnect();

   /** called when download start */
   virtual void opDownload();

   /** called when data is read */
   virtual void opDataRead(int datalen);

   /** called when resume state change */
   virtual void opCheckResume(RsmState newState);

   /** called when an error occurred */
   virtual void opError(const QString &err);

   /** called when a fatal error occurred (must kill the transfer */
   virtual void opFatalError(const QString &err);

   /** called when process send output */
   virtual void opProcessOutput(const QString &data);

   virtual int cmpPriorityShort(const KObjectImpl *obj) const;
   virtual int cmpPriorityLong(const KObjectImpl *obj) const;

protected slots:
   /** called when timeout expires */
   void slotTimeout();
   void slotKill();

signals:
   /** forward process output */
   void sigProcessOutput(const QString &data);
   /** emit percent */
   void sigPercent(int);

private: // Private state attributes
   /** url to dowload */
   KURL _remote;
   /** write data to file */
   KURL _local;
   /** tmp file */
   KURL _tmp;
   /** state of the transfer */
   TrnState _state;
   /** resume status */
   RsmState _rsmState;
   /** num of resuming */
   int _resumed;
   /** time to wait before resuming */
   QTimer _wait;
   /** time of process start */
   QTime _timeStart;
   /** last bytes read */
   int _lastBytesRead;
   /** starting from */
   int _lastBytesStart;
   /** medium bandwidth */
   int _mediumBandwidth;
   /** instantaneaous bandwidth */
   int _bandwidth;
   /** estimate time */
   QTime _timeEst;
   /** interval between sample, for bandwidth calc */
   QTime _lastSampleTime;
   /** file descriptor of log file */
   QFile _logFile;
   /** true if process killed by user */
   bool _killed;
   /** true if process is being stopped */
   bool _stopped;
   /** number of retries */
   int _retry;
   /** len of file to download */
   int _len;
   /** partial downloaded */
   int _partial;
   /** true when running in check resume mode */
   int _checkResume;

   // config attributes

   /** check for resumable */
   bool _cfg_checkResume;
   /** max num of retries */
   int _cfg_maxResume;
   /** autoresume after a broken connection */
   bool _cfg_autoResume;
   /** seconds to wait before resuming */
   int _cfg_waitResume;
   /** max num of retries */
   int _cfg_maxRetry;
   /** wait before retrying */
   int _cfg_waitRetry;
   /** enable logging of process output */
   bool _cfg_logProcess;
};

#endif

/* Local Variables: */
/* mode: c++ */
/* End: */
