//*************************************************************************
//* methods for class XObject                                             *
//*                                                                       *
//* pixmap management for objects on the screen                           *
//*                                                                       *
//* Andreas Rostin                                                        *
//* 15.03.99                                                              *
//*************************************************************************
#include <stdio.h>

#include <qpaintdevice.h>
#include <qpainter.h>
#include <qpixmap.h>

#include "xobject.h"
#include "grid.h"
#include "klogic.h"

//=====================================================================
//== constructor/destructor/initializing                             ==
//=====================================================================
XObject::XObject( int x, int y )
{
	img 			= QPixmap( 1, 1 );
	m_ptOffset		= QPoint( 0, VOFFSET );
	m_bIsSelected 	= false;

	// creating the grid as singleton
	Grid();

	initPos( QPoint( x, y ) );
}
//---------------------------------------------------------------------
//-- not needed anymore
//-- only for compatibility with xdevice.cpp:557
//-- only for compatibility with oldhandler.cpp:564
void XObject::initPos( int x, int y )
{
	initPos( QPoint( x, y ) );
}
//---------------------------------------------------------------------
//-- set initial position on the screen
void XObject::initPos( QPoint ptTopLeft )
{
	m_ptDelta = QPoint();

	// move rectangle contrary to the x and y axis
	ptTopLeft = Grid::getAligned( ptTopLeft ) - m_ptOffset;
	newPos.moveTopLeft( ptTopLeft );

	oldPos = newPos;
}
//---------------------------------------------------------------------

//=====================================================================
//== object movement                                                 ==
//=====================================================================
QRect XObject::getPos()
{
	return newPos;
}
//---------------------------------------------------------------------
QRect XObject::getOldPos()
{
	return oldPos;
}
//---------------------------------------------------------------------
//-- move on the screen (used by selection)
void XObject::addPos( int x, int y )
{
	newPos.moveBy(x, y);
}
//---------------------------------------------------------------------
//-- move on the screen
//-- Return value: true  object was moved  
//-- 			 : false indicates that no movement took place
bool XObject::setPos( QPoint ptMouse )
{
	bool bReturn = false;

	// requested top left position is actual mouse position minus
	// mouse-object-distance at the time of the mouse click
	QPoint ptTopLeft = ptMouse - m_ptDelta;

	int w = newPos.width();
	int h = newPos.height();

	// Is there a visible distance between actual and
	// requested object position ?
	bReturn = ( newPos.topLeft() != Grid::getAligned( ptTopLeft ) );

	if( bReturn )
	{
		// yes, really move the object

		// prevents hitting the screen boundaries, i.e. leaves a
		// visible gap ( gap < GRID ) between object and window border
		if( ptTopLeft.x() + m_ptOffset.x() < Grid::GRID )
			ptTopLeft.setX( Grid::GRID + m_ptOffset.x() );
		if( ptTopLeft.x() - m_ptOffset.x() + w > Global::Screen::VIRT_SCREEN_SIZE_X )
			ptTopLeft.setX( Global::Screen::VIRT_SCREEN_SIZE_X - w - m_ptOffset.x() );

		if( ptTopLeft.y() + m_ptOffset.y() < Grid::GRID )
			ptTopLeft.setY( Grid::GRID + m_ptOffset.y() );
		if( ptTopLeft.y() - m_ptOffset.y() + h > Global::Screen::VIRT_SCREEN_SIZE_Y )
			ptTopLeft.setY( Global::Screen::VIRT_SCREEN_SIZE_Y - h);

		ptTopLeft = Grid::getAligned( ptTopLeft ) - m_ptOffset;
		newPos.moveTopLeft( ptTopLeft );
	}

	return bReturn;
}
//---------------------------------------------------------------------

//=====================================================================
//== undo movement                                                   ==
//=====================================================================
void XObject::setOrigin()
{
	origin = newPos.topLeft();
}
//---------------------------------------------------------------------
void XObject::toOrigin()
{
	newPos.moveTopLeft( origin );
}
//---------------------------------------------------------------------
bool  XObject::isOrigin( QPoint pt )
{
	QRect r( origin, newPos.size() );

	return r.contains( pt );
}
//---------------------------------------------------------------------

//=====================================================================
//== vertical/horizontal image offset                                ==
//=====================================================================
//-- get horizontal offset of image
int XObject::getHOffset()
{
	return m_ptOffset.x();
}
//---------------------------------------------------------------------
//-- get vertical offset of image
int XObject::getVOffset()
{
	return m_ptOffset.y();
}
//---------------------------------------------------------------------
//-- set horizontal offset of image
void XObject::setHOffset( int nHOffset )
{
	setOffset( QPoint( nHOffset, m_ptOffset.y() ) );
}
//---------------------------------------------------------------------
//-- set vertical offset of image
void XObject::setVOffset( int nVOffset )
{
	setOffset( QPoint( m_ptOffset.x(), nVOffset ) );
}
//---------------------------------------------------------------------
//-- adjust positioning variables if offset changes
void XObject::setOffset( QPoint ptNewOffset )
{
	QPoint ptDelta = m_ptOffset - ptNewOffset;

	newPos.moveBy( ptDelta.x(), ptDelta.y() );
	oldPos.moveBy( ptDelta.x(), ptDelta.y() );

	m_ptOffset = ptNewOffset;
}
//---------------------------------------------------------------------

//=====================================================================
//== pixmap methods                                                  ==
//=====================================================================
void XObject::drawImage( QPaintDevice*, QPainter* p )
{
	p->drawPixmap( newPos, img );

	if( m_bIsSelected )
	{
		p->setPen( Qt::green );
		p->setBrush( Qt::NoBrush);
		p->drawRect( newPos );
	}
}
//---------------------------------------------------------------------
void XObject::setImage()
{
	QSize oImageSize( img.rect().size() );

	newPos.setSize( oImageSize );
	oldPos.setSize( oImageSize );
}
//---------------------------------------------------------------------
QPixmap* XObject::getImage()
{
	return &img;
}
//---------------------------------------------------------------------
void XObject::erase( QPainter* p )
{
	oldPos.setRight(  oldPos.right()  + Grid::GRID );
	oldPos.setBottom( oldPos.bottom() + Grid::GRID );

	Grid::draw( p, oldPos );

	oldPos = newPos;
}
//---------------------------------------------------------------------

//=====================================================================
//== selection methods                                               ==
//=====================================================================
void XObject::select( bool bIsSelected )
{
	m_bIsSelected = bIsSelected;
}
//---------------------------------------------------------------------
bool XObject::isSelected()
{
	return m_bIsSelected;
}
//---------------------------------------------------------------------

//=====================================================================
//==                                                                 ==
//=====================================================================
bool XObject::contains( QRect pos )
{
	return newPos.intersects( pos );
}
//---------------------------------------------------------------------
bool XObject::contains( QPoint ptMouse, int nLeftSize, int nRightSize )
{
	bool 	bReturn 	= false;

	// subtract input- and output-region from image-size
	// rcMiddle is the region for mouse click detection
	QRect 	rcMiddle	= QRect(
			QPoint( newPos.x() + nLeftSize,
				newPos.y() ),
			QPoint( newPos.right() - nRightSize,
				newPos.bottom() )
			);

	bReturn = rcMiddle.contains( ptMouse );

	if( bReturn )
	{
		// calculate delta's (mouse <-> object)
		m_ptDelta = ptMouse - newPos.topLeft();
	}

	return bReturn;
}
//---------------------------------------------------------------------



