#include <stdlib.h>

#include <qstring.h>
#include <qxml.h>
#include <qbuffer.h>

#include "klogic.h"
#include "circuit.h"
#include "xdevice.h"
#include "deviceTypes.h"
#include "xwire.h"
#include "xmlExport.h"
#include "xmlImport.h"
#include "xmlSymbolMap.h"
#include "klogicList.h"

/*******************************************/
/* XML Export Interface                    */
/*                                         */
/* Andreas Rostin 23.08.2001               */
/*******************************************/
const char * XmlObject::ATT_CIRCUIT = "CIRCUIT";
const char * XmlObject::ATT_SUBCIRCUIT = "SUBCIRCUIT";
const char * XmlObject::ATT_LIBRARY = "LIBRARY";

XmlObject::XmlObject(QString name)
	:XmlField(name, 0)
{
	dx = 0;
	dy = 0;
	_valid = true;
}

XmlObject::XmlObject(QString name, int level, int _dx, int _dy)
	:XmlField(name, level)
{
	dx = _dx;
	dy = _dy;
	_valid = false;
}

// virtual
XmlObject::~XmlObject()
{
}

// private virtual; to be redefined..
void XmlObject::createXML(bool)
{
	return;
}

// return value can be both, a circuit or a device
// use XDevice::devIsNet() to retrieve Circuit pointer from XDevice
XDevice * XmlObject::createInstance(Circuit *_net, int dx, int dy)
{
	QString content = getField();

	// xml reader does not work with string based textstreams!
	QBuffer f;
	f.open(IO_WriteOnly|IO_Truncate);
	QTextStream ts3(&f);
	ts3 << content;
	f.close();
	f.open(IO_ReadOnly);
	QTextStream ts2(&f);
	return XMLImportHandler::import(_net, dx, dy, ts2);
}

void XmlObject::setValid(bool __valid)
{
	_valid = __valid;
}

bool XmlObject::isValid()
{
	return _valid;
}

/*******************************************/
/* XML representation of a circuit         */
/*                                         */
/* because a circuit can consume a lot of  */
/* memory, this export is mainly done by   */
/* the Circuit itself!                  */
/* see klogicIO::writeXML() for details    */
/*                                         */
/* Andreas Rostin 24.08.2001               */
/*******************************************/
XmlNet::XmlNet(Circuit *_net, bool selected, int level, int dx, int dy)
	: XmlObject(XmlSymbolMap::m_oMap[XmlSymbol::CIRCUIT], level, dx, dy)
{
	net = _net;
	createXML(selected);
}

XmlNet::XmlNet()
	: XmlObject(XmlSymbolMap::m_oMap[XmlSymbol::CIRCUIT], 0, 0, 0)
{
	net = (Circuit *)NULL;
}

void XmlNet::createXML(bool selected)
{
	if (selected && !net->isSelected()) return;
	setValid(true);
	setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::ID], net->getID());
	addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::FUNCTION], net->type());
	addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::SIZE], net->size());
	int x = net->getPos().x() - net->getHOffset() - dx;
	if (x < 0) x = 0;
	int y = net->getPos().y() + net->getVOffset() - dy;
	if (y < 0) y = 0;
	addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::XPOS], x);
	addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::YPOS], y);

	XmlDevice dev(net, selected, -1, dx, dy);
	setContent(dev.getFieldContent());
}

/*******************************************/
/* XML representation of a connection      */
/*     between devices and wires           */
/*                                         */
/* Andreas Rostin 24.08.2001               */
/*******************************************/
XmlDeviceConnection::XmlDeviceConnection(XWire *_wire, bool selected, int level, int dx, int dy)
	: XmlObject(XmlSymbolMap::m_oMap[XmlSymbol::DEV_CONN], level, dx, dy)
{
	wire = _wire;
	createXML(selected);
}

void XmlDeviceConnection::createXML(bool selected)
{	int inverter;

	if (selected && !wire->isSelected()) return;

	QPoint pi1 = wire->getInputPosition(0, selected);
	QPoint pi2 = wire->getInputPosition(1, selected);
	QPoint po1 = wire->getOutputPosition(0, selected);
	QPoint po2 = wire->getOutputPosition(1, selected);
	QPoint pt;
	XDevice *dev;

	// maximum of 2 points
	if (!pi1.isNull() || !pi2.isNull() || !po1.isNull() || !po2.isNull()) {
		setValid(true);
		setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::ID], wire->getID());

		if (!pi1.isNull()) {
			dev = wire->inputDevice(0);
			inverter = wire->inputIsInverted(dev);
			pt.setX(pi1.x() - dx);
			pt.setY(pi1.y() - dy);
			setField(dev, pt, inverter);
		}
		if (!pi2.isNull()) {
			dev = wire->inputDevice(1);
			inverter = wire->inputIsInverted(dev);
			pt.setX(pi2.x() - dx);
			pt.setY(pi2.y() - dy);
			setField(dev, pt, inverter);
		}
		if (!po1.isNull()) {
			dev = wire->outputDevice(0);
			inverter = wire->outputIsInverted(dev);
			pt.setX(po1.x() - dx);
			pt.setY(po1.y() - dy);
			setField(dev, pt, inverter);
		}
		if (!po2.isNull()) {
			dev = wire->outputDevice(1);
			inverter = wire->outputIsInverted(dev);
			pt.setX(po2.x() - dx);
			pt.setY(po2.y() - dy);
			setField(dev, pt, inverter);
		}
	}
}

void XmlDeviceConnection::setField(XDevice *dev, QPoint pt, int inverter)
{	XmlField field;

	field.set(XmlSymbolMap::m_oMap[XmlSymbol::POS], pt);
	field.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::ID], dev->getID());
	field.addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::INV], inverter);
	addContent(field.getField());
}

/*******************************************/
/* XML representation of a connection      */
/*            between wires                */
/*                                         */
/* Andreas Rostin 24.08.2001               */
/*******************************************/
XmlWireConnection::XmlWireConnection(XWire *_wire, bool selected, int level, int dx, int dy)
	: XmlObject(XmlSymbolMap::m_oMap[XmlSymbol::WIRE_CONN], level, dx, dy)
{
	wire = _wire;
	createXML(selected);
}

void XmlWireConnection::createXML(bool selected)
{	XmlField field;

	if (selected && !wire->isSelected()) return;

	QPoint pi = wire->getIOPosition(1, selected);
	QPoint po = wire->getIOPosition(0, selected);
	QPoint pt;

	if ((!pi.isNull()) || (!po.isNull())) {
		setValid(true);
		setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::ID], wire->getID());

		if (!pi.isNull()) {
			pt.setX(pi.x() - dx);
			pt.setY(pi.y() - dy);
			
			field.set(XmlSymbolMap::m_oMap[XmlSymbol::POS], pt);
			field.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::ID], wire->ioWire(1)->getID());
			addContent(field.getField());
		}

		if (!po.isNull()) {
			pt.setX(po.x() - dx);
			pt.setY(po.y() - dy);
			field.set(XmlSymbolMap::m_oMap[XmlSymbol::POS], pt);
			field.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::ID], wire->ioWire(0)->getID());
			addContent(field.getField());
		}
	}
}

/*******************************************/
/* XML representation of a XWire           */
/*                                         */
/* Andreas Rostin 23.08.2001               */
/*******************************************/
XmlWire::XmlWire(XWire *_wire, bool selected, int level, int dx, int dy)
	: XmlObject(XmlSymbolMap::m_oMap[XmlSymbol::WIRE], level, dx, dy)
{
	wire = _wire;
	createXML(selected);
}

void XmlWire::createXML(bool selected)
{	QString _content;

	if (selected && !wire->isSelected()) return;
	setValid(true);
 
	// wire properties
	setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::ID], wire->getID());

	// wire content
	wire->setExportSelected(selected);
	wire->setExportDelta(dx, dy);
	*wire >> _content;

	setContent(_content);
}

/*******************************************/
/* XML representation of a XDevice         */
/*                                         */
/* Andreas Rostin 23.08.2001               */
/*******************************************/
XmlDevice::XmlDevice(XDevice *_dev, bool selected, int level, int dx, int dy)
	: XmlObject(XmlSymbolMap::m_oMap[XmlSymbol::DEVICE], level, dx, dy)
{
	dev = _dev;
	createXML(selected);
}

XmlDevice::XmlDevice()
	: XmlObject(XmlSymbolMap::m_oMap[XmlSymbol::DEVICE], 0, 0, 0)
{
	dev = (Circuit *)NULL;
}

void XmlDevice::createXML(bool selected)
{	XmlField field;
	QString text;

	if (selected && !dev->isSelected()) return;
	setValid(true);

	// device attributes
	setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::ID], dev->getID());
	addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::FUNCTION], dev->type());
	addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::SIZE], dev->size());
	addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::XPOS], dev->getPos().x() - dev->getHOffset() - dx);
	addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::YPOS], dev->getPos().y() + dev->getVOffset() - dy);

	// device content 
	text = dev->getName();
	field.set(XmlSymbolMap::m_oMap[XmlSymbol::TEXT], XmlField::quote(text));
	addContent(field.getField());

	if (dev->nameDisplayed()) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::TEXT_DISP], "");
		addContent(field.getField());
	}

	if (dev->type() != DeviceType::fNET &&
	    dev->type() != DeviceType::fLEDred &&
	    dev->type() != DeviceType::fLEDblue &&
	    dev->type() != DeviceType::fLEDgreen &&
	    dev->type() != DeviceType::fLEDyellow &&
	    dev->delay() != Device::DEFAULT_DELAY) { 
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::DELAY], dev->delay());
		addContent(field.getField());
	}

	if (dev->undef() != Device::DEFAULT_UNDEF) { 
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::UNDEF], dev->undef());
		addContent(field.getField());
	}
 
	if (dev->hasStaticInput()) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::STAT], dev->output());
		addContent(field.getField());
	}
 
	// flipflop properties
	if (dev->type() != DeviceType::fOSC && dev->hasClock()) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::CLK_TYPE], dev->clock());
		addContent(field.getField());

		field.set(XmlSymbolMap::m_oMap[XmlSymbol::MASTER], dev->hasMaster());
		addContent(field.getField());
	}

	// switch properties
	if (dev->type() == DeviceType::fSWI && dev->isToggle()) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::PUSH]);
		addContent(field.getField());
	}

	// width property: possible only for equ, net
	if ((dev->type() == DeviceType::fEQU || dev->type() == DeviceType::fNET) && dev->width() != XDevice::DEFAULT_WIDTH) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::WIDTH], dev->width());
		addContent(field.getField());
	}

	// default inverted output
	if (dev->isInverted()) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::INV]);
		addContent(field.getField());
	}

	// oscillator properties
	if (dev->type() == DeviceType::fOSC) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::CLK_TYPE], dev->clock());
		addContent(field.getField());

		if (dev->oszOn() != Device::DEFAULT_OSC_CNT) {
			field.set(XmlSymbolMap::m_oMap[XmlSymbol::OSC_ON], dev->oszOn());
			addContent(field.getField());
		}
	 
		if (dev->oszOff() != Device::DEFAULT_OSC_CNT) {
			field.set(XmlSymbolMap::m_oMap[XmlSymbol::OSC_OFF], dev->oszOff());
			addContent(field.getField());
		}
	 
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::OSC_STEP], dev->getCurrOszStep());
		addContent(field.getField());
	}

	// tristate flag
	if (dev->isTristate()) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::TRI], dev->getTristateControlPos());
		addContent(field.getField());
	}

	// named output
	if ((dev->type() == DeviceType::fEQU || dev->type() == DeviceType::fNET) && dev->hasNamedOutput()) {
		KlogicList<Calculator> *lo = dev->getNamedORef()->First()->Next();
		while (lo) {
			if (dev->type() == DeviceType::fEQU) {
				text = lo->Get()->getEquation();
				field.set(XmlSymbolMap::m_oMap[XmlSymbol::OUT_EQU], XmlField::quote(text));
				field.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::POS], lo->Get()->getPosition());
				field.addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::INTERN], lo->Get()->isInternal());
				text = lo->getText();
				field.addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::NAME], XmlField::quote(text));
			} else {
				field.set(XmlSymbolMap::m_oMap[XmlSymbol::OUT_ID], lo->getID1());
				field.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::POS], lo->Get()->getPosition());
			}

			addContent(field.getField());
			lo = lo->Next();
		}
	}

	// named input 
	if ((dev->type() == DeviceType::fEQU || dev->type() == DeviceType::fNET) && dev->hasNamedInput()) {
		KlogicList<Value> *li = dev->getNamedIRef()->First();
		while(li) {
			if (dev->type() == DeviceType::fEQU) {
				text = li->getText();
				field.set(XmlSymbolMap::m_oMap[XmlSymbol::IN_NAME], XmlField::quote(text));
				field.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::POS], li->Get()->getPosition());
				field.addAttribute(XmlSymbolMap::m_oMap[XmlSymbol::INTERN], li->Get()->isInternal());
			} else {
				field.set(XmlSymbolMap::m_oMap[XmlSymbol::IN_ID], li->getID1());
				field.setAttribute(XmlSymbolMap::m_oMap[XmlSymbol::POS], li->Get()->getPosition());
			}

			addContent(field.getField());
			li = li->Next();
		}
	}

	// ram content
	if (dev->type() == DeviceType::fRAM) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::RAM], dev->RAMSize(), dev->getRAMStorageCopy());
		addContent(field.getField());
	}

	// bus type
	if (dev->type() == DeviceType::fBUS) {
		field.set(XmlSymbolMap::m_oMap[XmlSymbol::BUSTYPE], dev->isMux());
		addContent(field.getField());
	}
}

