/*************************************************/
/* methods for class XDevice                     */
/*                                               */
/* visible Device                                */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qpainter.h>

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "mainw.h"
#include "xwire.h"
#include "xdevice.h"
#include "klogic.h"
#include "calculator.h"
#include "value.h"
#include "deviceTypes.h"
#include "busOutputArray.h"
#include "deviceText.h"
#include "grid.h"

/***************************************************/
/* static methods of XDevice class                 */
/***************************************************/
int XDevice::instance = 0;
int XDevice::STATdef_size = XDevice::DEFAULT_SIZE;
bool XDevice::STATdef_inverted = false;
bool XDevice::STATdisplay_name = false;
QFont XDevice::STATfont = QFont(FONTTYPE, FONTSIZE, FONTAPPEARANCE);
QString XDevice::STATfontFamily = FONTTYPE;
int XDevice::STATfontSize = FONTSIZE;
int XDevice::STATSymbolSet = XDevice::SYMBOLSET_DIN40900;
QColor XDevice::BlankColor = QColor(178, 178, 178);
int XDevice::STATdefWidth = DEFAULT_WIDTH;

int XDevice::pix_loaded = 0;
QPixmap *XDevice::pxPOS = (QPixmap *)NULL;
QPixmap *XDevice::pxNEG = (QPixmap *)NULL;
QPixmap *XDevice::pxRS = (QPixmap *)NULL;
QPixmap *XDevice::pxRSC = (QPixmap *)NULL;
QPixmap *XDevice::pxRSCMF = (QPixmap *)NULL;
QPixmap *XDevice::pxRSCM = (QPixmap *)NULL;
QPixmap *XDevice::pxDM = (QPixmap *)NULL;
QPixmap *XDevice::pxD = (QPixmap *)NULL;
QPixmap *XDevice::pxJKF = (QPixmap *)NULL;
QPixmap *XDevice::pxJK = (QPixmap *)NULL;
QPixmap *XDevice::pxOUT = (QPixmap *)NULL;
QPixmap *XDevice::pxIN = (QPixmap *)NULL;
QPixmap *XDevice::pxLED0red = (QPixmap *)NULL;
QPixmap *XDevice::pxLED1red = (QPixmap *)NULL;
QPixmap *XDevice::pxLED0blue = (QPixmap *)NULL;
QPixmap *XDevice::pxLED1blue = (QPixmap *)NULL;
QPixmap *XDevice::pxLED0green = (QPixmap *)NULL;
QPixmap *XDevice::pxLED1green = (QPixmap *)NULL;
QPixmap *XDevice::pxLED0yellow = (QPixmap *)NULL;
QPixmap *XDevice::pxLED1yellow = (QPixmap *)NULL;
QPixmap *XDevice::pxSSB = (QPixmap *)NULL;
QPixmap **XDevice::pxSS = (QPixmap **)NULL;
QPixmap *XDevice::pxSWI0 = (QPixmap *)NULL;
QPixmap *XDevice::pxSWI1 = (QPixmap *)NULL;

QPixmap *XDevice::PTOPH = (QPixmap *)NULL;
QPixmap *XDevice::PMIDH = (QPixmap *)NULL;
QPixmap *XDevice::PBOTH = (QPixmap *)NULL;

QPixmap *XDevice::PTOPL = (QPixmap *)NULL;
QPixmap *XDevice::PMIDL = (QPixmap *)NULL;
QPixmap *XDevice::PBOTL = (QPixmap *)NULL;

QPixmap *XDevice::PTOP = (QPixmap *)NULL;
QPixmap *XDevice::PMID = (QPixmap *)NULL;
QPixmap *XDevice::PBOT = (QPixmap *)NULL;

QPixmap *XDevice::PTOPBlank = (QPixmap *)NULL;
QPixmap *XDevice::PMIDBlank = (QPixmap *)NULL;
QPixmap *XDevice::PBOTBlank = (QPixmap *)NULL;

QPixmap *XDevice::syAND = (QPixmap *)NULL;
QPixmap *XDevice::syOR = (QPixmap *)NULL;
QPixmap *XDevice::syXOR = (QPixmap *)NULL;
QPixmap *XDevice::syIECAND = (QPixmap *)NULL;
QPixmap *XDevice::syIECOR = (QPixmap *)NULL;
QPixmap *XDevice::syIECXOR = (QPixmap *)NULL;
QPixmap *XDevice::syIECNOT = (QPixmap *)NULL;
QPixmap *XDevice::syINV = (QPixmap *)NULL;
QPixmap *XDevice::syOSC = (QPixmap *)NULL;
QPixmap *XDevice::syNET = (QPixmap *)NULL;
QPixmap *XDevice::sySS = (QPixmap *)NULL;
QPixmap *XDevice::syMONOFLOP = (QPixmap *)NULL;
const QString XDevice::INPUT_SS_2_0 = "2^0";
const QString XDevice::INPUT_SS_2_1 = "2^1";
const QString XDevice::INPUT_SS_2_2 = "2^2";
const QString XDevice::INPUT_SS_2_3 = "2^3";
const QString XDevice::INPUT_FF_S = "S";
const QString XDevice::INPUT_FF_1S = "1S";
const QString XDevice::INPUT_FF_R = "R";
const QString XDevice::INPUT_FF_1R = "1R";
const QString XDevice::INPUT_FF_C1 = "C1";
const QString XDevice::INPUT_FF_1D = "1D";
const QString XDevice::INTERNAL_FF_MASTER_Q = "MQ";
const QString XDevice::INTERNAL_C1_PREVIOUS = "C1-1";
const QString XDevice::INPUT_TRISTATE_CONTROL = "EN";
const QString XDevice::IO_BUS_ADDRESS[8] = {"A0", "A1", "A2", "A3", "A4", "A5", "A6", "A7"};
const QString BusOutputArray::IO_BUS_DATA[8] = {"D0", "D1", "D2", "D3", "D4", "D5", "D6", "D7"};

// static method
int XDevice::defSize()
{	return STATdef_size;
}

// static method
void XDevice::setDefSize(int newi)
{	if (newi > MAXSIZE) STATdef_size = MAXSIZE;
	else if (newi < MINSIZE) STATdef_size = MINSIZE;
	else STATdef_size = newi;
}

// static method
bool XDevice::isDefInverted()
{	return STATdef_inverted;
}

// static method
void XDevice::setDefInverted(bool newi)
{	if (newi) STATdef_inverted = true;
	else STATdef_inverted = false;
}

// static method
void XDevice::displayNameGlobal(bool flag)
{
	STATdisplay_name = flag;
}

// static method
bool XDevice::nameDisplayedGlobal()
{
	return STATdisplay_name;
}

// static method
void XDevice::setFont(const QString& family)
{
	STATfontFamily = family;
	STATfont = QFont(STATfontFamily, STATfontSize, FONTAPPEARANCE);
}

// static method
QString XDevice::getFont()
{
	return STATfontFamily;
}

// static method
void XDevice::setFontSize(int new_size)
{
	STATfontSize = new_size;
	STATfont = QFont(STATfontFamily, STATfontSize, FONTAPPEARANCE);
}

// static method
int XDevice::getFontSize()
{
	return STATfontSize;
}

// static method
void XDevice::setSymbolSet(int new_type)
{
	STATSymbolSet = new_type;
}

// static method
int XDevice::getSymbolSet()
{
	return STATSymbolSet;
}

// static method
void XDevice::setDefWidth(int extended_width)
{
	STATdefWidth = extended_width;
}

// static method
int XDevice::defWidth()
{
	return STATdefWidth;
}

// static method
void XDevice::loadPix()
{	QString val;
	int __i;

	pxPOS = new QPixmap;
	pxNEG = new QPixmap;
	pxRS = new QPixmap;
	pxRSC = new QPixmap;
	pxRSCMF = new QPixmap;
	pxRSCM = new QPixmap;
	pxDM = new QPixmap;
	pxD = new QPixmap;
	pxJKF = new QPixmap;
	pxJK = new QPixmap;
	pxOUT = new QPixmap;
	pxIN = new QPixmap;
	pxLED0red = new QPixmap;
	pxLED1red = new QPixmap;
	pxLED0blue = new QPixmap;
	pxLED1blue = new QPixmap;
	pxLED0green = new QPixmap;
	pxLED1green = new QPixmap;
	pxLED0yellow = new QPixmap;
	pxLED1yellow = new QPixmap;
	pxSSB = new QPixmap;
	pxSS = (QPixmap **)malloc(sizeof(QPixmap *) * 16);
	for (__i=0;__i < 16;__i++) {
		pxSS[__i] = new QPixmap;
	}
	pxSWI0 = new QPixmap;
	pxSWI1 = new QPixmap;

	PTOPH = new QPixmap;
	PMIDH = new QPixmap;
	PBOTH = new QPixmap;

	PTOPL = new QPixmap;
	PMIDL = new QPixmap;
	PBOTL = new QPixmap;

	PTOP = new QPixmap;
	PMID = new QPixmap;
	PBOT = new QPixmap;

	PTOPBlank = new QPixmap;
	PMIDBlank = new QPixmap;
	PBOTBlank = new QPixmap;

	syAND = new QPixmap;
	syOR = new QPixmap;
	syXOR = new QPixmap;
	syIECAND = new QPixmap;
	syIECOR = new QPixmap;
	syIECXOR = new QPixmap;
	syIECNOT = new QPixmap;
	syINV = new QPixmap;
	syOSC = new QPixmap;
	syNET = new QPixmap;
	sySS = new QPixmap;
	syMONOFLOP = new QPixmap;

	pxPOS->load(MainWidget::PATH + "pxPOS.xpm");
	pxNEG->load(MainWidget::PATH + "pxNEG.xpm");
	pxRS->load(MainWidget::PATH + "pxRS.xpm");
	pxRSC->load(MainWidget::PATH + "pxRSC.xpm");
	pxRSCMF->load(MainWidget::PATH + "pxRSCMF.xpm");
	pxRSCM->load(MainWidget::PATH + "pxRSCM.xpm");
	pxDM->load(MainWidget::PATH + "pxDM.xpm");
	pxD->load(MainWidget::PATH + "pxD.xpm");
	pxJKF->load(MainWidget::PATH + "pxJKF.xpm");
	pxJK->load(MainWidget::PATH + "pxJK.xpm");
	pxOUT->load(MainWidget::PATH + "pxOUT.xpm");
	pxIN->load(MainWidget::PATH + "pxIN.xpm");
	pxLED0red->load(MainWidget::PATH + "pxLED0red.xpm");
	pxLED1red->load(MainWidget::PATH + "pxLED1red.xpm");
	pxLED0green->load(MainWidget::PATH + "pxLED0green.xpm");
	pxLED1green->load(MainWidget::PATH + "pxLED1green.xpm");
	pxLED0blue->load(MainWidget::PATH + "pxLED0blue.xpm");
	pxLED1blue->load(MainWidget::PATH + "pxLED1blue.xpm");
	pxLED0yellow->load(MainWidget::PATH + "pxLED0yellow.xpm");
	pxLED1yellow->load(MainWidget::PATH + "pxLED1yellow.xpm");
	pxSSB->load(MainWidget::PATH + "pxSS.xpm");
	for (__i=0;__i < 16;__i++) {
		val.sprintf("%d", __i);
		pxSS[__i]->load(MainWidget::PATH + "pxSSs" + val + ".xpm");
	}
	pxSWI0->load(MainWidget::PATH + "pxSwitch0.xpm");
	pxSWI1->load(MainWidget::PATH + "pxSwitch1.xpm");

	PTOPH->load(MainWidget::PATH + "TopHuge.xpm");
	PMIDH->load(MainWidget::PATH + "MidHuge.xpm");
	PBOTH->load(MainWidget::PATH + "BotHuge.xpm");

	PTOPL->load(MainWidget::PATH + "TopLarge.xpm");
	PMIDL->load(MainWidget::PATH + "MidLarge.xpm");
	PBOTL->load(MainWidget::PATH + "BotLarge.xpm");

	PTOP->load(MainWidget::PATH + "Top.xpm");
	PMID->load(MainWidget::PATH + "Mid.xpm");
	PBOT->load(MainWidget::PATH + "Bot.xpm");

	PTOPBlank->load(MainWidget::PATH + "TopBlank.xpm");
	PMIDBlank->load(MainWidget::PATH + "MidBlank.xpm");
	PBOTBlank->load(MainWidget::PATH + "BotBlank.xpm");

	syAND->load(MainWidget::PATH + "AND.xpm");
	syOR->load(MainWidget::PATH + "OR.xpm");
	syXOR->load(MainWidget::PATH + "XOR.xpm");
	syIECAND->load(MainWidget::PATH + "IEC_AND.xpm");
	syIECOR->load(MainWidget::PATH + "IEC_OR.xpm");
	syIECXOR->load(MainWidget::PATH + "IEC_XOR.xpm");
	syIECNOT->load(MainWidget::PATH + "IEC_NOT.xpm");
	syINV->load(MainWidget::PATH + "INV.xpm");
	syOSC->load(MainWidget::PATH + "OSC.xpm");
	syNET->load(MainWidget::PATH + "NET.xpm");
	sySS->load(MainWidget::PATH + "pxSSsoff.xpm");
	syMONOFLOP->load(MainWidget::PATH + "MONOFLOP.xpm");
	pix_loaded = 1;
}

/***************************************************/
/* methods of XDevice instance                     */
/***************************************************/
XDevice::XDevice(int function, int x, int y, int size, int delay, int undef, int clock)
	:XObject(x, y), DeviceInterface(function, delay, undef, clock)
{
	instance++;

	if (size >= MINSIZE && size <= MAXSIZE) {
		setSize(size);
	} else {
		if (Device::IMPORT_IGNORE_GLOBAL) {
			setSize(XDevice::DEFAULT_SIZE);
		} else {
			setSize(STATdef_size);
		}
	}

	if (Device::IMPORT_IGNORE_GLOBAL)
		deviceOutputInverted = false;
	else
		deviceOutputInverted = STATdef_inverted;

	deviceMaxInput = MAXI;
	deviceMaxOutput = MAXO;
	graph_enabled = true;
	m_oTextDev = 0;
	tristate_input_moved = 0;
	tristate_pos_import = 0;
	deviceWidth = 1;
}

// remove device: remove all inputs and outputs first
XDevice::~XDevice()
{
	instance--;

	int idx;

	// remove all inputs
	while (in_w.First()) {
		idx = in_w.First()->Get()->getInputIndex(this);
		in_w.First()->Get()->disconnectInput(idx);
	}
	in_w.Destroy();

	// remove all outputs
	while (out_w.First()) {
		idx = out_w.First()->Get()->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::~XDevice() this is unknown in a connected wire !?");
		out_w.First()->Get()->disconnectOutput(idx);
	}
	out_w.Destroy();

	if (m_oTextDev) delete m_oTextDev;
}

int XDevice::contains(QPoint pt)
{
	// take attention about input and output area of a device
	if (!deviceMaxOutput && !deviceMaxInput)
		return XObject::contains(pt, 0, 0);
	else if (!deviceMaxInput && deviceMaxOutput)
		return XObject::contains(pt, 0, XObject::XObject::IOREG);
	else if (deviceMaxInput && !deviceMaxOutput)
		 return XObject::contains(pt, XObject::XObject::IOREG, 0);
	else
		 return XObject::contains(pt, XObject::XObject::IOREG, XObject::XObject::IOREG);
}

int XDevice::contains(QRect r)
{
	return XObject::contains(r);
}

int XDevice::size()
{	return deviceSize;
}

bool XDevice::setSize(int new_s)
{	KlogicList<XWire> *lw;
	XWire *w;
	int min_size = MINSIZE;
	int maxy = 0;
	int idx;
	bool retval;

	lw = in_w.First();
	while(lw) {
		w = lw->Get();
		idx = w->getInputIndex(this);
		if (w->getInputPosition(idx).y() - getPos().y() > maxy)
			maxy = w->getInputPosition(idx).y() - getPos().y();
		lw = lw->Next();
	}
	lw = out_w.First();
	while(lw) {
		w = lw->Get();
		idx = w->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::setSize() this is unknown in a connected wire !?");
		if (w->getOutputPosition(idx).y() - getPos().y() > maxy)
			maxy = w->getOutputPosition(idx).y() - getPos().y();
		lw = lw->Next();
	}
	maxy = maxy / Grid::GRID;
	if (maxy > min_size) min_size = maxy;

	if (new_s > min_size) {
		if (size() != new_s) retval = true;
		else retval = false;
		deviceSize = new_s;
	} else {
		if (size() != min_size) retval = true;
		else retval = false;
		deviceSize = min_size;
	}
	return retval;
}

int XDevice::neededSize()
{	int new_size = MINSIZE;

	KlogicList<Calculator> *lop = named_output.First();
	while(lop) {
		if ((lop->Get()->getPosition() + 1) > new_size)
			new_size = lop->Get()->getPosition();
		lop = lop->Next();
	}
	KlogicList<Value> *lv = named_input.First();
	while(lv) {
		if (lv->Get()->getPosition() + 1 > new_size)
			new_size = lv->Get()->getPosition();
		lv = lv->Next();
	}

	if (new_size < MINSIZE) new_size = MINSIZE;
	return new_size;
}

bool XDevice::sizeChangeable()
{
	int function = type();

        return (function != DeviceType::fSS ||
            function == DeviceType::fRS ||
            function == DeviceType::fLEDred ||
            function == DeviceType::fLEDblue ||
            function == DeviceType::fLEDgreen ||
            function == DeviceType::fLEDyellow ||
            function == DeviceType::fDFF ||
            function == DeviceType::fTRI ||
            function == DeviceType::fONE);
}

int XDevice::maxI()
{	return deviceMaxInput;
}

int XDevice::currI()
{       return in_w.counter();
}

void XDevice::setMaxI(int newmax)
{	if (newmax < currI()) deviceMaxInput = currI();
	else deviceMaxInput = newmax;
}

int XDevice::maxO()
{	return deviceMaxOutput;
}

int XDevice::currO()
{       return out_w.counter();
}

void XDevice::setMaxO(int newmax)
{	if (newmax < currO()) deviceMaxOutput = currO();
	else deviceMaxOutput = newmax;
}

// output inverted by default?
bool XDevice::isInverted()
{	return deviceOutputInverted;
}

// output inverted by default?
// 1=invert always
void XDevice::setInverted(bool flag)
{
	deviceOutputInverted = flag;
}

// virtual
void XDevice::setName(const QString& sName)
{
	Device::setName(sName);

        if (m_oTextDev) {
                m_oTextDev->setName(sName);
                setTextDevPos();
        }

	setImage();
}

// show/hide the name of the display
void XDevice::displayName(bool flag)
{
	if (!getID()) return;
	if (flag) {
		m_oTextDev = new TextDevice(0, 0);
		QString txt = getName();
		if (txt.isEmpty()) txt = " ";
		m_oTextDev->setName(txt);

		QRect dpos = getPos();
		QRect tpos = m_oTextDev->getPos();
		QPoint _newpos;
		_newpos.setX(dpos.x() + dpos.width() / 2 - tpos.width() / 2);
		_newpos.setY(dpos.y() - tpos.height());
		m_oTextDev->setVOffset(-3);
		m_oTextDev->setHOffset(-XObject::IOREG);
		m_oTextDev->initPos(_newpos.x(), _newpos.y());
                setTextDevPos();
	} else {
		if (!m_oTextDev) return;
		delete m_oTextDev;
		m_oTextDev = 0;
	}
}

// is the name of the device shown/hidden
bool XDevice::nameDisplayed()
{
	if (m_oTextDev) return true;
	return false;
}

// set all output to tristate enabled
// add/remove "EN" input
void XDevice::setTristate(bool flag)
{
	if (!tristate && !flag || tristate && flag) return;
	Device::setTristate(flag);

	// add/remove "EN" input if nessesary
	if (tristate) {
		setSize(neededSize() + 1);

		// if possible, add it at top position
		if (type() != DeviceType::fRAM && !in_w.counter() && !tristate_input_moved) {
			tristate_input_moved++;
			// move all inputs down
			KlogicList<Value> *lv = named_input.First();
			while(lv) {
				lv->Get()->setPosition(lv->Get()->getPosition() + 1);
				lv = lv->Next();
			}
			tristate_enable_id = addInputName(INPUT_TRISTATE_CONTROL, 0);
		} else {
			tristate_enable_id = addInputName(INPUT_TRISTATE_CONTROL);
		}
	} else {
		// if possible and needed, move all inputs up again
		KlogicList<Value> *lv = named_input.With(tristate_enable_id);
		if (!lv) return;
		if (!in_w.counter() && tristate_input_moved) {
			tristate_input_moved--;
			removeInputName(tristate_enable_id);
			lv = named_input.First();
			while(lv) {
				lv->Get()->setPosition(lv->Get()->getPosition() - 1);
				lv = lv->Next();
			}
		} else {
			removeInputName(tristate_enable_id);
		}
		tristate_enable_id = 0;
		setSize(neededSize());
	}
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
}

// check if all outputs are inverted
int XDevice::invertGraph(int value, int output_id)
{	KlogicList<XWire> *lw = out_w.First();
	int idx;

	if (!output_id) {
		while (lw) {
			if (!lw->Get()->outputIsInverted(this)) return value;
			lw = lw->Next();
		}
		if (value == 0) return 1;
		else return 0;
	} else {
		while (lw) {
			idx = lw->Get()->getOutputIndex(this);
			if (XWire::WFAIL == idx)
				fatal("XDevice::invertGraph() this is unknown in a connected wire !?");
			if (lw->Get()->outputID(idx) == output_id) {
				if (!lw->Get()->outputIsInverted(this)) return value;
				if (value == 0) return 1;
				else return 0;
			}
			lw = lw->Next();
		}
		// not found??
		return value;
	}
}

void XDevice::setClock(int new_clk)
{
	if (new_clk == clock()) return;

	if (clock() == CLK_NONE  && new_clk != CLK_NONE && type() == DeviceType::fRS) {
		// remove all named inputs...
		while(named_input.First())
			removeInputName(named_input.First()->getID1());
		// ... and add the new ones
		addInputName(INPUT_FF_1S, 0);
		addInputName(INPUT_FF_C1, 2);
		addInputName(INPUT_FF_1R, 4);
	}
	if (clock() != CLK_NONE && new_clk == CLK_NONE && type() == DeviceType::fRS) {
		// remove all named inputs...
		while(named_input.First())
			removeInputName(named_input.First()->getID1());
		// ... and add the new ones
		addInputName(INPUT_FF_S, 0);
		addInputName(INPUT_FF_R, 3);
	}
	Device::setClock(new_clk);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
}

// draw a graph in the simulation graph widget?
bool XDevice::drawGraph()
{
	return maxO();
}

// returns if device should be shown in the simulation graph
bool XDevice::graphEnabled()
{
	return graph_enabled;
}

// show device in the simulation graph
void XDevice::enableGraph(bool flag)
{
	graph_enabled = flag;
}

// get width
int XDevice::width()
{
	return deviceWidth;
}

// set width
void XDevice::setWidth(int new_width)
{	bool changed = false;

	if (type() != DeviceType::fNET && type() != DeviceType::fEQU)
		return;

	if (deviceWidth != new_width)
		changed = true;

	deviceWidth = new_width;

	if (changed) {
		XWire *w;
		int idx;
		QPoint p;

		setImage();
		KlogicList<XWire> *lw = out_w.First();
		while (lw) {
			w = lw->Get();
			// set new width in connected wire node
			idx = w->getOutputIndex(this);
			if (XWire::WFAIL == idx)
				fatal("XDevice::setWidth() this is unknown in a connected wire !?");
			p = w->getOutputPosition(idx);
			p.setX(getPos().x() + getPos().width() - 1);
			w->setOutputPosition(p, idx);
			lw = lw->Next();
		}
		if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	}
}

// update positions of device-connections
void XDevice::erase(QPainter *p)
{	KlogicList<XWire> *l;

	if (m_oTextDev) m_oTextDev->erase(p);

	updateWires();
	l = in_w.First();
	while(l) {
		l->Get()->erase(p);
		l = l->Next();
	}
	l = out_w.First();
	while(l) {
		l->Get()->erase(p);
		l = l->Next();
	}
	XObject::erase(p);
}

// garbage collection for connected wires
void XDevice::garbageCollection()
{	KlogicList<XWire> *lw;

	lw = in_w.First();
	while(lw) {
		lw->Get()->Wire::garbageCollection();
		lw = lw->Next();
	}
	lw = out_w.First();
	while(lw) {
		lw->Get()->Wire::garbageCollection();
		lw = lw->Next();
	}
} 

// update nodes of wires which are connected to the device
void XDevice::updateWires()
{	QRect p1, p2;
	int dx, dy;
	KlogicList<XWire> *lw;
	XWire *w;
	QPoint newpos;
	int idx;

	p1 = getPos();
	p2 = getOldPos();
	dx = p1.x() - p2.x();
	dy = p1.y() - p2.y();
	newpos.setX(dx);
	newpos.setY(dy);
	lw = in_w.First();
	while(lw) {
		w = lw->Get();
		idx = w->getInputIndex(this);
		w->setInputPosition(w->getInputPosition(idx) + newpos, idx);
		lw = lw->Next();
	}
	lw = out_w.First();
	while(lw) {
		w = lw->Get();
		idx = w->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::updateWires() this is unknown in a connected wire !?");
		w->setOutputPosition(w->getOutputPosition(idx) + newpos, idx);
		lw = lw->Next();
	}
}

// checks, if actice-node of wire is input or output
// connects if given, or disconnects if not given
// wire must be locked!
int XDevice::checkConnection(XWire *w, int invert)
{	int found;
	int ret1, ret2;

	// active node of w must be the first or last node of w
	if (!w->activeIsEnd()) return DOK;

	// set input-, device- and output-region
	setRegions();

	// check, if wire is already input
	if (in_w.Get(w)) found = 1;
	else found = 0;
	// new connected, connection lost or not connected?
	ret1 = checkInput(w, found, invert);
	if (ret1 != DOK && ret1 != DDISC) return ret1;

	// check, if wire is already output
	if (out_w.Get(w)) found = 1;
	else found = 0;
	// new connected, connection lost or not connected?
	ret2 = checkOutput(w, found, invert);
	if (ret2 != DOK && ret2 != DDISC) return ret2;

	if (ret1 == DDISC || ret2 == DDISC) return DDISC;
	return DOK;
}

int XDevice::checkInput(XWire *w, int found, int invert)
{	QPoint p = w->getActive();
	QPoint ap;
	int idx;
	int isconn = 0;
	KlogicList<Value> *li = named_input.First();
	int new_id = 0;
	int conn_id = 0;

	if (!found) {
		// check input connection
		if (!input_region.contains(p)) return DOK;

		// connections in all points allowed
		if (!hasNamedInput()) {
			return connectInput(w, p, invert);
		} else {
			// connections only in some points allowed (named inputs)
			while (li) {
				new_id = li->getID1();
				ap.setX(getPos().x());
				ap.setY(li->Get()->getPosition() * Grid::GRID + getPos().y() + getVOffset());
				if (ap == p) return connectInput(w, p, invert, new_id);
				li = li->Next();
			}
		}
	} else {
		// check input disconnection
		idx = w->getInputIndex(this);
		p = w->getInputPosition(idx);

		if (!hasNamedInput()) {
			if (!input_region.contains(p)) {
				w->disconnectInput(idx);
				return DDISC;
			}
		}
		else {
			while (li && !isconn) {
				new_id = li->getID1();
				ap.setX(getPos().x());
				ap.setY(li->Get()->getPosition() * Grid::GRID + getPos().y() + getVOffset());
				if (ap == p) isconn = 1;
				li = li->Next();
			}
			if (!isconn) {
				w->disconnectInput(idx);
				return DDISC;
			} else {
				conn_id = w->getConnectionID(this);
				if (conn_id != new_id) {
					w->disconnectInput(idx);
					connectInput(w, p, invert, new_id);
				}
			}
		}
	}
	return DOK;
}

// give wire the information to be an input to this device
// input-id: id of the named input of this device
int XDevice::connectInput(XWire *w, QPoint p, int invert, int input_id)
{
	if (XWire::WFAIL != w->activeIsInput()) return DOK;

	// check that another input is allowed
	if (maxI() <= currI())
		return DFAILMAXI;

	if (XWire::WFAIL == w->connectInput(this, invert, input_id)) return DFAIL;

	// set the position of the input-node of the wire exactly to the connection-point of this device
	p.setX(getPos().x());
	w->setInputPosition(p);

	return DCONN;
}

int XDevice::checkOutput(XWire *w, int found, int invert)
{	QPoint p = w->getActive();
	QPoint ap;
	QRect r = getPos();
	int isconn = 0;
	int new_id = 0;
	int conn_id = 0;

	if (!found) {
		// check output connection
		if (!output_region.contains(p)) {
			// fSWI became smaller .. workaround in order to load old files!
			if (type() == DeviceType::fSWI) {
				output_region.setWidth(output_region.width() + 10);
				if (!output_region.contains(p)) {
					output_region.setHeight(output_region.height() + 10);
					if (!output_region.contains(p)) {
						return DOK;
					} else {
						p.setY(p.y() - 10);
						w->updateNode(p);
					}
				} else {
					p.setX(p.y() - 10);
					w->updateNode(p);
				}
			} else {
				return DOK;
			}
		}

		if (!hasNamedOutput()) {
			if (XWire::WFAIL != w->activeIsOutput()) return DOK;
			// if the wire is not connected already, connect it now
			return connectOutput(w, p, invert);
		}
		// connections in some points allowed only
		else {
			KlogicList<Calculator> *lo = named_output.First()->Next();
			while (lo) {
				if (!lo->Get()->isInternal()) {
					new_id = lo->getID1();
					ap.setX(r.x() + r.width() - 1);
					ap.setY(lo->Get()->getPosition() * Grid::GRID + r.y() + getVOffset());
					if (ap == p) return connectOutput(w, p, invert, new_id);
				}
				lo = lo->Next();
			}
		}
	}
	// check output disconnection
	// if wire is connected, disconnect it now
	else {
		int idx = w->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::checkOutput() this is unknown in a connected wire !?");
		p = w->getOutputPosition(idx);
		if (!hasNamedOutput()) {
			if (!output_region.contains(p)) {
				w->disconnectOutput(idx);
				return DDISC;
			}
		}
		else {
			KlogicList<Calculator> *lo = named_output.First()->Next();
			while (lo && !isconn) {
				if (!lo->Get()->isInternal()) {
					new_id = lo->getID1();
					ap.setX(r.x() + r.width() - 1);
					ap.setY(lo->Get()->getPosition() * Grid::GRID + r.y() + getVOffset());
					if (ap == p) isconn = 1;
				}
				lo = lo->Next();
			}
			if (!isconn) {
				w->disconnectOutput(idx);
				return DDISC;
			} else {
				conn_id = w->getConnectionID(this);
				if (conn_id != new_id) {
					w->disconnectOutput(idx);
					return connectOutput(w, p, invert, new_id); 
				}
			}
		}
	}
	return DOK;
}

// give wire the information to get input from this (output-)device
// output-id: id of the named output of this device
int XDevice::connectOutput(XWire *w, QPoint p, int invert, int output_id)
{	int ret;

	if (XWire::WFAIL != w->activeIsInput()) return DOK;

	// check that another output is allowed
	if (maxO() <= currO())
		return DFAILMAXO;

	if (WOK != (ret = w->connectOutput(this, invert, output_id))) return ret;

	p.setX(getPos().x() + getPos().width() - 1);
	w->setOutputPosition(p);

	return DCONN;
}

void XDevice::setInputLine(XWire *w, QColor color, int idx, QPainter *prt, int x, int y)
{	QPixmap *pic = getImage();
	QPoint conn_pt, pt1, pt2;
	QRect r;
	QPainter *p;

	if (!prt) {
		p = new QPainter();
		p->begin(pic);
	} else p = prt;

	conn_pt = w->getInputPosition(idx);
	r = getPos();

	if (conn_pt.isNull()) return;

	pt1.setX(x);
	pt1.setY(y + conn_pt.y() - r.y());
	pt2.setX(x + XObject::IOREG);
	pt2.setY(pt1.y());

	p->setPen(color);
	if (w->inputIsInverted(this)) {
		p->setBrush(color);
		p->drawPie(pt1.x(), pt1.y() - 3, 5, 6, 0, 5760);
	} else {
		p->drawLine(pt1, pt2);
	}

	if (color == Qt::white) {
		// inverter-pie lies 1 pixel in the base image, redraw base image line
		if (type() != DeviceType::fPWR && type() != DeviceType::fBUS) {
			if (STATSymbolSet == XDevice::SYMBOLSET_DIN40900 ||
			    (type() != DeviceType::fAND && type() != DeviceType::fOR &&
			     type() != DeviceType::fXOR && type() != DeviceType::fONE))
				p->setPen(Qt::black);
			else
				p->setPen(BlankColor);
			p->drawLine(pt2.x(), pt1.y() - 2, pt2.x(), pt1.y() + 2);
		}

		// redraw input line if other wires are connected
		KlogicList<XWire> *lw = in_w.First();
		p->setPen(Qt::black);
		p->setBrush(Qt::black);
		while(lw) {
			if (lw->Get() != w &&
			  (conn_pt == lw->Get()->getInputPosition(0) || conn_pt == lw->Get()->getInputPosition(1))) {
				if (lw->Get()->inputIsInverted(this)) {
					p->drawPie(pt1.x(), pt1.y() - 3, 5, 6, 0, 5760);
				} else {
					p->drawLine(pt1, pt2);
				}
			}
			lw = lw->Next();
		}
	}

	if (!prt) {
		p->end();
		delete p;
	}
}

// called by XWire
// returns invertation of another wire connected in the same point
int XDevice::inputIsInvertedAt(QPoint conn_pt)
{
	KlogicList<XWire> *lw = in_w.First();
	while(lw) {
		if (conn_pt == lw->Get()->getInputPosition(0) || conn_pt == lw->Get()->getInputPosition(1)) {
			return lw->Get()->inputIsInverted(this);
		}
		lw = lw->Next();
	}
	return 0;
}

// called by XWire
// add wire to input-wire-list
// draw wire-connection into image of device
void XDevice::addInput(XWire *w, int idx)
{
	setInputLine(w, Qt::black, idx);
	in_w.Append(w);
}

// called by XWire
// remove wire from input-wire-list
void XDevice::removeInput(XWire *w, int idx)
{
	setInputLine(w, Qt::white, idx);
	in_w.Destroy(w);
}

// draws tiny black line in the connection area of device (part of device image)
// also draws the inverter pies
void XDevice::setOutputLine(XWire *w, QColor color, int idx, QPainter *prt, int x, int y)
{	QPixmap *pic = getImage();
	QPixmap pfkt;
	QPoint conn_pt, pt1, pt2;
	QRect r;
	QPainter *p;

	if (!prt) {
		p = new QPainter();
		p->begin(pic);
	} else p = prt;

	conn_pt = w->getOutputPosition(idx);
	r = getPos();

	pt1.setX(x + r.width() - XObject::IOREG - 1);
	pt1.setY(y + conn_pt.y() - r.y());
	pt2 = pt1;
	pt2.setX(x + r.width());

	p->setPen(color);
	if (w->outputIsInverted(this)) {
		p->setBrush(color);
		p->drawPie(pt1.x(), pt1.y() - 3, 5, 6, 0, 5760);
		p->setBrush(Qt::white);
	}
	else {
		p->drawLine(pt1, pt2);
	}

	// inverter-pie lies 1 pixel in the base image, redraw base image line
	if (type() != DeviceType::fPWR && type() != DeviceType::fBUS) {
		if (STATSymbolSet == XDevice::SYMBOLSET_DIN40900 ||
		    (type() != DeviceType::fAND && type() != DeviceType::fOR &&
		     type() != DeviceType::fXOR && type() != DeviceType::fONE))
			p->setPen(Qt::black);
		else
			p->setPen(BlankColor);
		p->drawLine(pt1.x(), pt1.y() - 2, pt1.x(), pt1.y() +2);
	}

	// redraw input line if other wires are connected
	if (color == Qt::white) {
		KlogicList<XWire> *lw = out_w.First();
		p->setPen(Qt::black);
		p->setBrush(Qt::black);
		while(lw) {
			if (lw->Get() != w &&
			  (conn_pt == lw->Get()->getOutputPosition(0) || conn_pt == lw->Get()->getOutputPosition(1))) {
				if (lw->Get()->outputIsInverted(this)) {
					p->drawPie(pt1.x(), pt1.y() - 3, 5, 6, 0, 5760);
				} else {
					p->drawLine(pt1, pt2);
				}
			}
			lw = lw->Next();
		}
	}

	if (!prt) {
		p->end();
		delete p;
	}
}

// called by XWire
// returns invertation of another wire connected in the same point
int XDevice::outputIsInvertedAt(QPoint conn_pt)
{
	KlogicList<XWire> *lw = out_w.First();
	while(lw) {
		   if (conn_pt == lw->Get()->getOutputPosition(0) || conn_pt == lw->Get()->getOutputPosition(1)) {
			return lw->Get()->outputIsInverted(this);
		}
		lw = lw->Next();
	}
	return 0;
}

// called by XWire
// add wire to output-wire-list
void XDevice::addOutput(XWire *w, int idx)
{
	setOutputLine(w, Qt::black, idx);
	out_w.Append(w);
}

// called by XWire
// remove wire from output-wire-list
void XDevice::removeOutput(XWire *w, int idx)
{
	setOutputLine(w, Qt::white, idx);
	out_w.Destroy(w);
}

// export method
// get tristate control input position
int XDevice::getTristateControlPos()
{
	KlogicList<Value> *lv = named_input.With(tristate_enable_id);
	if (!lv) return 0;

	return lv->Get()->getPosition();
}

void XDevice::setTristateControlPos(int pos)
{
	Device::setTristate(true);
	tristate_pos_import = pos;
}

// private, used for import
// shift existing inputs around that the tristate input "EN" can be inserted
void XDevice::prepareTristateInputImport()
{
	// look if given entry is free
	KlogicList<Value> *li = named_input.First();
	bool moved = false;
	while (li) {
		if (!li->Get()->isInternal() && li->Get()->getPosition() >= tristate_pos_import) {
			if (!moved) {
				moved = true;
				tristate_input_moved++;
			}
			li->Get()->setPosition(li->Get()->getPosition() + 1);
		}
		li = li->Next();
	}
}

// import method
void XDevice::setTristateInputImport()
{
	KlogicList<Value> *lv = named_input.With("EN");
	if (lv) {
		tristate_enable_id = lv->getID1();
	} else {
		setSize(neededSize() + 1);

		// shift existing inputs
		prepareTristateInputImport();
		// now insert "EN" input
		tristate_enable_id = addInputName("EN", tristate_pos_import);
	}
}

// called by device-net(-device)
// add an possible named input (no connection!)
int XDevice::addInputName(const QString& nam, int pos, int input_id)
{	int y = pos;
	int maxy = size();
	int found;
	KlogicList<Value> *li;
	int ret;

	if (pos >= 0) {
		// look if given entry is free
		found = 0;
		li = named_input.First();
		while (li) {
			if (!li->Get()->isInternal() && li->Get()->getPosition() == y)
				found = 1;
			li = li->Next();
		}
	} else {
		// search for a free entry
		do {
			found = 0;
			y ++;
			li = named_input.First();
			while (li) {
				if (!li->Get()->isInternal() && li->Get()->getPosition() == y)
					found = 1;
				li = li->Next();
			}
		} while (found && (y < maxy));
	}
	if (found) return ADDNAMED_FULL;

	// add input
	if (0 >= (ret = Device::addInputName(nam, y, input_id))) return ret;
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	return ret;
}

// change an input name
// if the input does not exist, create a new named input
int XDevice::changeInputName(int input_id, const QString& nam)
{
	if (!input_id || !named_input.With(input_id))
		input_id = addInputName(nam);
	else
		Device::changeInputName(input_id, nam);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	return input_id;
}

// remove a named input
void XDevice::removeInputName(int input_id)
{	KlogicList<XWire> *lw = in_w.First();

	// remove all connections to this input
	while(lw) {
		if (lw->Get()->isNamedInput(this, input_id)) {
			int idx = lw->Get()->getInputIndex(this);
			lw->Get()->disconnectInput(idx);
			lw = in_w.First();
		} else lw = lw->Next();
	}

	// remove input name
	Device::removeInputName(input_id);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
}

// called by device-net(-device) and equation-device properties
// add an possible named output (no connection!)
int XDevice::addOutputName(const QString& nam, int pos, int dev_id)
{	int y = pos;
	int maxy = size();
	int found = 1;
	KlogicList<Calculator> *lo;

	// ignore ...
	if (!nam.length() || !nam.compare("(null)")) return ADDNAMED_OK;

	// search for a free entry
	if (pos >= 0) {
		// look if given entry is still free
		found = 0;
		lo = named_output.First();
		while (lo && !found) {
			if (!lo->Get()->isInternal() && lo->Get()->getPosition() == y)
				found = 1;
			lo = lo->Next();
		}
	} else {
		// try to find an empty entry
		do {
			found = 0;
			y ++;
			lo = named_output.First();
			while (lo) {
				if (!lo->Get()->isInternal() && lo->Get()->getPosition() == y)
					found = 1;
				lo = lo->Next();
			}
		} while (found && (y < maxy));
	}
	if (found) return ADDNAMED_FULL;

	// add output
	int ret;
	if (0 >= (ret = Device::addOutputName(nam, y, dev_id))) return ret;
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	return ret;
}

int XDevice::changeOutputName(int output_id, const QString& nam)
{
	if (!output_id || !named_output.With(output_id))
		output_id = addOutputName(nam);
	else
		Device::changeOutputName(output_id, nam);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	return output_id;
}

// called by device-net(-device) and equation-devices properties
// remove output name
void XDevice::removeOutputName(int output_id)
{	KlogicList<XWire> *lw = out_w.First();

	// remove all connections to this output
	while(lw) {
		if (lw->Get()->isNamedOutput(this, output_id)) {
			int idx = lw->Get()->getOutputIndex(this);
			if (XWire::WFAIL == idx)
				fatal("XDevice::removeOutputName() this is unknown in a connected wire !?");
			lw->Get()->disconnectOutput(idx);
			lw = out_w.First();
		} else lw = lw->Next();
	}

	// remove output name
	Device::removeOutputName(output_id);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
}

KlogicList<XWire> * XDevice::getIRef()
{
	return &in_w;
}

KlogicList<XWire> * XDevice::getORef()
{
	return &out_w;
}

// create the device image
void XDevice::setImage()
{	QPixmap *pic = getImage();
	QPixmap *part = (QPixmap *)NULL;
	QPainter p;
	QPoint pt(0,0);
	QPoint pp(15,35);
	KlogicList<Calculator> *lop;
	KlogicList<Value> *lv;
	int i;
	int pos;
	int xdistance;
	int xwidth;

	if (type() == DeviceType::fINV_INTERNAL) return;

	// first resize the device pixmap, assign the base image to "part"..
	switch (this->type()) {
		case DeviceType::fRS:
			if (!hasMaster()) {
				if (clock() == CLK_NONE) part = pxRS;
				else part = pxRSC;
			} else {
				if (clock() == CLK_RISING_1EDGE || clock() == CLK_FALLING_1EDGE) part = pxRSCMF;
				else part = pxRSCM;
			}
			pic->resize(part->width(), part->height());
			break;
		case DeviceType::fDFF:
			if (hasMaster()) part = pxDM;
			else part = pxD;
			pic->resize(part->width(), part->height());
			break;
		case DeviceType::fSS:
			pic->resize(pxSSB->width(), pxSSB->height());
			break;
		case DeviceType::fLEDred:
			pic->resize(pxLED0red->width(), pxLED0red->height());
			break;
		case DeviceType::fLEDblue:
			pic->resize(pxLED0blue->width(), pxLED0blue->height());
			break;
		case DeviceType::fLEDgreen:
			pic->resize(pxLED0green->width(), pxLED0green->height());
			break;
		case DeviceType::fLEDyellow:
			pic->resize(pxLED0yellow->width(), pxLED0yellow->height());
			break;
		case DeviceType::fTRI:
		case DeviceType::fEQU:
		case DeviceType::fNET:
			if (width() == 3) {
				i =  PTOPH->height() + (PMIDH->height() * size()) + PBOTH->height();
				pic->resize(PTOPH->width(), i);
			} else {
				i =  PTOPL->height() + (PMIDL->height() * size()) + PBOTL->height();
				pic->resize(PTOPL->width(), i);
			}
			break;
		default:
			if (XDevice::getSymbolSet() == XDevice::SYMBOLSET_DIN40900 ||
				(type() != DeviceType::fAND && type() != DeviceType::fOR &&
				 type() != DeviceType::fXOR && type() != DeviceType::fONE)) {
				i =  PTOP->height() + (PMID->height() * size()) + PBOT->height();
				pic->resize(PTOP->width(), i);
			} else {
				i =  PTOPBlank->height() + (PMIDBlank->height() * size()) + PBOTBlank->height();
				pic->resize(PTOPBlank->width(), i);
			}
			break;
	}

	p.begin(pic);

	// ..then draw the device image
	switch (this->type()) {
		case DeviceType::fRS:
			if (!part) fatal("no pixmap??");
			p.drawPixmap(pt, *part);
			switch (clock()) {
				case CLK_FALLING_1EDGE:
				case CLK_LOW_VALUE:
					p.drawLine(pic->width() - 15, 3, pic->width() - 9, 3);
					p.drawLine(pic->width() - 9, 3, pic->width() - 9, 7);
					break;
				case CLK_RISING_1EDGE:
				case CLK_HIGH_VALUE:
					p.drawLine(pic->width() - 15, 7, pic->width() - 9, 7);
					p.drawLine(pic->width() - 9, 3, pic->width() - 9, 7);
					break;
				default:
					break;
			}
			break;
		case DeviceType::fDFF:
			if (!part) fatal("no pixmap??");
			p.drawPixmap(pt, *part);
			break;
		case DeviceType::fSS:
			p.drawPixmap(pt, *pxSSB);
			pp.setX(pic->width()/2 - sySS->width()/2 + Grid::GRIDHALF - 3);
			pp.setY(pic->height()/2 - sySS->height()/2);
			p.drawPixmap(pp, *pxSS[0]);
			break;
		case DeviceType::fLEDred:
			p.drawPixmap(pt, *pxLED0red);
			break;
		case DeviceType::fLEDblue:
			p.drawPixmap(pt, *pxLED0blue);
			break;
		case DeviceType::fLEDgreen:
			p.drawPixmap(pt, *pxLED0green);
			break;
		case DeviceType::fLEDyellow:
			p.drawPixmap(pt, *pxLED0yellow);
			break;
		case DeviceType::fTRI:
		case DeviceType::fEQU:
		case DeviceType::fNET:
			if (width() == 3) {
				// top of the device
				p.drawPixmap(pt, *PTOPH);
				pt.setY(pt.y() + PTOPH->height());

				// middle-part of device
				for(i=0;i<size();i++) {
					p.drawPixmap(pt, *PMIDH);
					pt.setY(pt.y() + PMIDH->height());
				}

				// bottom of device
				p.drawPixmap(pt, *PBOTH);
			} else {
				// top of the device
				p.drawPixmap(pt, *PTOPL);
				pt.setY(pt.y() + PTOPL->height());

				// middle-part of device
				for(i=0;i<size();i++) {
					p.drawPixmap(pt, *PMIDL);
					pt.setY(pt.y() + PMIDL->height());
				}

				// bottom of device
				p.drawPixmap(pt, *PBOTL);
			}

			// draw input- and output names
			p.setFont(STATfont);
			lv = named_input.First();
			xdistance = XObject::IOREG + 3;
			while(lv) {
				if (!lv->Get()->isInternal()) {
					pos = lv->Get()->getPosition() * Grid::GRID;
					if (lv->Get()->getPosition() == 0)
						p.drawText(xdistance, pos + 7, lv->getText());
					else if (pos == (size() * Grid::GRID))
						p.drawText(xdistance, pos + 5, lv->getText());
					else 
						p.drawText(xdistance, pos + 6, lv->getText());
				}
				lv = lv->Next();
			}
			lop = named_output.First()->Next();
			xdistance = pic->width() / 2;
			xwidth = xdistance - 6;
			while(lop) {
				if (!lop->Get()->isInternal()) {
					pos = lop->Get()->getPosition() * Grid::GRID;
					if (pos == 0)
						p.drawText(xdistance, pos - 2, xwidth, 10,
							Qt::AlignRight, lop->getText());
					else if (pos == (size() * Grid::GRID))
						p.drawText(xdistance, pos - 4 , xwidth, 10,
							Qt::AlignRight, lop->getText());
					else 
						p.drawText(xdistance, pos - 3, xwidth, 10,
							Qt::AlignRight, lop->getText());
				}
				lop = lop->Next();
			}

			break;
		default:
			if (XDevice::getSymbolSet() == XDevice::SYMBOLSET_DIN40900 ||
				(type() != DeviceType::fAND && type() != DeviceType::fOR &&
				 type() != DeviceType::fXOR && type() != DeviceType::fONE)) {
				// top of the device
				p.drawPixmap(pt, *PTOP);
				pt.setY(pt.y() + PTOP->height());

				// middle-part of device
				for(i=0;i<size();i++) {
					p.drawPixmap(pt, *PMID);
					pt.setY(pt.y() + PMID->height());
				}

				// bottom of device
				p.drawPixmap(pt, *PBOT);
			} else {
				// top of the device
				p.drawPixmap(pt, *PTOPBlank);
				pt.setY(pt.y() + PTOPBlank->height());

				// middle-part of device
				for(i=0;i<size();i++) {
					p.drawPixmap(pt, *PMIDBlank);
					pt.setY(pt.y() + PMIDBlank->height());
				}

				// bottom of device
				p.drawPixmap(pt, *PBOTBlank);
			}

			// draw symbol
			switch(this->type()) {
				case DeviceType::fAND:
					if (XDevice::getSymbolSet() == XDevice::SYMBOLSET_DIN40900) {
						pp.setX(pic->width()/2 - syAND->width()/2);
						pp.setY(pic->height()/2 - syAND->height()/2);
						p.drawPixmap(pp, *syAND);
					} else {
						pp.setX(pic->width()/2 - syIECAND->width()/2);
						pp.setY(pic->height()/2 - syIECAND->height()/2);
						p.drawPixmap(pp, *syIECAND);
					}
					break;
				case DeviceType::fOR:
					if (XDevice::getSymbolSet() == XDevice::SYMBOLSET_DIN40900) {
						pp.setX(pic->width()/2 - syOR->width()/2);
						pp.setY(pic->height()/2 - syOR->height()/2);
						p.drawPixmap(pp, *syOR);
					} else {
						pp.setX(pic->width()/2 - syIECOR->width()/2);
						pp.setY(pic->height()/2 - syIECOR->height()/2);
						p.drawPixmap(pp, *syIECOR);
					}
					break;
				case DeviceType::fXOR:
					if (XDevice::getSymbolSet() == XDevice::SYMBOLSET_DIN40900) {
						pp.setX(pic->width()/2 - syXOR->width()/2);
						pp.setY(pic->height()/2 - syXOR->height()/2);
						p.drawPixmap(pp, *syXOR);
					} else {
						pp.setX(pic->width()/2 - syIECXOR->width()/2);
						pp.setY(pic->height()/2 - syIECXOR->height()/2);
						p.drawPixmap(pp, *syIECXOR);
					}
					break;
				case DeviceType::fONE:
					if (XDevice::getSymbolSet() == XDevice::SYMBOLSET_DIN40900) {
						pp.setX(pic->width()/2 - syINV->width()/2);
						pp.setY(pic->height()/2 - syINV->height()/2);
						p.drawPixmap(pp, *syINV);
					} else {
						pp.setX(pic->width()/2 - syIECNOT->width()/2);
						pp.setY(pic->height()/2 - syIECNOT->height()/2);
						p.drawPixmap(pp, *syIECNOT);
					}
					break;
				default:
					break;
			}

			break;
	}

	// draw connection lines of the new image
	drawConnectionLines(&p);

	// set image dimension, clear position
	XObject::setImage();

	p.end();

	if (m_oTextDev) m_oTextDev->setImage();
}

int XDevice::setPos(QPoint newpos)
{	int ret;

	ret = XObject::setPos(newpos);
	setTextDevPos();
	return ret;
}

void XDevice::setTextDevPos()
{
	if (!m_oTextDev) return;

	QRect dpos = getPos();
	QRect tpos = m_oTextDev->getPos();
	QPoint _newpos;
	_newpos.setX(dpos.x() + dpos.width() / 2 - tpos.width() / 2);
	_newpos.setY(dpos.y() - tpos.height());
	m_oTextDev->setPos(_newpos);
}

// draw device to the screen
void XDevice::drawImage(QPaintDevice *paintd, QPainter *p)
{
	XObject::drawImage(paintd, p);
	if (m_oTextDev) m_oTextDev->drawImage(paintd, p);
}

// draw device as lines and text to a printer device
void XDevice::drawImage(QPainter *p)
{	QRect r;
	QPixmap *pic = getImage();
	int x,y;
	KlogicList<XWire> *lw;
	int idx;

	// position
	x = getPos().x();
	y = getPos().y();

	// draw connection lines
	lw = in_w.First();
	while (lw) {
		idx = lw->Get()->getInputIndex(this);
		setInputLine(lw->Get(), Qt::black, idx, p, x, y);
		lw = lw->Next();
	}
	lw = out_w.First();
	while (lw) {
		idx = lw->Get()->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::drawImage() this is unknown in a connected wire !?");
		setOutputLine(lw->Get(), Qt::black, idx, p, x, y);
		lw = lw->Next();
	}

	x += XObject::IOREG;
	y -= getVOffset() / 2;

	// draw device rect
	r.setX(x);
	r.setY(y);
	r.setWidth(pic->width() - (XObject::IOREG * 2) - 1);
	r.setHeight(pic->height());
	p->setPen(Qt::black);
	if (type() != DeviceType::fPWR && type() != DeviceType::fSWI) {
		p->drawRect(r);
	}

	printImage(p, r);
}

void XDevice::printImage(QPainter *p, const QRect& r)
{
	KlogicList<Value> *lv;
	KlogicList<Calculator> *lop;
	int pos;
	int a,b,c,d,e;
	int picwidth = getImage()->width();

	// draw io-names and type specific symbol
	switch(type()) {
		case DeviceType::fRS:
		case DeviceType::fDFF:
		case DeviceType::fNET:
			p->setFont(STATfont);
			lv = named_input.First();
			while(lv) {
				pos = lv->Get()->getPosition() * Grid::GRID;
				if (type() == DeviceType::fDFF && strcmp(lv->getText(), "C1") == 0 && hasMaster()) {
					a = r.x();
					b = r.x() + 6;
					c = r.y() + pos + 2;
					p->drawLine(a, c - 3, b, c);
					p->drawLine(a, c + 3, b, c);
					p->drawText(r.x() + 6, r.y() + pos + 5, lv->getText());
				} else {
					if (pos == 0)
						p->drawText(r.x() + 1, r.y() + pos + 7, lv->getText());
					else if (pos == (size() * Grid::GRID))
						p->drawText(r.x() + 1, r.y() + pos + 5, lv->getText());
					else
						p->drawText(r.x() + 1, r.y() + pos + 6, lv->getText());
				}
				lv = lv->Next();
			}
			if (type() == DeviceType::fDFF || type() == DeviceType::fRS) {
				p->drawText(r.x() + (picwidth / 2), r.y() - 2,
					picwidth / 2 - 7, 10, Qt::AlignRight, "Q");
			} else {
				lop = named_output.First();
				while(lop) {
					pos = lop->Get()->getPosition() * Grid::GRID;
					if (pos == 0)
						p->drawText(r.x() + (picwidth / 2), r.y() + pos - 2,
							picwidth / 2 - 7, 10, Qt::AlignRight, lop->getText());
					else if (pos == (size() * Grid::GRID))
						p->drawText(r.x() + (picwidth / 2), r.y() + pos - 4,
							picwidth / 2 - 7, 10, Qt::AlignRight, lop->getText());
					else
						p->drawText(r.x() + (picwidth / 2), r.y() + pos - 2,
							picwidth / 2 - 7, 10, Qt::AlignRight, lop->getText());
					lop = lop->Next();
				}
			}
			break;
		case DeviceType::fAND:
			p->setFont( QFont( "helvetica", 16, QFont::Bold ) );
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 5, "&");
			break;
		case DeviceType::fOR:
			p->setFont( QFont( "helvetica", 16, QFont::Bold ) );
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 1, ">");
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 1, "_");
			break;
		case DeviceType::fXOR:
			p->setFont( QFont( "helvetica", 16, QFont::Bold ) );
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 3, "=");
			break;
		case DeviceType::fONE:
			p->setFont( QFont( "helvetica", 16, QFont::Bold ) );
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 5, "1");
			break;
		case DeviceType::fSS:
		case DeviceType::fLEDred:
		case DeviceType::fLEDblue:
		case DeviceType::fLEDgreen:
		case DeviceType::fLEDyellow:
			a = r.x() + XObject::IOREG;
			b = r.x() + r.width() - XObject::IOREG;
			c = r.y() + (r.height() / 2);
			d = c - 4;
			e = c + 4;
			p->drawLine(r.x() + 2, c, r.x() + r.width() - 2, c);	// -
			p->drawLine(a, d, a, e);				// |
			p->drawLine(b, d, b, e);				//   |
			p->drawLine(a, d, b, c);				//  backslash
			p->drawLine(a, e, b, c);				//  /
			a = a + 2;
			d = d - 1;
			p->drawLine(a, d, a + 2, d - 2);
			p->drawLine(a + 1, d - 2, a + 2, d - 2);
			p->drawLine(a + 2, d - 2, a + 2, d - 1);
			a = a + 2;
			p->drawLine(a, d, a + 2, d - 2);
			p->drawLine(a + 1, d - 2, a + 2, d - 2);
			p->drawLine(a + 2, d - 2, a + 2, d - 1);
			break;
		default:
			break;
	}
}

void XDevice::eraseConnectionLines(QPainter *prt)
{	QPixmap *pic;
	QPainter *p;
	KlogicList<XWire> *lw;
	int idx;

	if (!prt) {
		pic = getImage();
		p = new QPainter();
		p->begin(pic);
	} else p = prt;

	// draw input lines
	lw = in_w.First();
	while (lw) {
		idx = lw->Get()->getInputIndex(this);
		setInputLine(lw->Get(), Qt::white, idx, p);
		lw = lw->Next();
	}

	// draw output lines
	lw = out_w.First();
	while (lw) {
		idx = lw->Get()->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::eraseConnectionLines() this is unknown in a connected wire !?");
		setOutputLine(lw->Get(), Qt::white, idx, p);
		lw = lw->Next();
	}

	if (!prt) {
		p->end();
		delete p;
	}
}

void XDevice::drawConnectionLines(QPainter *prt)
{	QPixmap *pic;
	QPainter *p;
	KlogicList<XWire> *lw;
	int idx;

	if (!prt) {
		pic = getImage();
		p = new QPainter();
		p->begin(pic);
	} else p = prt;

	// draw input lines
	lw = in_w.First();
	while (lw) {
		idx = lw->Get()->getInputIndex(this);
		setInputLine(lw->Get(), Qt::black, idx, p);
		lw = lw->Next();
	}

	// draw output lines
	lw = out_w.First();
	while (lw) {
		idx = lw->Get()->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::drawConnectionLines() this is unknown in a connected wire !?");
		setOutputLine(lw->Get(), Qt::black, idx, p);
		lw = lw->Next();
	}

	if (!prt) {
		p->end();
		delete p;
	}
}

void XDevice::setRegions()
{	QPixmap *pic = getImage();
	QRect p = getPos();

	input_region.setX(p.x() - 1);
	input_region.setY(p.y());
	input_region.setWidth(XObject::IOREG + 2);
	input_region.setHeight(pic->height());

	output_region.setX(p.x() + pic->width() - XObject::IOREG - 1);
	output_region.setY(p.y());
	output_region.setWidth(XObject::IOREG + 2);
	output_region.setHeight(pic->height());
}

void XDevice::init()
{
	switch (type()) {
		case DeviceType::fONE:
			deviceOutputInverted = true;
			setSize(MINSIZE);
			deviceMaxInput = 1;
			break;
		case DeviceType::fSS:
			deviceOutputInverted = false;
			deviceMaxInput = 4;
			deviceMaxOutput = 0;
			addInputName(INPUT_SS_2_0, 0);
			addInputName(INPUT_SS_2_1, 1);
			addInputName(INPUT_SS_2_2, 2);
			addInputName(INPUT_SS_2_3, 3);
			break;
		case DeviceType::fLEDred:
		case DeviceType::fLEDblue:
		case DeviceType::fLEDgreen:
		case DeviceType::fLEDyellow:
			deviceOutputInverted = false;
			deviceMaxInput = 4;
			deviceMaxOutput = 0;
			break;
		case DeviceType::fRS:
			deviceOutputInverted = false;
			addInputName(INPUT_FF_S, 0);
			addInputName(INPUT_FF_R, 3);
			mq_id = addInternalName(INTERNAL_FF_MASTER_Q);
			m_c1_1_id = addInternalName(INTERNAL_C1_PREVIOUS);
			break;
		case DeviceType::fDFF:
			deviceOutputInverted = false;
			addInputName(INPUT_FF_1D, 0);
			addInputName(INPUT_FF_C1, 3);
			mq_id = addInternalName(INTERNAL_FF_MASTER_Q);
			m_c1_1_id = addInternalName(INTERNAL_C1_PREVIOUS);
			break;
		case DeviceType::fEQU:
			setSize(MINSIZE);
			if (Device::IMPORT_IGNORE_GLOBAL) deviceWidth = XDevice::DEFAULT_WIDTH;
			else deviceWidth = STATdefWidth;
			break;
		case DeviceType::fNET:
			if (Device::IMPORT_IGNORE_GLOBAL) deviceWidth = XDevice::DEFAULT_WIDTH;
			else deviceWidth = STATdefWidth;
			break;
		case DeviceType::fTRI:
			// fEQU with a special function
			deviceOutputInverted = false;
			deviceMaxInput = 9;
			deviceMaxOutput = 8;
			tristate_enable_id = addInputName(INPUT_TRISTATE_CONTROL, 0);
			output_bus = new BusOutputArray(this, true, true);
			setSize(neededSize());
			break;
	}

	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	setEquation();
	parseEquation();
	if (STATdisplay_name && type() != DeviceType::fTXT)
		displayName(true);
}

// calculation method:
// modifies image of a device in dependance of input
bool XDevice::setColor()
{
	if (!outputChanged())
		return false;

	QString path;
	QPixmap *pic = getImage();
	QPixmap *img = (QPixmap *)NULL;
	QPoint pp(15,35);
	QPainter p;
	QString value;
	int val;

	// determine current value-dependant image
	switch(this->type()) {
	case DeviceType::fLEDred:
	case DeviceType::fLEDblue:
	case DeviceType::fLEDgreen:
	case DeviceType::fLEDyellow:
		val = this->output();
		if (val == TRISTATE) val = undef();
		switch(this->type()) {
		case DeviceType::fLEDred:
			if (val) img = pxLED1red;
			else img = pxLED0red;
			break;
		case DeviceType::fLEDblue:
			if (val) img = pxLED1blue;
			else img = pxLED0blue;
			break;
		case DeviceType::fLEDgreen:
			if (val) img = pxLED1green;
			else img = pxLED0green;
			break;
		case DeviceType::fLEDyellow:
			if (val) img = pxLED1yellow;
			else img = pxLED0yellow;
			break;
		}
		p.begin(pic);
		pp.setX(0);
		pp.setY(0);
		p.drawPixmap(pp, *img);
		drawConnectionLines(&p);
		p.end();
		return true;
		break;
	case DeviceType::fSS:
		val = this->output();
		if (val == TRISTATE) val = 0;
		if (val > 15) {
			warning("7S: value > 15!!");
			val = 0;
		}
		img = pxSS[val];
		p.begin(pic);
		pp.setX(pic->width()/2 - (img->width()/2) + Grid::GRIDHALF - 3);
		pp.setY(pic->height()/2 - img->height()/2);
		p.drawPixmap(pp, *img);
		p.end();
		return true;
		break;
	}
	return false;
}

