/*************************************************/
/* methods for class Wire                        */
/*                                               */
/* managing the nodes of a wire                  */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qpainter.h>
#include <qpen.h>

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "xwire.h"
#include "klogic.h"
#include "grid.h"

/***************************************************/
/* methods of Wire                                 */
/***************************************************/
Wire::Wire()
{	from_node = (QPoint *)NULL;
	current_node = (QPoint *)NULL;
	to_node = (QPoint *)NULL;
	o_node.setX(0);
	o_node.setY(0);
	o_start_node.setX(0);
	o_start_node.setY(0);
	o_end_node.setX(0);
	o_end_node.setY(0);
	o_ins_node1.setX(0);
	o_ins_node1.setY(0);
	o_ins_node2.setX(0);
	o_ins_node2.setY(0);
	bMustErase = 0;
	color = Qt::black;
	input[0] = (QPoint *)NULL;
	input[1] = (QPoint *)NULL;
	output[0] = (QPoint *)NULL;
	output[1] = (QPoint *)NULL;
	io_point[0] = (QPoint *)NULL;
	io_point[1] = (QPoint *)NULL;
	size = 1;
}

Wire::Wire(QPoint src)
{	align(&src);
	QPoint *p = new QPoint(src);
	wire.Append(p);
	from_node = (QPoint *)NULL;
	current_node = p;
	to_node = (QPoint *)NULL;
	o_node = src;
	o_start_node.setX(0);
	o_start_node.setY(0); 
	o_end_node.setX(0);
	o_end_node.setY(0); 
	o_ins_node1.setX(0);
	o_ins_node1.setY(0);
	o_ins_node2.setX(0);
	o_ins_node2.setY(0);
	bMustErase = 0;
	color = Qt::black;
	input[0] = (QPoint *)NULL;
	input[1] = (QPoint *)NULL;
	output[0] = (QPoint *)NULL;
	output[1] = (QPoint *)NULL;
	io_point[0] = (QPoint *)NULL;
	io_point[1] = (QPoint *)NULL;
	size = 1;
}

Wire::~Wire()
{	KlogicList<QPoint> *l = wire.First();
	QPoint *pt;

	while(l && (NULL != (pt = l->Get()))) {
		if (!nodeIsReference(pt))
			delete pt;
                l = l->Next();
	}
	wire.Destroy();
	io_point_foreign.Destroy();
	selnode.Destroy();
}

// node clicked? then lock the node
// if node is already connected to another wire/device,
// try other ones! (sometimes we have some odd wires ..)
KlogicList<QPoint> *Wire::contains(QPoint p)
{	KlogicList<QPoint> *l = wire.First();
	KlogicList<QPoint> *retList = 0;	// bad choose

	align(&p);

	// if point is in a node, lock node 
	while(l) {
		if (*(l->Get()) == p) {
			if (l->Get() == io_point[0] || l->Get() == io_point[1] ||
			    l->Get() == output[0] || l->Get() == output[1] ||
			    l->Get() == input[0] || l->Get() == input[1])
				// looks connected, try others!
				retList = l;
			else {
				// looks good, return it!
				lockNode(l);
				return l;
			}
		}
		l = l->Next();
	}

	// return bad choose if any (a connected node!)
	if (retList) {
		lockNode(retList);
		return retList;
	}
	return (KlogicList<QPoint> *)NULL;
}

// (selection method) select nodes contained in rect or all nodes
bool Wire::select(QRect r, bool all)
{	KlogicList<QPoint> *l = wire.First();
	bool ret = false;

	while(l) {
		if (all || r.contains(*l->Get())) {
			if (!selnode.With(l->Get())) selnode.Append(l->Get());
			ret = true;
		} else {
			selnode.Destroy(l->Get());
		}
		l = l->Next();
	}
	return ret;
}

// (selection method) unselect all nodes
bool Wire::isSelected()
{
	if (selnode.First()) return true;
	return false;
}

// (selection method) unselect all nodes
void Wire::unselect()
{
	selnode.Destroy();
}

// return minimum x cood over all selected nodes
int Wire::getMinSelX()
{	KlogicList<QPoint> *l = selnode.First();
	int ret;

	if (!l) return 0;
	else ret = l->Get()->x();

	while(l) {
		if (l->Get()->x() < ret) ret = l->Get()->x();
		l = l->Next();
	}
	return ret;
}

// return minimum x cood over all selected nodes
int Wire::getMaxSelX()
{	KlogicList<QPoint> *l = selnode.First();
	int ret;

	if (!l) return 0;
	else ret = l->Get()->x();

	while(l) {
		if (l->Get()->x() > ret) ret = l->Get()->x();
		l = l->Next();
	}
	return ret;
}

// return minimum y cood over all selected nodes
int Wire::getMinSelY()
{	KlogicList<QPoint> *l = selnode.First();
	int ret;

	if (!l) return 0;
	else ret = l->Get()->y();

	while(l) {
		if (l->Get()->y() < ret) ret = l->Get()->y();
		l = l->Next();
	}
	return ret;
}

// return minimum y cood over all selected nodes
int Wire::getMaxSelY()
{	KlogicList<QPoint> *l = selnode.First();
	int ret;

	if (!l) return 0;
	else ret = l->Get()->y();

	while(l) {
		if (l->Get()->y() > ret) ret = l->Get()->y();
		l = l->Next();
	}
	return ret;
}

// set size of the wire line
// only set by XWire
void Wire::setLineSize(int new_size)
{
	if (size != new_size) {
		old_size = size;
		enforceCompleteDrawing();
		size = new_size;
	}
}

int Wire::lockSelectedNode()
{	current_node = (QPoint *)NULL;
	from_node = (QPoint *)NULL;
	to_node = (QPoint *)NULL;

	if (selnode.First()) current_node = selnode.First()->Get();

	if (current_node) {
		o_node = *current_node;
		return 1;
	}
	return 0;
}

// move all selected nodes
void Wire::addPos(int dx, int dy)
{	KlogicList<QPoint> *l = selnode.First();
	QPoint *pt;

	while(l) {
		pt = l->Get();
		// do not move nodes connected to a device
		if (pt != input[0] && pt != input[1] && pt != output[0] && pt != output[1]) {
			// do not move nodes wich are references
			if (pt != io_point[0] && pt != io_point[1]) {
				pt->setX(pt->x() + dx);
				pt->setY(pt->y() + dy);
			}
		}
		l = l->Next();
	}
}

// creates new node, if not i/o and not end-node
// otherwise locks the node
int Wire::createNode(const QPoint& p)
{	KlogicList<QPoint> *l;
	QPoint *pt;

	// method contains() locks the node, if wire contains p
	if (NULL != (l = contains(p))) {
		pt = l->Get();

		// new node, if no successor and not connected, otherwise existing node
		if (l->Prev() && !l->Next()) {
			if ((pt != input[0]) && (pt != input[1]) && (pt != output[0]) && (pt != output[1]) && (pt != io_point[0]) && (pt != io_point[1])) {
				current_node = new QPoint(*pt);
				return lockNode(wire.Append(current_node));
			}
		}

		// new node, if no successor and not connected, otherwise existing node
		if (!l->Prev() && l->Next()) {
			if ((pt != input[0]) && (pt != input[1]) && (pt != output[0]) && (pt != output[1]) && (pt != io_point[0]) && (pt != io_point[1])) {
				current_node = new QPoint(*pt);
				return lockNode(wire.Insert(current_node, 0, pt));
			}
		}

		// new node, if wire with single node
		if (!l->Prev() && !l->Next()) {
			current_node = new QPoint(*pt);
			return lockNode(wire.Append(current_node));
		}

		// somewhat in between the wire..
		return 1;
	}
	return 0;
}

// locks point by content: used for import only!
// tries only to lock the first and the last node!
int Wire::lockNode(QPoint pt)
{	KlogicList<QPoint> *f = wire.First();
	QPoint *fp = f->Get();
	KlogicList<QPoint> *l = wire.Last();
	QPoint *lp = l->Get();

	if (*fp == pt && io_point[0] != fp && io_point[1] != fp)
		return lockNode(f);

	if (*lp == pt && io_point[0] != lp && io_point[1] != lp)
		return lockNode(l);

	return NODE_NOLOCK;
}

// lock a node
int Wire::lockNode(QPoint *pt)
{	KlogicList<QPoint> *l = wire.With(pt);

	if (l) return lockNode(l);
	return NODE_NOLOCK;
}

// lock a node in the list of nodes
int Wire::lockNode(KlogicList<QPoint> *l)
{	int ret = 0;

	current_node = l->Get();
	if (!l->Prev() && !l->Next()) {
		from_node = (QPoint *)NULL;
		to_node = (QPoint *)NULL;
		ret = NODE_SINGLE;
	}
	else if (l->Prev() && !l->Next()) {
		from_node = l->Prev()->Get();
		to_node = (QPoint *)NULL;
		ret = NODE_FRONT;
	}
	else if (!l->Prev() && l->Next()) {
		from_node = l->Next()->Get();
		to_node = (QPoint *)NULL;
		ret = NODE_FRONT;
	}
	else if (l->Prev() && l->Next()) {
		from_node = l->Prev()->Get();
		to_node = l->Next()->Get();
		ret = NODE_MIDDLE;
	}
	o_node = *current_node;
	return ret;
}

// lock the start node of the wire
bool Wire::lockLastNode()
{	current_node = (QPoint *)NULL;
	from_node = (QPoint *)NULL;
	to_node = (QPoint *)NULL;

	if (wire.First()) current_node = wire.First()->Get();

	if (current_node) {
		o_node = *current_node;
		return true;
	}
	return false;
}

// remove the node from the list
// (does not destroy the node)
void Wire::cutNode()
{	if (current_node) {
		wire.Destroy(current_node);
		io_point_foreign.Destroy(current_node);
		selnode.Destroy(current_node);
		if (io_point[0] == current_node) io_point[0] = (QPoint *)NULL;
		if (io_point[1] == current_node) io_point[1] = (QPoint *)NULL;
		current_node = (QPoint *)NULL;
	}
}

// remove the node from the list
// (destroys the node)
void Wire::removeNode()
{
	if (current_node) {
		wire.Destroy(current_node);
		io_point_foreign.Destroy(current_node);
		selnode.Destroy(current_node);
		if (io_point[0] == current_node) io_point[0] = (QPoint *)NULL;
		if (io_point[1] == current_node) io_point[1] = (QPoint *)NULL;
		if (from_node == current_node)
			from_node = (QPoint *)NULL;
		if (to_node == current_node)
			to_node = (QPoint *)NULL;
		delete current_node;
		current_node = (QPoint *)NULL;
	}
}

void Wire::releaseNode()
{	from_node = (QPoint *)NULL;
	current_node = (QPoint *)NULL;
	to_node = (QPoint *)NULL;
}

int Wire::countNodes()
{
	return wire.counter();
}

int Wire::isPart(QPoint *pt)
{	if (wire.With(pt)) return 1;
	return 0;
}

// check connection between wires
// given point lies near a node or a line of this wire
// if connection required, connect it (insert ref, lock node, set input/output)
QPoint *Wire::checkConnection(QPoint *pt)
{	KlogicList<QPoint> *l1 = (KlogicList<QPoint> *)NULL;
	KlogicList<QPoint> *l2 = (KlogicList<QPoint> *)NULL;
	int der_eq;
	int inbetw_eq;

	// first of all, check if point lies near a point of this wire
	if (NULL != (l1 = contains(*pt))) {
		// never connect input- or output nodes!
		if ((current_node == input[0]) || (current_node == input[1]) || (current_node == output[0]) || (current_node == output[1]))
			return (QPoint *)NULL;

		// insert foreign node into wire, append pt before the existing own node
		wire.Insert(pt, 0, l1->Get());
		io_point_foreign.Append(pt);
		// remove the existing (currently locked) node!
		if (!activeIsReference() && !activeIsForeign()) {
			removeNode();
		}
		l1 = wire.With(pt);
		if (!l1) fatal("Wire::checkConnection(..) node not inserted??");
		lockNode(l1);
	  	return current_node;
	}

	// check each line..
	l1 = wire.First();
	l2 = wire.First()->Next();
	while (l1 && l2) {
		// look if line goes through the given point
		der_eq = derive(*l1->Get(), *l2->Get(), *pt);
		// if it is so, look for coordinates..
		if (der_eq) {
			inbetw_eq = inBetween(*l1->Get(), *l2->Get(), *pt);
			if (inbetw_eq) {
				o_ins_node1 = *l1->Get();
				o_ins_node2 = *l2->Get();

				// insert foreign node, append pt between the existing two own nodes
				wire.Insert(pt, 0, l2->Get());
				io_point_foreign.Append(pt);
				lockNode(pt);
				return current_node;
			}
		}
		l1 = l2;
		l2 = l2->Next();
	}
	return (QPoint *)NULL;
}

// internal: derivation pt1:pt2 ~= pt1:pt?
int Wire::derive(QPoint pt1, QPoint pt2, QPoint pt)
{	double d1, d2, alpha, beta;

	long dx1, dy1, dx2, dy2;

	// reduce influence of node-distance by factor for pt1
	dx1 = pt1.x() - pt2.x();
	dy1 = pt1.y() - pt2.y();
	dx2 = pt1.x() + dx1 - pt.x();
	dy2 = pt1.y() + dy1 - pt.y();

	if (dy1 == 0) d1 = 0.0;
	else if (dx1 != 0) d1 = (double)dy1 / (double)dx1;
	else d1 = 999999.0;
	if (dy2 == 0) d2 = 0.0;
	else if (dx2 != 0) d2 = (double)dy2 / (double)dx2;
	else d2 = 999999.0;

	alpha = atan(d1);
	beta = atan(d2);

	if ((alpha >= beta - 0.05) && (alpha <= beta + 0.05)) return 1;
	return 0;
}

// internal: point between line(-rect)?
int Wire::inBetween(QPoint pt1, QPoint pt2, QPoint pt)
{	QRect r;

	if (pt1.x() > pt2.x()) {
		r.setX(pt2.x());
		r.setWidth(pt1.x() - pt2.x());
	}
	else {
		r.setX(pt1.x() - 1);
		r.setWidth(pt2.x() - pt1.x() + 2);
        }
	if (pt1.y() > pt2.y()) {
		r.setY(pt2.y());
		r.setHeight(pt1.y() - pt2.y());
	}
	else {
		r.setY(pt1.y() - 1);
		r.setHeight(pt2.y() - pt1.y() + 2);
	}
	if (r.contains(pt)) return 1;
	return 0;
}

QPoint Wire::getActive()
{	return *current_node;
}

QPoint *Wire::getActiveNode()
{	return current_node;
}

// returns index of active node (wire-connection; conn1 or conn2)
int Wire::getActiveConnIndex()
{	if (!current_node) return WFAIL;
	if (current_node == io_point[0]) return 0;
	if (current_node == io_point[1]) return 1;
	return WFAIL;
}

// returns idx, if active node is the input-node
int Wire::activeIsInput()
{
	if (!current_node) return WFAIL;
	if (current_node == input[0]) return 0;
	if (current_node == input[1]) return 1;
	return WFAIL;
}

// returns idx, if active node is the input-node
int Wire::nodeIsInput(QPoint *pt)
{
	if (!pt) return WFAIL;
	if (pt == input[0]) return 0;
	if (pt == input[1]) return 1;
	return WFAIL;
}

// returns idx, if active node is the output-node
int Wire::activeIsOutput()
{
	if (!current_node) return WFAIL;
	if (current_node == output[0]) return 0;
	if (current_node == output[1]) return 1;
	return WFAIL;
}

// returns idx, if active node is the output-node
int Wire::nodeIsOutput(QPoint *pt)
{
	if (!pt) return WFAIL;
	if (pt == output[0]) return 0;
	if (pt == output[1]) return 1;
	return WFAIL;
}

// returns true, if active node is start- or end-point of wire
bool Wire::activeIsEnd()
{
	if ((current_node == wire.First()->Get()) || 
	    (current_node == wire.Last()->Get())) return true;
	return false;
}

// returns 1, if active node is reference of a connection
bool Wire::activeIsReference()
{
	if ((current_node == io_point[0]) || (current_node == io_point[1])) return true;
	return false;
}

// returns 1, if active node is a connection node
//            which another wire posesses
bool Wire::activeIsForeign()
{
	if (io_point_foreign.With(current_node)) return true;
	return false;
}

// returns 1 if point is reference of a connection (first or last node)
// this wire posesses the node
bool Wire::nodeIsReference(QPoint *pt)
{	if ((pt == io_point[0]) || (pt == io_point[1])) {
		// lock the point
		current_node = pt;
		o_node = *current_node;
		return true;
	}
	return false;
}

// returns 1, if node is reference of a connection (first or last node of another wire)
// another wire posesses the node
bool Wire::nodeIsForeign(QPoint *pt)
{
	if (io_point_foreign.With(pt)) return true;
	return false;
}

bool Wire::completeDrawingIsEnforced()
{
	return bMustErase;
}

// erase everything before refreshing the wire next time
void Wire::enforceCompleteDrawing()
{
	bMustErase = true;
}

// draw wire after erasing active parts
void Wire::drawImage(QPainter *p)
{	KlogicList<QPoint> *l = wire.First();
	int f = 0;
	QColor &_color = color;

	if (bMustErase)
		eraseWire(p);

	while(l && l->Get() && l->Next() && l->Next()->Get()) {
		// draw selected nodes green
		if (selnode.With(l->Get()) || selnode.With(l->Next()->Get())) {
			if (!f) {
				f = 1;
				_color = Qt::green;
			}
		}
		else if (f) {
			f = 0;
		}
		p->setPen(QPen(_color, size));
		p->drawLine(*(l->Get()), *(l->Next()->Get()));
		l = l->Next();
	}
	if (io_point[0]) drawSolderedPoint(p, Qt::black, *io_point[0]);
	if (io_point[1]) drawSolderedPoint(p, Qt::black, *io_point[1]);
}

// erase whole wire
void Wire::eraseWire(QPainter *p)
{	KlogicList<QPoint> *l = wire.First();
	QPoint *p1, *p2;

	if (io_point[0]) drawSolderedPoint(p, Qt::white, *io_point[0]);
	if (io_point[1]) drawSolderedPoint(p, Qt::white, *io_point[1]);
	while(l && l->Next()) {
		p1 = l->Get();
		p2 = l->Next()->Get();
		if (bMustErase)
			p->setPen(QPen(Qt::white, old_size));
		else
			p->setPen(QPen(Qt::white, size));
		p->drawLine(*p1, *p2);

		restoreArry(p, p1->x(), p1->y(), p2->x(), p2->y());
		l = l->Next();
	}
	old_size = 0;
	bMustErase = false;
}

// erase first, active, and last part of wire
void Wire::erase(QPainter *p)
{	QPoint pt2;
	KlogicList<QPoint> *first = wire.First();
	QPoint *firstp = (QPoint *)NULL;
	if (first) firstp = first->Get();
	KlogicList<QPoint> *last = wire.Last();
	QPoint *lastp = (QPoint *)NULL;
	if (last) lastp = last->Get();

	if ((!firstp) || (!lastp)) return;

	if (o_start_node.isNull()) o_start_node = *firstp;
	if (o_end_node.isNull()) o_end_node = *lastp;

	// node inserted: erase old part of wire
	if (!o_ins_node1.isNull() && !o_ins_node2.isNull()) {
		// delete old wire-part and restore grid
		p->setPen(QPen(Qt::white, size));
		p->drawLine(o_ins_node1, o_ins_node2);
		restoreArry(p, o_ins_node1.x(), o_ins_node1.y(), o_ins_node2.x(), o_ins_node2.y());
		o_ins_node1.setX(0);
		o_ins_node1.setY(0);
		o_ins_node2.setX(0);
		o_ins_node2.setY(0);
	}

	// erase old active part of wire
	if (o_node.x() != -1 && o_node.y() != -1) {
		// delete old wire-part and restore grid
		if (from_node) {
			p->setPen(QPen(Qt::white, size));
			p->drawLine(*from_node, o_node);
			restoreArry(p, from_node->x(), from_node->y(), o_node.x(), o_node.y());
		}
		if (to_node) {
			p->setPen(QPen(Qt::white, size));
			p->drawLine(o_node, *to_node);
			restoreArry(p, to_node->x(), to_node->y(), o_node.x(), o_node.y());
		}
		// remove connection-point
		if (current_node && ((current_node == io_point[0]) || (current_node == io_point[1])))
			drawSolderedPoint(p, Qt::white, o_node);
		o_node.setX(-1);
		o_node.setY(-1);
	}

	// o_start_node: old position of first point of wire
	// firstp: new position of first node of wire
	if (o_start_node != *firstp) {
		// remove connection-point
		if ((firstp == io_point[0]) || (firstp == io_point[1]))
			drawSolderedPoint(p, Qt::white, o_start_node);
		// erase first part of wire if changed
		// value of old start-node (first node) is in o_start_node
		if (first && first->Next()) {
			pt2 = *first->Next()->Get();
			// erase old wire-part
			if (!o_start_node.isNull()) {
				p->setPen(QPen(Qt::white, size));
				p->drawLine(o_start_node, pt2);
			}
			// restore grid
			restoreArry(p, o_start_node.x(), o_start_node.y(), pt2.x(), pt2.y());
		}
		o_start_node = *firstp;
	}

	// o_end_node: old position of last point of wire
	// lastp: new position of last node of wire
	if (o_end_node != *lastp) {
		// remove connection-point
		if ((lastp == io_point[0]) || (lastp == io_point[1]))
			drawSolderedPoint(p, Qt::white, o_end_node);
		// erase last part of wire if changed
		// value of last end-node (last node) is in o_end_node
		if (last && last->Prev()) {
			pt2 =*last->Prev()->Get();
			// erase old wire-part
			if (!o_end_node.isNull()) {
				p->setPen(QPen(Qt::white, size));
				p->drawLine(o_end_node, pt2);
			}
			// restore grid
			restoreArry(p, o_end_node.x(), o_end_node.y(), pt2.x(), pt2.y());
		}
		o_end_node = *lastp;
	}
}

// draw active part of wire
void Wire::draw(QPainter *p)
{
	if (current_node) {
		p->setPen(QPen(color, size));
		if (from_node) p->drawLine(*from_node, *current_node);
		if (to_node) p->drawLine(*current_node, *to_node);
	}
}

// draw connection between wires
void Wire::drawSolderedPoint(QPainter *p, QColor _color, QPoint pt)
{	if (bMustErase)
		p->setPen(QPen(_color, old_size));
	else
		p->setPen(QPen(_color, size));
	p->setBrush(_color);
	p->drawPie(pt.x() - 2, pt.y() - 2, 4, 4, 0, 5760);
}

void Wire::restoreArry(QPainter *p, int x1, int y1, int x2, int y2)
{	int i, j, a, b, c, d;

	p->setPen(Qt::black);
	if (x1 < x2) {
		a = x1;
		b = x2;
	}
	else {
		b = x1;
		a = x2;
	}
	if (y1 < y2) {
		c = y1;
		d = y2;
	}
	else {
		d = y1;
		c = y2;
	}
	for(i=a;i <= b;i += Grid::GRID) {
		for(j=c; j <= d; j += Grid::GRID) {
			p->drawPoint(i,j);
		}
	}
}

void Wire::updateNode(QPoint p)
{
	align(&p);

	if (!current_node) fatal(NOACTIVE);
	o_node = *current_node;
	*current_node = p;
}

QPoint *Wire::align(QPoint *p)
{	p->setX((p->x() + Grid::GRIDHALF) / Grid::GRID * Grid::GRID);
	p->setY((p->y() + Grid::GRIDHALF) / Grid::GRID * Grid::GRID);
	return p;
}

void Wire::setColor(QColor c)
{
	color = c;
}

// remove useless nodes
void Wire::garbageCollection()
{	KlogicList<QPoint> *lp = wire.First();
	QPoint *last = (QPoint *)NULL;
	QPoint *act = (QPoint *)NULL;
	int modified = 0;
	int flag1, flag2, flag3, flag4;

	while(lp && lp->Get()) {
		act = lp->Get();
		if (last && (*last == *act)) {
			// try to remove point "act"
			flag1 = nodeIsReference(act);
			flag2 = nodeIsForeign(act);
			flag3 = nodeIsInput(act);
			flag4 = nodeIsOutput(act);
			if (!flag1 && !flag2 && flag3 == WFAIL && flag4 == WFAIL) {
				// remove point act
				lp->Destroy(act);
				if (from_node == act)
					from_node = (QPoint *)NULL;
				if (to_node == act)
					to_node = (QPoint *)NULL;
				if (current_node == act)
					current_node = (QPoint *)NULL;
				delete act;
				modified = 1;
			}

			// try to remove point "last"
			else {
				flag1 = nodeIsReference(last);
				flag2 = nodeIsForeign(last);
				flag3 = nodeIsInput(last);
				flag4 = nodeIsOutput(last);
				if (!flag1 && !flag2 && flag3 == WFAIL && flag4 == WFAIL) {
					// remove point last
					lp->Destroy(last);
					if (from_node == last)
						from_node = (QPoint *)NULL;
					if (to_node == act)
						to_node = (QPoint *)NULL;
					if (current_node == last)
						current_node = (QPoint *)NULL;
					delete last;
					modified = 1;
				}
			}

		}
		if (modified) {
			modified = 0;
			last = (QPoint *)NULL;
			lp = wire.First();
		} else {
			last = lp->Get();
			lp = lp->Next();
		}
	}
}
	
// set the active node as input-node, if it's an endpoint
int Wire::connectInput(int idx)
{
	if(current_node && (!to_node || !from_node)) {
		if (input[idx])
			return WFAIL;

		input[idx] = current_node;
		return idx;
	}
	fatal(NOACTIVECONN);
	return WFAIL;
}

// reset input-node
void Wire::disconnectInput(int idx)
{
	input[idx] = (QPoint *)NULL;
}

// set the active node as output-node, if it's an endpoint
int Wire::connectOutput(int idx)
{
	if(current_node && (!to_node || !from_node)) {
		if (output[idx])
			return WFAIL;

		output[idx] = current_node;
		return idx;
	}
	fatal(NOACTIVECONN);
	return WFAIL;
}

// reset output-node
void Wire::disconnectOutput(int idx)
{
	output[idx] = (QPoint *)NULL;
}

// set io-point for connection to another wire
int Wire::connectWire(int idx)
{
	if (current_node) {
		if (io_point[idx])
			return WFAIL;

		io_point[idx] = current_node;
		return idx;
	}
	return WFAIL;
}

void Wire::disconnectWire(int idx)
{
	io_point[idx] = (QPoint *)NULL;
}

// change position of input-node (device-connection nodes) on the currently locked node
void Wire::setInputPosition(QPoint newp)
{
	if (!current_node) fatal("Wire::setInputPosition: no current node!?");
	if (current_node == input[0]) *(input[0]) = newp;
	else if (current_node == input[1]) *(input[1]) = newp;
}

// change position of input-node (device-connection nodes)
void Wire::setInputPosition(QPoint newp, int idx)
{
	if (input[idx]) *(input[idx]) = newp;
	else fatal("Wire::setInputPosition: no connection point in >%d<", idx);
}

// change position of output-node (device-connection nodes)
void Wire::setOutputPosition(QPoint newp)
{
	if (!current_node) fatal("Wire::setOutputPosition: no current node!?");
	if (current_node == output[0]) *(output[0]) = newp;
	else if (current_node == output[1]) *(output[1]) = newp;
}

// change position of output-node (device-connection nodes)
void Wire::setOutputPosition(QPoint newp, int idx)
{
	if (output[idx]) *(output[idx]) = newp;
	else fatal("Wire::setOutputPosition: no connection point in >%d<", idx);
}

// get position of input-node (device-connection nodes)
QPoint Wire::getInputPosition(int idx, int selonly)
{
	QPoint p(0,0);
	if (input[idx]) {
		if (!selonly) return *(input[idx]);
		if (selnode.With(input[idx])) return *(input[idx]);
	}
	return p;
}

// get position of input-node (device-connection nodes)
QPoint Wire::getOutputPosition(int idx, int selonly)
{
	if (output[idx]) {
		if (!selonly) return *(output[idx]);
		if (selnode.With(output[idx])) return *(output[idx]);
	}

	return QPoint(0,0);
}

// get position of input/output-node (wire-connection nodes)
QPoint Wire::getIOPosition(int idx, int selonly)
{
	if (io_point[idx]) {
		if (!selonly) return *(io_point[idx]);
		if (selnode.With(io_point[idx])) return *(io_point[idx]);
	}
	return QPoint(0,0);
}

// import/export methods
void Wire::setExportSelected(bool selected)
{
	exp_selected = selected;
}

void Wire::setExportDelta(int dx, int dy)
{
	exp_dx = dx;
	exp_dy = dy;
}

// serialize
QString Wire::operator>> (QString &ret)
{	QString buf;
	bool first = true;
	QPoint pt;

	KlogicList<QPoint> *lp = wire.First();
	KlogicList<QPoint> *ls = selnode.First();

	while(lp) {
		if ((exp_selected && ls->With(lp->Get())) || !exp_selected) {
			pt.setX(lp->Get()->x() - exp_dx);
			pt.setY(lp->Get()->y() - exp_dy);
			if (first) {
				buf.sprintf("%d:%d", pt.x(), pt.y());
				first = false;
			} else buf.sprintf(",%d:%d", pt.x(), pt.y());
			ret += buf;
		}
		lp = lp->Next();
	}
	return ret;
}

// deserialize
bool Wire::operator<< (const QString &swire)
{
	if (swire.contains("{")) return string2wire(swire);

	int nodecounter = 0;
	int pos_delim = 0;
	int pos_y = 0;
	int pos_x = 0;
	int x;
	int y;
	QPoint *firstp = (QPoint *)NULL;
	QPoint *lastp = (QPoint *)NULL;

	while (pos_delim != -1) {
		pos_y = swire.find(':', pos_x);
		pos_delim = swire.find(',', pos_x);

		if (pos_y == -1) return false;
		x = swire.mid(pos_x, pos_y - pos_x).toInt();
		if (pos_delim == -1)
			y = swire.mid(pos_y + 1).toInt();
		else
			y = swire.mid(pos_y + 1, pos_delim - pos_y - 1).toInt();

		current_node = new QPoint(x + exp_dx, y + exp_dy);
		wire.Append(current_node);
		if (!firstp) firstp = current_node;
		lastp = current_node;
		nodecounter++;

		pos_x = pos_delim + 1;
	}
	if (firstp) o_start_node = *firstp;
	if (lastp) o_end_node = *lastp;

	if (nodecounter > 1) return true;
	return false;
}

// private, historic method
// makes a wire from a string containing points
// return 0 if unsuccessful
int Wire::string2wire(const char *swire)
{       QPoint pt(0,0);
	int nodecounter = 0;
	char *buf = (char *)swire;
	char *posa, *pose;
	QPoint *firstp = (QPoint *)NULL;
	QPoint *lastp = (QPoint *)NULL;
 
	while(NULL != (posa = strchr(buf, '{'))) {
		if (!(pose = strchr(buf, ':'))) return 0;
		*pose = '\0';
		pt.setX(atoi(++posa) + exp_dx);
		posa = pose + 1;
		if (!(pose = strchr(posa, '}'))) return 0;
		*pose = '\0';
		pt.setY(atoi(posa) + exp_dy);
		buf = pose + 1;
 
		current_node = new QPoint(pt);
		wire.Append(current_node);
		if (!firstp) firstp = current_node;
		lastp = current_node;
		nodecounter++;
	}
	if (firstp) o_start_node = *firstp;
	if (lastp) o_end_node = *lastp;
 
	if (nodecounter > 1)
		return 1;
	else
		return 0;
}

