/*************************************************/
/* methods for class Circuit                  */
/*                                               */
/* widget containing a circuit                   */
/* actions with/at objects in the net-widget     */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qfile.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qstring.h>
#include <qpoint.h>
#include <qpaintdevice.h>

#include "klogic.h"
#include "klogicIO.h"
#include "mainw.h"
#include "netw.h"
#include "circuit.h"
#include "xwire.h"
#include "devLib.h"
#include "selection.h"
#include "xmlExport.h"
#include "deviceTypes.h"
#include "deviceFactory.h"
#include "deviceMuxDemux.h"
#include "deviceOsc.h"
#include "deviceRAM.h"
#include "deviceJKFF.h"

/***************************************************/
/* methods of Circuit                           */
/***************************************************/
int Circuit::instance = 0;
int Circuit::magic_version = 0;	// to be removed soon!
DeviceFactory * Circuit::devFactory = DeviceFactory::getInstance();

	Circuit::Circuit()
:XDevice(DeviceType::fNET, 0, 0)
{

	instance++;
	m_parent = 0;
}

	Circuit::Circuit(int x, int y)
: XDevice(DeviceType::fNET, x, y)
{

	instance++;
	m_parent = 0;
}

	Circuit::Circuit(int x, int y, int size)
: XDevice(DeviceType::fNET, x, y, size)
{

	instance++;
	m_parent = 0;
}

Circuit::~Circuit()
{
	instance--;

	KlogicList<XDevice> *d = netdev.First();
	KlogicList<XWire> *w = netwire.First();
	KlogicList<Circuit> *n = netnet.First();

	// destroy all subcircuits
	while(n) {
		delete n->Get();
		netdev.Destroy((XDevice *)(n->Get()));
		n = n->Next();
	}
	// destroy all devices
	while(d) {
		delete d->Get();
		d = d->Next();
	}
	// destroy all wires
	while(w) {
		delete w->Get();
		w = w->Next();
	}
	// destroy lists
	netnet.Destroy();
	netdev.Destroy();
	netwire.Destroy();
	netin.Destroy();
	netout.Destroy();

	initImport();
}

bool Circuit::hasImpossibleEquations()
{
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		if (ld->Get()->type() == DeviceType::fRAM || ld->Get()->isTristate()) return true;
		ld = ld->Next();
	}
	return false;
}

// returns 1 if all names are unique, otherwise 0
int Circuit::checkCircuitNames()
{
	KlogicList<XDevice> *ld1 = netdev.First();
	KlogicList<XDevice> *ld2;

	while(ld1) {
		// sub circuits ...
		Circuit *net = ld1->Get()->devIsCircuit();
		if (net && !net->checkCircuitNames()) return 0;

		// devices ...
		ld2 = netdev.First();
		while(ld2) {
			if (ld2->Get() != ld1->Get()) {
				if (!strcmp(ld1->Get()->getName(), ld2->Get()->getName())) return 0;
			}
			ld2 = ld2->Next();
		}

		ld1 = ld1->Next();
	}
	return 1;
}

// change all duplicate names to be unique
int Circuit::unifyCircuitNames()
{
	KlogicList<XDevice> *ld1 = netdev.First();
	KlogicList<XDevice> *ld2;

	while(ld1) {
		// sub circuits ...
		Circuit *net = ld1->Get()->devIsCircuit();
		if (net && !net->checkCircuitNames()) return 0;

		// devices ...
		ld2 = netdev.First();
		while(ld2) {
			if (ld2->Get() != ld1->Get()) {
				if (!strcmp(ld1->Get()->getName(), ld2->Get()->getName())) {
					QString buffer;
					buffer.sprintf("uni%d", ld2->Get()->getID());
					setName(ld2->Get()->getID(), buffer);
				}
			}
			ld2 = ld2->Next();
		}

		ld1 = ld1->Next();
	}
	return 1;
}

void Circuit::getAllEquations(map<QString,OutputInfo>& oEquations, const QString& sQPrefix, bool bIsolate)
{
	QString sNewPrefix = sQPrefix + getName() + ".";
	sNewPrefix.replace(QRegExp(" "), "_" );

	KlogicList<XDevice> *ld = netdev.First();

	while(ld) {
		Device *dev = ld->Get();
		dev->getAllEquations(oEquations, sNewPrefix, bIsolate);
		ld = ld->Next();
	}
}

void Circuit::deleteCircuit(QPainter *p)
{
	// destroy all subcircuits
	KlogicList<Circuit> *n = netnet.First();
	while(n) {
		//n->Get()->deleteCircuit(p);
		delete n->Get();
		netdev.Destroy((XDevice *)(n->Get()));
		n = n->Next();
	}
	// destroy all devices
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		deleteDevice(p, ld->Get());
		ld = netdev.First();
	}
	// destroy all wires
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		deleteWire(lw->Get());
		lw = netwire.First();
	}
	// destroy lists
	netnet.Destroy();
	netdev.Destroy();
	netwire.Destroy();
	netin.Destroy();
	netout.Destroy();
	Device::resetCounter();
}

Circuit * Circuit::rootParent()
{
	Circuit *top;

	top = this;
	while(top->parent()) top = top->parent();
	return top;
}

void Circuit::setParent(XDevice *parent)
{
	m_parent = parent->devIsCircuit();
}

Circuit * Circuit::parent()
{
	return m_parent;
}

KlogicList<XDevice> *Circuit::devPointer()
{
	return &netdev;
}

bool Circuit::empty()
{
	if (!netnet.First() && !netdev.First() &&
			!netwire.First() && !netin.First() && !netout.First())
		return true;
	return false;
}

void Circuit::applyDefaults()
{
	KlogicList<XDevice> *ld = netdev.First();
	Circuit *net;

	setDelay(Device::defDelay());
	setInverted(XDevice::isDefInverted());
	setWidth(XDevice::defWidth());
	setUndef(Device::defUndefined());
	invertTrigger(Device::triggerInverted());
	displayName(XDevice::nameDisplayedGlobal());
	setImage();

	while(ld) {
		net = ld->Get()->devIsCircuit();
		if (net) net->applyDefaults();
		else {
			ld->Get()->setDelay(Device::defDelay());
			ld->Get()->setInverted(XDevice::isDefInverted());
			ld->Get()->setWidth(XDevice::defWidth());
			ld->Get()->setUndef(Device::defUndefined());
			ld->Get()->invertTrigger(Device::triggerInverted());
			ld->Get()->displayName(XDevice::nameDisplayedGlobal());
			ld->Get()->setEquation();
			ld->Get()->parseEquation();
			setName(ld->Get()->getID(), ld->Get()->getName());
			ld->Get()->setImage();
		}
		ld = ld->Next();
	}
}

void Circuit::mapImportID(int old_id, int dev_id)
{
	XDevice *devi = netin.Get(dev_id);
	XDevice *devo = netout.Get(dev_id);

	if (devi) {
		KlogicList<Value> *lv = named_input.With(old_id);
		if (lv) {
			lv->setText(devi->getName());
			lv->setID1(dev_id);
		}
	}

	if (devo) {
		KlogicList<Calculator> *lop = named_output.With(old_id);
		if (lop) {
			lop->setText(devo->getName());
			lop->setID1(dev_id);
		}
	}
}

void Circuit::setName(int dev_id, const QString& new_text)
{
	XDevice *d = netdev.Get(dev_id);
	if (d) {
		d->setName(new_text);
		if ((d->type() == DeviceType::fIN || d->type() == DeviceType::fOUT) && !new_text.isEmpty() && new_text.length()) {
			// actualize names within circuit image
			if (d->type() == DeviceType::fIN) {
				KlogicList<Value> *lv = named_input.With(dev_id);
				if (lv) lv->setText(new_text);
				if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
			}
			if (d->type() == DeviceType::fOUT) {
				KlogicList<Calculator> *lop = named_output.With(dev_id);
				if (lop) lop->setText(new_text);
				if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
			}
		}
	}
}

bool Circuit::hasBooleanOutput(int output_id)
{
	map<int, bool>::iterator iter = m_mHasBooleanOutput.find(output_id);
	if (iter != m_mHasBooleanOutput.end())
		return iter->second;
	return true;
}

void Circuit::setHasBooleanOutput(int output_id, bool bHasBooleanInput)
{
	m_mHasBooleanOutput[output_id] = bHasBooleanInput;
}

void Circuit::garbageCollection(QPainter *p)
{
	KlogicList<XWire> *lw = netwire.First();

	while(lw) {
		lw->Get()->Wire::garbageCollection();
		if (!wireOK(p, lw->Get())) {
			lw = netwire.First();
		} else {
			lw = lw->Next();
		}
	}
}

//*************************************************************************
//* drawing methods
//*************************************************************************

// erase device, then draw all
void Circuit::drawDev(QPaintDevice *paintd, QPainter *p, XDevice *dev)
{
	// erase device and its connected wires
	dev->erase(p);
	// redraw all
	drawAll(paintd, p);
}

// draw all devices and wires in the net
// if paintd is NULL then drawing device is a printer
void Circuit::drawAll(QPaintDevice *paintd, QPainter *p)
{
	KlogicList<XDevice> *ld = netdev.First();
	KlogicList<XWire> *lw = netwire.First();

	while(ld) {
		if (paintd) ld->Get()->drawImage(paintd, p);
		else ld->Get()->drawImage(p);
		ld = ld->Next();
	}
	while(lw) {
		if (paintd) {
			lw->Get()->drawImage(p);
		}
		else {
			// printer output
			lw->Get()->Wire::setColor(Qt::black);
			lw->Get()->drawImage(p);
			lw->Get()->setColor(true);
		}
		lw = lw->Next();
	}
}

// force setting status dependant images
void Circuit::forceOutputChange()
{
	Device::forceOutputChange();
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		ld->Get()->forceOutputChange();
		ld = ld->Next();
	}
}

// draw all wires with its actual potential
// draw all devices with a changed image
// returns wether a runtime shortcut occured (WSHORT) or not (0)
int Circuit::drawStatus(QPaintDevice *paintd, QPainter *p)
{
	int ret_code = 0;
	int ret_tmp;

	// some devices like 7S or LED
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		// if image changes..
		if (ld->Get()->setColor()) {
			// ..draw device
			ld->Get()->drawImage(paintd, p);
		}
		ld = ld->Next();
	}

	// draw all wires in color depending of device-output
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		// if color changes..
		if (0 != (ret_tmp = lw->Get()->setColor())) {
			// ..draw wire
			lw->Get()->drawImage(p);
			// .. runtime shortcut detection
			if (ret_tmp == WSHORT) ret_code = WSHORT;
		}
		lw = lw->Next();
	}

	// .. runtime shortcut detection
	return ret_code;
}

void Circuit::setUndef(int undef)
{
	KlogicList<XDevice> *ld = netout.First();
	while(ld) {
		ld->Get()->setUndef(undef);
		ld = ld->Next();
	}
	Device::setUndef(undef);
}

//*************************************************************************
//* device methods
//*************************************************************************

// create new device
XDevice * Circuit::newDevice(int function, int x, int y, int size)
{
	return devFactory->createDevice(this, function, x, y, size);
}

// add new device
bool Circuit::setChild(XDevice *dev, bool bImport)
{
	int type = dev->type();
	if (type == DeviceType::fIN || type == DeviceType::fOUT) {
		QString text = dev->getName();
		if (type == DeviceType::fIN) {
			if (!bImport) {
				if (ADDNAMED_OK > addInputName(text, -1, dev->getID()))
					return false;
			}
			netin.Append(dev, dev->getID());
		}
		if (type == DeviceType::fOUT) {
			if (!bImport) {
				QString text = dev->getName();
				if (ADDNAMED_OK > addOutputName(text, -1, dev->getID()))
					return false;
			}
			dev->setUndef(undef());
			netout.Append(dev, dev->getID());
		}
	}
	if (dev->devIsCircuit())
		netnet.Append(dev->devIsCircuit(), dev->getID());

	netdev.Append(dev, dev->getID());

	return true;
}


// virtual
// returns pointer to the net-device if device is a net-device
Circuit * Circuit::devIsCircuit()
{
	return this;
}

// look for a device containing point
XDevice * Circuit::containsDev(QPoint p)
{
	KlogicList<XDevice> *l = netdev.First();

	while(l) {
		if (l->Get()->contains(p))
			return l->Get();
		l  = l->Next();
	}
	return (XDevice *)NULL;
}

// look for a device containing device
XDevice * Circuit::containsDev(XDevice *dev)
{
	KlogicList<XDevice> *l = netdev.First();

	while(l) {
		if (l->Get() != dev) {
			if (l->Get()->contains(dev->getPos())) {
				return l->Get();
			}
		}
		l  = l->Next();
	}
	return (XDevice *)NULL;
}

// mark all devices in rect as selected
void Circuit::selectDev(QRect r, KlogicList<XDevice> *d)
{
	KlogicList<XDevice> *l = netdev.First();

	while(l) {
		if (l->Get()->contains(r)) {
			if (!d->With(l->Get())) d->Append(l->Get());
			l->Get()->select(true);
		}
		else {
			d->Destroy(l->Get());
			l->Get()->select(false);
		}
		l = l->Next();
	}
}

// called from class Selection
// add all selected devices to the selection (paste operation)
void Circuit::grabSelection(Selection *sel)
{
	KlogicList<XDevice> *ld = netdev.First();
	KlogicList<XWire> *lw = netwire.First();

	while(ld) {
		if (ld->Get()->isSelected()) sel->add(ld->Get());
		ld = ld->Next();
	}
	while(lw) {
		if (lw->Get()->isSelected()) sel->add(lw->Get());
		lw = lw->Next();
	}
}

// delete a device
void Circuit::deleteDevice(QPainter *p, XDevice *dev)
{
	if (dev->type() == DeviceType::fIN) {
		netin.Destroy(dev);
		removeInputName(dev->getID());
	}
	if (dev->type() == DeviceType::fOUT) {
		netout.Destroy(dev);
		removeOutputName(dev->getID());
	}

	// erase bitmap from screen
	dev->erase(p);

	// delete device reference in list
	Circuit *net = dev->devIsCircuit();
	if (net) {
		net->deleteCircuit(p);
		netnet.Destroy(net);
	}
	netdev.Destroy(dev);

	// delete device, in destructor: delete connections
	delete dev;
}

//*************************************************************************
//* wire methods
//*************************************************************************

// look for a wire containing point
XWire * Circuit::containsWire(QPoint p)
{
	KlogicList<XWire> *l = netwire.First();

	while(l) {
		// if wire contains point, the node will be locked automaticly
		// if not, a new node is created
		if (l->Get()->createNode(p)) return l->Get();
		l = l->Next();
	}
	return (XWire *)NULL;
}

// look for a wire containing (parts of) rect and select them
void Circuit::selectWire(QRect r, KlogicList<XWire> *w)
{
	KlogicList<XWire> *l = netwire.First();

	while(l) {
		// catch nodes, mark them as selected
		if (l->Get()->select(r)) {
			if (!w->With(l->Get())) w->Append(l->Get());
		}
		else {
			w->Destroy(l->Get());
		}
		l = l->Next();
	}
}

int Circuit::removeNode(QPainter *p, QPoint pt)
{
	KlogicList<XWire> *l = netwire.First();
	XWire *w1, *w2;
	int ret = 0;
	int nextl = 1;

	// can remove more than one node if required
	while(l) {
		w1 = l->Get();
		if (w1->contains(pt)) {
			w1->erase(p);
			w2 = w1->removeNode();
			if (!wireOK(p, w1)) {
				nextl = 0;
				l = netwire.First();
			}
			if (w2 && !wireOK(p, w2)) {
				nextl = 0;
				l = netwire.First();
			}
			ret++;
		}
		if (nextl) l = l->Next();
		else nextl = 1;
	}
	return ret;
}

// look for a device or wire, connected to given wire
// connections will be made automaticly on active node of wire
int Circuit::checkConnection(XWire *w)
{
	KlogicList<XDevice> *ld = netdev.First();
	KlogicList<XWire> *lw = netwire.First();
	int ret;

	// look for device connection/disconnection
	while(ld) {
		if (DOK != (ret = ld->Get()->checkConnection(w))) {
			w->setColor(true);
			return ret;
		}
		ld = ld->Next();
	}
	// look for wire connection/disconnection
	while(lw) {
		if (w != lw->Get()) {
			if (WOK != (ret = lw->Get()->checkConnection(w))) {
				w->setColor(true);
				return ret;
			}
		}
		lw = lw->Next();
	}
	return NOK;
}

// look for a device, connected to given wire
// (used as import-method only)
int Circuit::checkConnection(int devid, int inverted, XWire *w)
{
	XDevice *d = netdev.Get(devid);
	int ret;

	// look for new device connection
	if (d) {
		if (DOK != (ret = d->checkConnection(w, inverted))) return ret;
	}
	return NFAIL;
}

// create a new wire (import)
XWire *Circuit::newWire()
{
	XWire *w = new XWire();
	netwire.Append(w, w->getID());
	return w;
}

// get a wire (import)
XWire *Circuit::getWire(int id)
{
	return netwire.Get(id);
}

// create a new wire
XWire *Circuit::newWire(QPainter *p, QPoint pt)
{
	XWire *w;
	int ret;

	// create new wire and its first node
	w = new XWire(pt);

	// look for connection in the first point
	ret = checkConnection(w);
	switch (ret) {
		case WCONN:
		case DCONN:
			pt = w->getActive();	// same point, but aligned to the grid
			break;
	}

	// wire possibly has changed
	w->erase(p);
	w->draw(p);

	// lock new wire, create second node
	if (!w->createNode(pt)) {
		delete w;
		warning(NEWWIRE);
		return (XWire *)NULL;
	}
	// append new wire to wirelist
	netwire.Append(w, w->getID());
	return w;
}

// no wires with single nodes allowed
int Circuit::wireOK(QPainter *p, XWire *w)
{
	XWire *w2;

	if (w->countNodes() < 2) {
		w->lockLastNode();
		// erase last locked node
		w->erase(p);
		w2 = w->removeNode();
		if (w2) wireOK(p, w2);	// recursive call
		// remove nodeless wire
		netwire.Destroy(w);
		delete w;
		return 0;
	}
	return 1;
}

// wire released after clicked with the mouse
int Circuit::releaseWire(QPainter *p, XWire *w)
{
	w->Wire::garbageCollection();

	// check, that wire has at leat two nodes
	if (!wireOK(p, w)) return 0;
	w->releaseNode();
	return 1;
}

// delete wire
void Circuit::deleteWire(XWire *w)
{
	netwire.Destroy(w);
	delete w;
}

//*************************************************************************
//* calculation methods
//*************************************************************************

// set net to init-state (flush all calculation-fifos)
void Circuit::flush()
{
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		ld->Get()->flush(0);
		ld = ld->Next();
	}
}

// calculate burst (normally bust is 1)
void Circuit::Burst(int burst)
{	int burst_step;

	switchInterface(IF_INPUT);

	// calculate new values
	for (burst_step = 0; burst_step < burst; burst_step++) {
		Calculate(burst_step);
		Propagate(burst_step);
	}

	switchInterface(IF_OUTPUT);
}

// switch input- or output devices of this
void Circuit::switchInterface(int if_type)
{
	KlogicList<XDevice> *ld;
	KlogicList<Circuit> *ln;

	// set output-values of input-devices by requesting named inputs of this sub circuit
	// input devices are part of this, and connected to named inputs of this
	if (if_type == IF_INPUT) {
		ld = netin.First();
		while(ld) {
			XDevice *oInputDev = ld->Get();
			// set input device from the named input value
			bool bIsBoolean;
			int iVal = input(oInputDev->getID(), bIsBoolean);
			oInputDev->setStaticInput(iVal);
			oInputDev->setHasBooleanOutput(0, bIsBoolean);
			ld = ld->Next();
		}
	}

	// set output values of named outputs by requesting output devices
	// output devices are part of this, and named outputs are connected to them
	if (if_type == IF_OUTPUT) {
		ld = netout.First();
		while (ld) {
			XDevice *oOutputDev = ld->Get();
			// set the named output value from output device
			setStaticOutput(oOutputDev->getID(), oOutputDev->output());
			setHasBooleanOutput(oOutputDev->getID(), oOutputDev->hasBooleanOutput(0));
			ld = ld->Next();
		}
	}

	ln = netnet.First();
	while(ln) {
		ln->Get()->switchInterface(if_type);
		ln = ln->Next();
	}
}

// virtual
// calculate all devices of this circuit and all sub circuits
void Circuit::Calculate(int burst_step)
{
	KlogicList<XDevice> *ld = netdev.First();
	while(ld) {
		ld->Get()->Calculate(burst_step);       // recursive by virtual
		ld = ld->Next();
	}
}

// virtual
// propagate values to the outputs of this circuit and all sub circuits
void Circuit::Propagate(int burst_step)
{
	KlogicList<XDevice> *ld;
	ld = netdev.First();
	while(ld) {
		ld->Get()->Propagate(burst_step);	// recursive by virtual
		ld = ld->Next();
	}
}

/***************************************************/
/* import/export operations on a net-widget        */
/***************************************************/
// XML export method for library devices: export to XmlDevice
bool Circuit::exportXML(XmlNet& net)
{
	return exportXML((QTextStream *)NULL, &net, false, 0, 0, 0);
}

// XML export method for files: write directly to the file
bool Circuit::exportXML(QTextStream &ts, bool selected, int level, int dx, int dy)
{
	// top level circuit with selected flag ignored!
	XmlNet net(this, false, level, dx, dy);
	return exportXML(&ts, &net, selected, level, dx, dy);
}

// XML export method
bool Circuit::exportXML(QTextStream *ts, XmlNet *net, bool selected, int level, int dx, int dy)
{
	bool ownNet = (net == (XmlNet *)NULL);

	// circuit parameter and header
	if (ownNet) {
		net = new XmlNet(this, selected, level, dx, dy);
		if (!net->isValid()) {
			delete net;
			return false;
		}
	}

	if (ts) {
		*ts << net->getFieldStart();
		*ts << net->getFieldContent();
		level++;
	}

	// circuit wires
	KlogicList<XWire> *lw = netwire.First();
	while(lw) {
		XmlWire wire(lw->Get(), selected, level, dx, dy);
		if (wire.isValid()) {
			if (ts) *ts << wire.getField();
			else net->addContent(wire.getField());
		}

		lw = lw->Next();
	}

	// circuit devices
	KlogicList<XDevice> *ld = netdev.First();
	Circuit *pNet;
	while(ld) {
		pNet = ld->Get()->devIsCircuit();
		if (pNet) {
			XmlNet subCircuit(pNet, selected, level, dx, dy);
			if (subCircuit.isValid()) {
				// for sub-circuits always set selected=false!
				pNet->exportXML(ts, &subCircuit, false, level, 0, 0);	// recursion!

				// add net content (if ts avail: content already added)
				if (!ts) net->addContent(subCircuit.getField());
			}
		} else {
			XmlDevice dev(ld->Get(), selected, level, dx, dy);
			if (dev.isValid()) {
				if (ts) *ts << dev.getField();
				else net->addContent(dev.getField());
			}
		}
		ld = ld->Next();
	}

	// circuit wire <-> wire connections
	lw = netwire.First();
	while(lw) {
		XmlWireConnection wire_conn(lw->Get(), selected, level, dx, dy);
		if (wire_conn.isValid()) {
			if (ts) *ts << wire_conn.getField();
			else net->addContent(wire_conn.getField());
		}

		lw = lw->Next();
	}

	// circuit device <-> wire connections
	lw = netwire.First();
	while(lw) {
		XmlDeviceConnection dev_conn(lw->Get(), selected, level, dx, dy);
		if (dev_conn.isValid()) {
			if (ts) *ts << dev_conn.getField();
			else net->addContent(dev_conn.getField());
		}

		lw = lw->Next();
	}

	if (ts) {
		level--;
		*ts << net->getFieldEnd();
	}

	if (ownNet) {
		delete net;
	}
	return true;
}

// proprietary export format, recursively called!
// subnet is true for the very first circuit, if a sub circuit should be stored
// for all lower circuits, subnet is always true!
bool Circuit::exportNet(QTextStream &ts, bool selected, bool subnet, int dx, int dy)
{
	QString s;

	if (subnet) ts << IO_B_SUBNET << "\n";
	else ts << IO_B_NET << "\n";

	ts << exportInstance() << "\n";

	// write wire position information
	ts << IO_B_WIRE << "\n";
	s = exportWires(selected, dx, dy);
	if (!s.isEmpty()) ts << s << "\n";
	ts << IO_E_WIRE << "\n";

	// write device type/position information
	ts << IO_B_DEVICE << "\n";
	s = exportDevices(selected, dx, dy);
	if (!s.isEmpty()) ts << s << "\n";
	ts << IO_E_DEVICE << "\n";

	// write subnet information
	KlogicList<Circuit> *ln = netnet.First();
	while(ln) {
		if ((selected && ln->Get()->isSelected()) || !selected) {
			if (!ln->Get()->exportNet(ts, false, true, 0, 0)) {
				return false;
			}
		}
		ln = ln->Next();
	}

	// write wire-wire connections
	ts << IO_B_CONNWIRE << "\n";
	s = exportWireConn(selected, dx, dy);
	if (!s.isEmpty()) ts << s << "\n";
	ts << IO_E_CONNWIRE << "\n";

	// write wire-device connections
	ts << IO_B_CONNDEVICE << "\n";
	s = exportDevConn(selected, dx, dy);
	if (!s.isEmpty()) ts << s << "\n";
	ts << IO_E_CONNDEVICE << "\n";

	if (subnet) ts << IO_E_SUBNET << "\n";
	else ts << IO_E_NET << "\n";

	return true;
}

