/***************************************************************************
  preferences.h
  -------------------
  A class to access persistant preferences for an application. Utilizes XML/DOM.
  Basic format is:
    <!DOCTYPE preferences>
    <preferences version="0.1" application="MyApp" >
        <group name="Default" >
            <option key="alpha" value="true" />
            <option key="beta" value="99" />
            <option key="gamma" value="test" />
        </group>
    </preferences>
  -------------------
  begin         Tue Sep 12 2000
  author        David Johnson, david@usermode.org
  -------------------
  Copyright 2000, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef PREFERENCES_H
#define PREFERENCES_H

#include <qapplication.h>
#include <qstring.h>
#include <qmap.h>
#include <qfont.h> //for the font saving stuff

class QColor;
class QDomElement;

class Preferences {
public:
    // constructor
    Preferences(const QString& filename, const QString& format, const QString& version);
    // destructor
    virtual ~Preferences();

    // preference file information
    const QString& format();
    const QString& version();

    // group settings
    const QString& getGroup();
    void setGroup(const QString& group);

    // boolean data storage
    bool getBool(const QString& key, bool def = false);
    void setBool(const QString& key, bool value);
    // number data storage
    long getNumber(const QString& key, long def = 0);
    void setNumber(const QString& key, long value);
    // string data storage
    QString getCDATA(const QString& key, const QString& def = "NULL");
    QString getString(const QString& key, const QString& def = "NULL");
    QString getAttr1(const QString& key, const QString& def = "NULL");
    QString getAttr2(const QString& key, const QString& def = "NULL");
    QString getAttr3(const QString& key, const QString& def = "NULL");
    void setCDATA(const QString& key, const QString& def = QString::null);
    void setAttr1(const QString& key, const QString& def = QString::null);
    void setAttr2(const QString& key, const QString& def = QString::null);
    void setAttr3(const QString& key, const QString& def = QString::null);
    void setString(const QString& key, const QString& value, const QString& attr1 = QString::null, 
										const QString &attr2 = QString::null, const QString &attr3 = QString::null);

		// font data storage
		QFont getFont(const QString& key, QFont &def);
		void setFont(const QString& key, QFont &value);

    // remove a key/value from the preferences
    void removeKey(const QString& key);
    void removeGroup();

    // flush the preferences out to file
    bool flush();

protected:
    // serialization
    void readData();
    bool writeData();
    void processGroup(QDomElement group);

private:
    bool _dirty;
    QString _currentgroup;
    QString _file;
    QString _format;
    QString _version;
    QString _buffer;

		typedef struct {
			//QString key;
			QString value;
			QString attr1;
			QString attr2;
			QString attr3;
			bool cdata;
		} Pref;
    typedef QMap<QString, Pref> PrefMap;
    QMap<QString, PrefMap> _groups;
};

//////////////////////////////////////////////////////////////////////////////
// Inline methods

inline const QString& Preferences::format() { return _format; }

inline void Preferences::setGroup(const QString& group) { _currentgroup = group; }

inline const QString& Preferences::getGroup() { return _currentgroup; }

#endif // PREFERENCES
