/*
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 * 
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 * 
 * The Original Code is mozilla.org code.
 * 
 * The Initial Developer of the Original Code is Christopher Blizzard.
 * Portions created by Christopher Blizzard are Copyright (C)
 * Christopher Blizzard.  All Rights Reserved.
 * 
 * Contributor(s):
 *   Christopher Blizzard <blizzard@mozilla.org>
 *   Ramiro Estrugo <ramiro@eazel.com>
 */

#include <stdio.h>

#include "gtkmozembed.h"
#include "gtkmozembedprivate.h"
#include "gtkmozembed_internal.h"
#include "kz-marshalers.h"

#include "EmbedPrivate.h"
#include "EmbedWindow.h"

// so we can do our get_nsIWebBrowser later...
#include <nsIWebBrowser.h>

class nsIDirectoryServiceProvider;

// class and instance initialization

static void
gtk_moz_embed_class_init(GtkMozEmbedClass *klass);

static void
gtk_moz_embed_init(GtkMozEmbed *embed);

// GtkObject methods

static void
gtk_moz_embed_destroy(GtkObject *object);

// GtkWidget methods

static void
gtk_moz_embed_realize(GtkWidget *widget);

static void
gtk_moz_embed_unrealize(GtkWidget *widget);

static void
gtk_moz_embed_size_allocate(GtkWidget *widget, GtkAllocation *allocation);

static void
gtk_moz_embed_map(GtkWidget *widget);

static void
gtk_moz_embed_unmap(GtkWidget *widget);

#ifdef MOZ_ACCESSIBILITY_ATK
static AtkObject* gtk_moz_embed_get_accessible (GtkWidget *widget);
#endif

static gint
handle_child_focus_in(GtkWidget     *aWidget,
		      GdkEventFocus *aGdkFocusEvent,
		      GtkMozEmbed   *aEmbed);

static gint
handle_child_focus_out(GtkWidget     *aWidget,
		       GdkEventFocus *aGdkFocusEvent,
		       GtkMozEmbed   *aEmbed);

// globals for this type of widget

static GtkBinClass *embed_parent_class;

guint moz_embed_signals[EMBED_LAST_SIGNAL] = { 0 };

// GtkObject + class-related functions

GType
gtk_moz_embed_get_type(void)
{
  static GType moz_embed_type = 0;
  if (!moz_embed_type)
  {
    static const GTypeInfo moz_embed_info =
    {
      sizeof(GtkMozEmbedClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gtk_moz_embed_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (GtkMozEmbed),
      0,
      (GInstanceInitFunc) gtk_moz_embed_init,
    };

    moz_embed_type = g_type_register_static (GTK_TYPE_BIN, "GtkMozEmbed",
					     &moz_embed_info, (GTypeFlags)0);
  }

  return moz_embed_type;
}

static void
gtk_moz_embed_class_init(GtkMozEmbedClass *klass)
{
  GtkWidgetClass     *widget_class;
  GtkObjectClass     *object_class;
  
  widget_class    = GTK_WIDGET_CLASS(klass);
  object_class    = GTK_OBJECT_CLASS(klass);

  embed_parent_class = (GtkBinClass *) g_type_class_peek_parent(klass);

  widget_class->realize = gtk_moz_embed_realize;
  widget_class->unrealize = gtk_moz_embed_unrealize;
  widget_class->size_allocate = gtk_moz_embed_size_allocate;
  widget_class->map = gtk_moz_embed_map;
  widget_class->unmap = gtk_moz_embed_unmap;

#ifdef MOZ_ACCESSIBILITY_ATK
  widget_class->get_accessible = gtk_moz_embed_get_accessible;
#endif

  object_class->destroy = gtk_moz_embed_destroy;
  
  // set up our signals

  moz_embed_signals[LINK_MESSAGE] = 
    g_signal_new ("link_message",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET(GtkMozEmbedClass, link_message),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  moz_embed_signals[JS_STATUS] =
    g_signal_new ("js_status",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET(GtkMozEmbedClass, js_status),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  moz_embed_signals[LOCATION] =
    g_signal_new ("location",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET(GtkMozEmbedClass, location),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
  moz_embed_signals[TITLE] = 
    g_signal_new("title",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, title),
		 NULL, NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  moz_embed_signals[PROGRESS] =
    g_signal_new("progress",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, progress),
		 NULL, NULL,
		 _kz_marshal_VOID__INT_INT,
		 G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_INT);
  moz_embed_signals[PROGRESS_ALL] = 
    g_signal_new("progress_all",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, progress_all),
		 NULL, NULL,
		 _kz_marshal_VOID__POINTER_INT_INT,
		 G_TYPE_NONE, 3, G_TYPE_POINTER,
		 G_TYPE_INT, G_TYPE_INT);
  moz_embed_signals[NET_STATE] =
    g_signal_new("net_state",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, net_state),
		 NULL, NULL,
		 _kz_marshal_VOID__INT_INT,
		 G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_UINT);
  moz_embed_signals[NET_STATE_ALL] =
    g_signal_new("net_state_all",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, net_state_all),
		 NULL, NULL,
		 _kz_marshal_VOID__POINTER_INT_INT,
		 G_TYPE_NONE, 3, G_TYPE_POINTER,
		 G_TYPE_INT, G_TYPE_UINT);
  moz_embed_signals[NET_START] =
    g_signal_new("net_start",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, net_start),
		 NULL, NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  moz_embed_signals[NET_STOP] =
    g_signal_new("net_stop",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, net_stop),
		 NULL, NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  moz_embed_signals[NEW_WINDOW] =
    g_signal_new("new_window",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, new_window),
		 NULL, NULL,
		 _kz_marshal_VOID__POINTER_INT,
		 G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_UINT);
  moz_embed_signals[VISIBILITY] =
    g_signal_new("visibility",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, visibility),
		 NULL, NULL,
		 g_cclosure_marshal_VOID__BOOLEAN,
		 G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
  moz_embed_signals[DESTROY_BROWSER] =
    g_signal_new("destroy_browser",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, destroy_brsr),
		 NULL, NULL,
		 g_cclosure_marshal_VOID__VOID,
		 G_TYPE_NONE, 0);
  moz_embed_signals[OPEN_URI] = 
    g_signal_new("open_uri",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, open_uri),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[SIZE_TO] =
    g_signal_new("size_to",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, size_to),
		 NULL, NULL,
		 _kz_marshal_VOID__INT_INT,
		 G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_INT);
  moz_embed_signals[DOM_KEY_DOWN] =
    g_signal_new("dom_key_down",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_key_down),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_KEY_PRESS] =
    g_signal_new("dom_key_press",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_key_press),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_KEY_UP] =
    g_signal_new("dom_key_up",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_key_up),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_MOUSE_DOWN] =
    g_signal_new("dom_mouse_down",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_mouse_down),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_MOUSE_UP] =
    g_signal_new("dom_mouse_up",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_mouse_up),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_MOUSE_CLICK] =
    g_signal_new("dom_mouse_click",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_mouse_click),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_MOUSE_DBL_CLICK] =
    g_signal_new("dom_mouse_dbl_click",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_mouse_dbl_click),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_MOUSE_OVER] =
    g_signal_new("dom_mouse_over",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_mouse_over),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_MOUSE_OUT] =
    g_signal_new("dom_mouse_out",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_mouse_out),
		 NULL, NULL,
		 _kz_marshal_BOOLEAN__POINTER,
		 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[SECURITY_CHANGE] =
    g_signal_new("security_change",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, security_change),
		 NULL, NULL,
		 _kz_marshal_VOID__POINTER_INT,
		 G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_UINT);
  moz_embed_signals[STATUS_CHANGE] =
    g_signal_new("status_change",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(GtkMozEmbedClass, status_change),
		 NULL, NULL,
		 _kz_marshal_VOID__POINTER_INT_POINTER,
		 G_TYPE_NONE, 3,
		 G_TYPE_POINTER, G_TYPE_INT, G_TYPE_POINTER);
  moz_embed_signals[DOM_ACTIVATE] =
    g_signal_new("dom_activate",
                 G_TYPE_FROM_CLASS (klass),
                 G_SIGNAL_RUN_LAST,
                 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_activate),
		 NULL, NULL,
                 _kz_marshal_BOOLEAN__POINTER,
                 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_FOCUS_IN] =
    g_signal_new("dom_focus_in",
                 G_TYPE_FROM_CLASS (klass),
                 G_SIGNAL_RUN_LAST,
                 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_focus_in),
		 NULL, NULL,
                 _kz_marshal_BOOLEAN__POINTER,
                 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
  moz_embed_signals[DOM_FOCUS_OUT] =
    g_signal_new("dom_focus_out",
                 G_TYPE_FROM_CLASS (klass),
                 G_SIGNAL_RUN_LAST,
                 G_STRUCT_OFFSET(GtkMozEmbedClass, dom_focus_out),
		 NULL, NULL,
                 _kz_marshal_BOOLEAN__POINTER,
                 G_TYPE_BOOLEAN, 1, G_TYPE_POINTER);
}

static void
gtk_moz_embed_init(GtkMozEmbed *embed)
{
  EmbedPrivate *priv = new EmbedPrivate();
  embed->data = priv;
  gtk_widget_set_name(GTK_WIDGET(embed), "gtkmozembed");

  GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(embed), GTK_NO_WINDOW);
}

GtkWidget *
gtk_moz_embed_new(void)
{
  return GTK_WIDGET(g_object_new(gtk_moz_embed_get_type(), NULL));
}

// GtkObject methods

static void
gtk_moz_embed_destroy(GtkObject *object)
{
  GtkMozEmbed  *embed;
  EmbedPrivate *embedPrivate;
  
  g_return_if_fail(object != NULL);
  g_return_if_fail(GTK_IS_MOZ_EMBED(object));

  embed = GTK_MOZ_EMBED(object);
  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate) {

    // Destroy the widget only if it's been Init()ed.
    if(embedPrivate->mMozWindowWidget != 0) {
      embedPrivate->Destroy();
    }

    delete embedPrivate;
    embed->data = NULL;
  }
}

// GtkWidget methods

static void
gtk_moz_embed_realize(GtkWidget *widget)
{
  GtkMozEmbed    *embed;
  EmbedPrivate   *embedPrivate;
  GdkWindowAttr   attributes;
  gint            attributes_mask;
  
  g_return_if_fail(widget != NULL);
  g_return_if_fail(GTK_IS_MOZ_EMBED(widget));

  embed = GTK_MOZ_EMBED(widget);
  embedPrivate = (EmbedPrivate *)embed->data;

  GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK;

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
				   &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, embed);

  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);

  // initialize the window
  nsresult rv;
  rv = embedPrivate->Init(embed);
  g_return_if_fail(NS_SUCCEEDED(rv));
  
  PRBool alreadyRealized = PR_FALSE;
  rv = embedPrivate->Realize(&alreadyRealized);
  g_return_if_fail(NS_SUCCEEDED(rv));

  // if we're already realized we don't need to hook up to anything below
  if (alreadyRealized)
    return;

  if (embedPrivate->mURI.Length())
    embedPrivate->LoadCurrentURI();

  // connect to the focus out event for the child
  GtkWidget *child_widget = GTK_BIN(widget)->child;
  g_signal_connect_object(G_OBJECT(child_widget),
			  "focus_out_event",
			  G_CALLBACK(handle_child_focus_out),
			  embed,
			  G_CONNECT_AFTER);
  g_signal_connect_object(G_OBJECT(child_widget),
			  "focus_in_event",
			  G_CALLBACK(handle_child_focus_in),
			  embed,
			  G_CONNECT_AFTER);
}

static void
gtk_moz_embed_unrealize(GtkWidget *widget)
{
  GtkMozEmbed  *embed;
  EmbedPrivate *embedPrivate;
  
  g_return_if_fail(widget != NULL);
  g_return_if_fail(GTK_IS_MOZ_EMBED(widget));

  embed = GTK_MOZ_EMBED(widget);
  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate) {
    embedPrivate->Unrealize();
  }

  if (GTK_WIDGET_CLASS(embed_parent_class)->unrealize)
    (* GTK_WIDGET_CLASS(embed_parent_class)->unrealize)(widget);
}

static void
gtk_moz_embed_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
  GtkMozEmbed  *embed;
  EmbedPrivate *embedPrivate;
  
  g_return_if_fail(widget != NULL);
  g_return_if_fail(GTK_IS_MOZ_EMBED(widget));

  embed = GTK_MOZ_EMBED(widget);
  embedPrivate = (EmbedPrivate *)embed->data;
  
  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED(widget))
  {
#if 0
    PRInt32 width, height;

    embedPrivate->GetSize (&width, &height);
#endif
    
    gdk_window_move_resize(widget->window,
			   allocation->x, allocation->y,
			   allocation->width, allocation->height);
#if 0
    if ((allocation->width != width) || (allocation->height != height))
#endif
    embedPrivate->Resize(allocation->width, allocation->height);
  }
}

static void
gtk_moz_embed_map(GtkWidget *widget)
{
  GtkMozEmbed  *embed;
  EmbedPrivate *embedPrivate;
  
  g_return_if_fail(widget != NULL);
  g_return_if_fail(GTK_IS_MOZ_EMBED(widget));

  embed = GTK_MOZ_EMBED(widget);
  embedPrivate = (EmbedPrivate *)embed->data;

  GTK_WIDGET_SET_FLAGS(widget, GTK_MAPPED);

  embedPrivate->Show();

  gdk_window_show(widget->window);
  
}

static void
gtk_moz_embed_unmap(GtkWidget *widget)
{
  GtkMozEmbed  *embed;
  EmbedPrivate *embedPrivate;
  
  g_return_if_fail(widget != NULL);
  g_return_if_fail(GTK_IS_MOZ_EMBED(widget));

  embed = GTK_MOZ_EMBED(widget);
  embedPrivate = (EmbedPrivate *)embed->data;

  GTK_WIDGET_UNSET_FLAGS(widget, GTK_MAPPED);

  gdk_window_hide(widget->window);

  embedPrivate->Hide();
}

#ifdef MOZ_ACCESSIBILITY_ATK
static AtkObject*
gtk_moz_embed_get_accessible (GtkWidget *widget)
{
  g_return_val_if_fail(widget != NULL, NULL);
  g_return_val_if_fail(GTK_IS_MOZ_EMBED(widget), NULL);

  GtkMozEmbed  *embed = GTK_MOZ_EMBED(widget);;
  EmbedPrivate *embedPrivate = (EmbedPrivate *)embed->data;
  return static_cast<AtkObject *>(embedPrivate->GetAtkObjectForCurrentDocument());
}
#endif /* MOZ_ACCESSIBILITY_ATK */

static gint
handle_child_focus_in(GtkWidget     *aWidget,
		      GdkEventFocus *aGdkFocusEvent,
		      GtkMozEmbed   *aEmbed)
{
  EmbedPrivate *embedPrivate;

  embedPrivate = (EmbedPrivate *)aEmbed->data;

  embedPrivate->ChildFocusIn();

  return FALSE;
}

static gint
handle_child_focus_out(GtkWidget     *aWidget,
		       GdkEventFocus *aGdkFocusEvent,
		       GtkMozEmbed   *aEmbed)
{
  EmbedPrivate *embedPrivate;

  embedPrivate = (EmbedPrivate *)aEmbed->data;

  embedPrivate->ChildFocusOut();
 
  return FALSE;
}

// Widget methods

void
gtk_moz_embed_push_startup(void)
{
  EmbedPrivate::PushStartup();
}

void
gtk_moz_embed_pop_startup(void)
{
  EmbedPrivate::PopStartup();
}

void
gtk_moz_embed_set_comp_path(const char *aPath)
{
  EmbedPrivate::SetCompPath(aPath);
}

void
gtk_moz_embed_set_profile_path(const char *aDir, const char *aName)
{
  EmbedPrivate::SetProfilePath(aDir, aName);
}

void
gtk_moz_embed_set_directory_service_provider(nsIDirectoryServiceProvider *appFileLocProvider) {
  EmbedPrivate::SetDirectoryServiceProvider(appFileLocProvider);
}

void
gtk_moz_embed_load_url(GtkMozEmbed *embed, const char *url)
{
  EmbedPrivate *embedPrivate;
  
  g_return_if_fail(embed != NULL);
  g_return_if_fail(GTK_IS_MOZ_EMBED(embed));

  embedPrivate = (EmbedPrivate *)embed->data;

  embedPrivate->SetURI(url);

  // If the widget is realized, load the URI.  If it isn't then we
  // will load it later.
  if (GTK_WIDGET_REALIZED(embed))
    embedPrivate->LoadCurrentURI();
}

void
gtk_moz_embed_stop_load(GtkMozEmbed *embed)
{
  EmbedPrivate *embedPrivate;
  
  g_return_if_fail(embed != NULL);
  g_return_if_fail(GTK_IS_MOZ_EMBED(embed));

  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate->mNavigation)
    embedPrivate->mNavigation->Stop(nsIWebNavigation::STOP_ALL);
}

gboolean
gtk_moz_embed_can_go_back(GtkMozEmbed *embed)
{
  PRBool retval = PR_FALSE;
  EmbedPrivate *embedPrivate;

  g_return_val_if_fail ((embed != NULL), FALSE);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), FALSE);

  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate->mNavigation)
    embedPrivate->mNavigation->GetCanGoBack(&retval);
  return retval;
}

gboolean
gtk_moz_embed_can_go_forward(GtkMozEmbed *embed)
{
  PRBool retval = PR_FALSE;
  EmbedPrivate *embedPrivate;

  g_return_val_if_fail ((embed != NULL), FALSE);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), FALSE);

  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate->mNavigation)
    embedPrivate->mNavigation->GetCanGoForward(&retval);
  return retval;
}

void
gtk_moz_embed_go_back(GtkMozEmbed *embed)
{
  EmbedPrivate *embedPrivate;

  g_return_if_fail (embed != NULL);
  g_return_if_fail (GTK_IS_MOZ_EMBED(embed));

  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate->mNavigation)
    embedPrivate->mNavigation->GoBack();
}

void
gtk_moz_embed_go_forward(GtkMozEmbed *embed)
{
  EmbedPrivate *embedPrivate;

  g_return_if_fail (embed != NULL);
  g_return_if_fail (GTK_IS_MOZ_EMBED(embed));

  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate->mNavigation)
    embedPrivate->mNavigation->GoForward();
}

char *
gtk_moz_embed_get_link_message(GtkMozEmbed *embed)
{
  char *retval = nsnull;
  EmbedPrivate *embedPrivate;
  nsEmbedCString embedString;

  g_return_val_if_fail ((embed != NULL), (char *)NULL);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), (char *)NULL);

  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate->mWindow) {
    NS_UTF16ToCString(embedPrivate->mWindow->mLinkMessage,
		      NS_CSTRING_ENCODING_UTF8, embedString);
    
    retval = strdup(embedString.get());
  }

  return retval;
}

char *
gtk_moz_embed_get_js_status(GtkMozEmbed *embed)
{
  char *retval = nsnull;
  EmbedPrivate *embedPrivate;
  nsEmbedCString embedString;

  g_return_val_if_fail ((embed != NULL), (char *)NULL);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), (char *)NULL);

  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate->mWindow) {
    NS_UTF16ToCString(embedPrivate->mWindow->mJSStatus,
		      NS_CSTRING_ENCODING_UTF8, embedString);
    retval = strdup(embedString.get());
  }

  return retval;
}

char *
gtk_moz_embed_get_title(GtkMozEmbed *embed)
{
  char *retval = nsnull;
  EmbedPrivate *embedPrivate;
  nsEmbedCString embedString;

  g_return_val_if_fail ((embed != NULL), (char *)NULL);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), (char *)NULL);

  embedPrivate = (EmbedPrivate *)embed->data;

  if (embedPrivate->mWindow) {
    NS_UTF16ToCString(embedPrivate->mWindow->mTitle,
		      NS_CSTRING_ENCODING_UTF8, embedString);
    retval = strdup(embedString.get());
  }

  return retval;
}

char *
gtk_moz_embed_get_location(GtkMozEmbed *embed)
{
  char *retval = nsnull;
  EmbedPrivate *embedPrivate;
  nsEmbedCString embedString;

  g_return_val_if_fail ((embed != NULL), (char *)NULL);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), (char *)NULL);

  embedPrivate = (EmbedPrivate *)embed->data;
  
  if (embedPrivate->mURI.Length()) {
    NS_UTF16ToCString(embedPrivate->mURI,
		      NS_CSTRING_ENCODING_UTF8, embedString);
    retval = strdup(embedString.get());
  }

  return retval;
}

void
gtk_moz_embed_reload(GtkMozEmbed *embed, gint32 flags)
{
  EmbedPrivate *embedPrivate;

  g_return_if_fail (embed != NULL);
  g_return_if_fail (GTK_IS_MOZ_EMBED(embed));

  embedPrivate = (EmbedPrivate *)embed->data;

  PRUint32 reloadFlags = 0;
  
  // map the external API to the internal web navigation API.
  switch (flags) {
  case GTK_MOZ_EMBED_FLAG_RELOADNORMAL:
    reloadFlags = 0;
    break;
  case GTK_MOZ_EMBED_FLAG_RELOADBYPASSCACHE:
    reloadFlags = nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE;
    break;
  case GTK_MOZ_EMBED_FLAG_RELOADBYPASSPROXY:
    reloadFlags = nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY;
    break;
  case GTK_MOZ_EMBED_FLAG_RELOADBYPASSPROXYANDCACHE:
    reloadFlags = (nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY |
		   nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE);
    break;
  case GTK_MOZ_EMBED_FLAG_RELOADCHARSETCHANGE:
    reloadFlags = nsIWebNavigation::LOAD_FLAGS_CHARSET_CHANGE;
    break;
  default:
    reloadFlags = 0;
    break;
  }

  embedPrivate->Reload(reloadFlags);
}

void
gtk_moz_embed_set_chrome_mask(GtkMozEmbed *embed, guint32 flags)
{
  EmbedPrivate *embedPrivate;

  g_return_if_fail (embed != NULL);
  g_return_if_fail (GTK_IS_MOZ_EMBED(embed));

  embedPrivate = (EmbedPrivate *)embed->data;

  embedPrivate->SetChromeMask(flags);
}

guint32
gtk_moz_embed_get_chrome_mask(GtkMozEmbed *embed)
{
  EmbedPrivate *embedPrivate;

  g_return_val_if_fail ((embed != NULL), 0);
  g_return_val_if_fail (GTK_IS_MOZ_EMBED(embed), 0);

  embedPrivate = (EmbedPrivate *)embed->data;

  return embedPrivate->mChromeMask;
}

void
gtk_moz_embed_get_nsIWebBrowser  (GtkMozEmbed *embed, nsIWebBrowser **retval)
{
  EmbedPrivate *embedPrivate;

  g_return_if_fail (embed != NULL);
  g_return_if_fail (GTK_IS_MOZ_EMBED(embed));

  embedPrivate = (EmbedPrivate *)embed->data;
  
  if (embedPrivate->mWindow)
    embedPrivate->mWindow->GetWebBrowser(retval);
}

// class and instance initialization

GType
gtk_moz_embed_single_get_type(void);

static void
gtk_moz_embed_single_class_init(GtkMozEmbedSingleClass *klass);

static void
gtk_moz_embed_single_init(GtkMozEmbedSingle *embed);

GtkMozEmbedSingle *
gtk_moz_embed_single_new(void);

enum {
  NEW_WINDOW_ORPHAN,
  SINGLE_LAST_SIGNAL
};

guint moz_embed_single_signals[SINGLE_LAST_SIGNAL] = { 0 };

// GtkObject + class-related functions

GType
gtk_moz_embed_single_get_type(void)
{
  static GType moz_embed_single_type = 0;
  if (!moz_embed_single_type)
  {
    static const GTypeInfo moz_embed_single_info =
    {
      sizeof(GtkMozEmbedSingleClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) gtk_moz_embed_single_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (GtkMozEmbedSingle),
      0,
      (GInstanceInitFunc) gtk_moz_embed_single_init,
    };
    moz_embed_single_type = g_type_register_static (G_TYPE_OBJECT, "GtkMozEmbedSingle",
						    &moz_embed_single_info, (GTypeFlags)0);
  }

  return moz_embed_single_type;
}

static void
gtk_moz_embed_single_class_init(GtkMozEmbedSingleClass *klass)
{
  GObjectClass     *object_class;

  object_class = (GObjectClass *)klass;

  // set up our signals

  moz_embed_single_signals[NEW_WINDOW_ORPHAN] =
    g_signal_new("new_window_orphan",
		 G_TYPE_FROM_CLASS (klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMozEmbedSingleClass, new_window_orphan),
		 NULL, NULL,
		 _kz_marshal_VOID__POINTER_INT,
		 G_TYPE_NONE, 2, G_TYPE_POINTER, G_TYPE_UINT);
}

static void
gtk_moz_embed_single_init(GtkMozEmbedSingle *embed)
{
  // this is a placeholder for later in case we need to stash data at
  // a later data and maintain backwards compatibility.
  embed->data = nsnull;
}

GtkMozEmbedSingle *
gtk_moz_embed_single_new(void)
{
  return (GtkMozEmbedSingle *)g_object_new(gtk_moz_embed_single_get_type(), NULL);
}

GtkMozEmbedSingle *
gtk_moz_embed_single_get(void)
{
  static GtkMozEmbedSingle *singleton_object = nsnull;
  if (!singleton_object)
  {
    singleton_object = gtk_moz_embed_single_new();
  }

  return singleton_object;
}

// our callback from the window creator service
void
gtk_moz_embed_single_create_window(GtkMozEmbed **aNewEmbed,
				   guint         aChromeFlags)
{
  GtkMozEmbedSingle *single = gtk_moz_embed_single_get();

  *aNewEmbed = nsnull;

  if (!single)
    return;

  g_signal_emit(G_OBJECT(single),
		moz_embed_single_signals[NEW_WINDOW_ORPHAN], 0,
		aNewEmbed, aChromeFlags);

}
