#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <glob.h>
#include <pwd.h>
#include <utime.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <netinet/in.h>

#include "sftp.h"

#ifndef LINK_AS_LIBRARY
static int sftpscrwidth();
#endif
#define SFTPSCRWIDTH (sftpscrwidth() - 1)
#define skipspaces(s) {while (s && *s && isspace((int)*s)) s++;}

extern volatile int cancelled;
extern u_int32_t version;
extern int verbose;

extern int data_fd;
extern int pipe_to_par;
extern int pipe_from_par;


int parent_ipc(int pipe_r, int pipe_w, const char* name, int size) {

	char numbuf[128];
	char* namebuf;

	/* write the following data to the parent:
	 *
	 * size\r\n
	 * name\r\n
	 * 
	 * If we send a directory listing, we set size == -1
	 * If there is an error, we set size == -2 and send the error in
	 * name
	 */
	
	sprintf(numbuf, "%d\r\n", size);
	write(pipe_w, numbuf, strlen(numbuf));

	namebuf = (char*) malloc(strlen(name) + 3);
	strcpy(namebuf, name);
	strcat(namebuf, "\r\n");
	write(pipe_w, namebuf, strlen(namebuf));
	free(namebuf);

	/* wait for the parent to respond */
	read(pipe_r, numbuf, sizeof(numbuf));

	close(pipe_r);
	close(pipe_w);
	return 0;
}



int
next_token(char *str, char **arg) {
	char *s, *orig, *startq;
	int escaped = 0, quoted = 0;

	orig = s = str;
	startq = NULL;
	while (1) {
		if (*s == 0) {
			*arg = str;
			return (s - orig);
		}
		if (escaped) {
			memmove(str + 1, str, s - str - 1);
			str++;
			escaped = 0;
		}
		else if (quoted) {
			if (*s == '"') {
				memmove(startq + 2, startq + 1, s - startq - 1);
				memmove(str + 2, str, startq - str);
				str += 2;
				quoted = 0;
			}
		}
		else {
			if (*s == '"') {
				quoted = 1;
				startq = s;
			}
			else if (*s == '\\')
				escaped = 1;
			else if (isspace((int)*s))
				break;
		}
		s++;
	}
	*s++ = 0;
	*arg = str;
	return (s - orig);
}

int
tokenize(char *str, char **args, int maxargs) {
	int i;

	for (i = 0; str && *str && i < maxargs - 1; i++) {
		skipspaces(str);
		if (*str == 0)
			break;
		str += next_token(str, &args[i]);
	}
	args[i] = NULL;
	return i;
}

int
find_match(char *str, map *action) {
	int i, guess = -1, badguess = 0;
	for (i=0; action[i].str; i++) {
		if (strcmp(action[i].str, str) == 0)
			return i;
		if (strncmp(action[i].str, str, strlen(str)) == 0) {
			if (guess < 0 && !badguess)
				guess = i;
			else {
				badguess = 1;
				guess = -1;
			}
		}
	}
	return guess;
}

int
do_action(char *str, map *action) {
	int index, nargs;
	char *args[MAXARGS];

	if (str[strlen(str)-1] == '\n')
		str[strlen(str)-1] = 0;
	skipspaces(str);
	if (*str == 0)
		return 0;
	if (*str == '!') {
		system(str+1);
		return 0;
        }
	nargs = tokenize(str, args, MAXARGS);
	if (nargs == 0)
		return 0;
	index = find_match(args[0], action);
	if (index >= 0)
		return action[index].func(args, nargs);
	else
		printf ("%s: invalid command\n", args[0]);
	return -1;
}


int
do_send1file(int sock, char *file) {
	struct stat statbuf;
	u_int32_t size, mtime;
	u_int16_t mode;
	char modestr[9];
	int n, fd;
	char buf[BUFSIZE];
	char *base =0;
	message m;
	char *str;
	u_int8_t c = 1;
	u_int32_t skipped = 0;
	
	if (data_fd == -1) {
		if (stat(file, &statbuf) || !S_ISREG(statbuf.st_mode)) {
			str = "File doesn't exist or is not a regular file";
			send_message(sock, _data_message(c, ERROR, str, strlen(str)+1));
			return -1;
		}
		fd = open(file, O_RDONLY);
		if (fd < 0) {
			str = "File could not be opened";
			send_message(sock, _data_message(c, ERROR, str, strlen(str)+1));
			return -1;
		}
	

		mode = statbuf.st_mode & (S_IRWXU|S_IRWXG|S_IRWXO);
		modetostr(mode, modestr);
		size = htonl(statbuf.st_size);
		mtime = htonl(statbuf.st_mtime);
	} else {
		fd = data_fd;
		base = findbasename(file);
		size = 0;
		mtime = 0;
		strcpy(modestr, "rw-r----");
		modestr[8] = '-';
	}

	send_message(sock, _message(SENDFILE, &c, 1));
	send_message(sock, _data_message(c, FILENAME, base, strlen(base)+1));
	n = recv_message(sock, &m);

	if (n < 0)
		return -1;

	
	if (!(m.command == FILEOK && m.len == 1 && *(char *)m.data != 0)) {
		xfree(m.data);
		n = recv_message(sock, &m);
		/* The message is of type DATA and m.data contains the 
		 * error message */
		/* We inform the other process through the pipe pipe_to_par
		 * */

		write(pipe_to_par, m.data, m.len);
		xfree(m.data);
		return -1;
	} else {
		write(pipe_to_par, "OK", 2);
	}

	xfree(m.data);
	send_message(sock, _data_message(c, FILESIZE, &size, 4));
	send_message(sock, _data_message(c, FILEMODE, modestr, 9));
	if (version >= SFTP_VERSION(0,7,0))
		send_message(sock, _data_message(c, FILETIME, &mtime, 4));

	if (skipped > 0) {
		lseek(fd, skipped, SEEK_SET);
		skipped = htonl(skipped);
		send_message(sock, _data_message(c, SKIPBYTES, &skipped, 4));
	}
	while ((n = read(fd, buf, BUFSIZE)) > 0) {
		if (cancelled)
			break;
		send_message(sock, _data_message(c, DATA, buf, n));
	}
	send_message(sock, _data_message(c, ENDDATA, NULL, 0));
	close (fd);
	return 0;
}


int
do_sendfile(int sock, char *file) {
	int ret, i;
	glob_t globbuf;

	if (*file == 0) {
		send_message(sock, _message(NOFILEMATCH, NULL, 0));
		return 0;
	}
	ret = glob(file, 0, NULL, &globbuf);
	if (ret != 0)
		return -1;
	for (i = 0; i < globbuf.gl_pathc; i++) {
		if (cancelled)
			break;
		printf("sending %s\n", globbuf.gl_pathv[i]);
		do_send1file(sock, globbuf.gl_pathv[i]);
	}
	globfree(&globbuf);
	return i;
}

int
do_recvfile(int sock, u_int8_t channel, int resume) {
	int n;
	u_int32_t size = 0, mtime = 0;
	u_int16_t mode;
	char *file = NULL;
	int fd = -1, total = 0;
	u_int8_t status;
	message m;
	u_int32_t startsize = 0;
	int sentcancel = 0;

	while (1) {
		n = recv_message(sock, &m);
		if (n < 0)
			return -1;
		switch (m.command) {
			case SENDFILE:
				channel = *(u_int8_t*)m.data;
			case FILENAME:
				file = strdup(m.data);
				fd = data_fd;
				status = (fd >= 0);
				if (status > 0 && resume) {
					struct stat statbuf;
					u_int32_t nsize;
					fstat(fd, &statbuf);
					startsize = statbuf.st_size;
/*					lseek(fd, startsize, SEEK_SET);*/
					nsize = htonl(startsize);
					send_message(sock,
						     _data_message(channel,
								   SKIPBYTES,
								   &nsize, 4));
				}
				else {
					send_message(sock,
						     _data_message(channel,
								   FILEOK,
								   &status, 1));
				}
				if (status == 0) {
					free(file);
					return -1;
				}
				break;
			case FILESIZE:
				if (fd < 0 || m.len != 4)
					break;
				size = ntohl(*(u_int32_t *)m.data);

				parent_ipc(pipe_from_par, pipe_to_par,
							file, size);
				if (verbose)
					printf("%d bytes\n%s: ", size, file);
				break;
			case FILEMODE:
				if (fd < 0 || m.len != 9)
					break;
				mode = strtomode((char *)m.data);
				break;
			case FILETIME:
				if (fd < 0 || m.len != 4)
					break;
				mtime = ntohl(*(u_int32_t *)m.data);
				break;
			case DATA:
				if (fd < 0)
					break;
				write(fd, m.data, m.len);
				total += m.len;
				if ((size - startsize) == 0)
					break;
				break;
			case ENDDATA:
				close(fd);
				if (mtime != 0) {
					struct utimbuf ubuf;
					ubuf.actime = time(NULL);
					ubuf.modtime = mtime;
					/* utime(file, &ubuf); */
				}
				free(file);
				return total - startsize;
		}
		xfree(m.data);
		if (cancelled && !sentcancel &&
		    version >= SFTP_VERSION(0, 8, 0))
		{
			send_message(sock,
				     _data_message(channel, CANCEL, NULL, 0));
			sentcancel = 1;
		}
	}
}

struct timeval
timediff(struct timeval *tv1, struct timeval *tv2) {
	struct timeval tv;
	tv.tv_sec = tv1->tv_sec - tv2->tv_sec;
	tv.tv_usec = tv1->tv_usec - tv2->tv_usec;
	if (tv.tv_usec < 0) {
		tv.tv_usec+=1000000;
		tv.tv_sec--;
	}
	return tv;
}

#ifndef LINK_AS_LIBRARY
static int
sftpscrwidth() {
#ifdef TIOCGWINSZ
	static int width = -1;
	int ret;
	struct winsize win;

	if (width != -1)
		return width;
	ret = ioctl(1, TIOCGWINSZ, (char *)&win);
	if (ret != 0)
		width = 80;
	else
		width = win.ws_col;
	return width;
#else
	return 80;
#endif
}

#endif  /* not link as library */

