/* Simple file transfer protocol over a tcp socket */

#include "config.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <limits.h>
#include <fcntl.h>
#include <errno.h>
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#else
#include <sys/signal.h>
#endif
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#ifdef HAVE_PTY_H
#include <pty.h>
#endif
#ifdef HAVE_LIBREADLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include "sftp.h"

#define check_connected() \
	if (connected == 0) {\
		printf("%s: Not connected\n", args[0]);\
		return -1;\
	}


extern char *shell;
char* sftp_out_buffer;
	
	
int do_help (char **args, int nargs);
int do_exec(char **args, int nargs);
int do_delete(char **args, int nargs);
int do_open (char **args, int nargs);
int do_close (char **args, int nargs);
int do_get (char **args, int nargs);
int do_put (char **args, int nargs);
int do_cd (char **args, int nargs);
int do_pwd (char **args, int nargs);
int do_lcd (char **args, int nargs);
int do_dir (char **args, int nargs);
int do_quit (char **args, int nargs);
int do_version (char **args, int nargs);
int do_binary (char **args, int nargs);

map action[] = {
	{"?",		do_help},
	{"help",	do_help},
	{"open",	do_open},
	{"close",	do_close},
	{"get",		do_get},
	{"mget",	do_get},
	{"put",		do_put},
	{"mput",	do_put},
	{"cd",		do_cd},
	{"pwd",		do_pwd},
	{"ls",		do_exec},
	{"dir",		do_dir},
	{"mdir",	do_dir},
	{"rm",		do_exec},
	{"delete",	do_delete},
	{"mdelete",	do_delete},
	{"lcd",		do_lcd},
	{"exec",	do_exec},
	{"quit",	do_quit},
	{"exit",	do_quit},
	{"bye",		do_quit},
	{"binary",	do_binary},
	{"ascii",	do_binary},
	{"image",	do_binary},
	{"version",	do_version},
	{NULL,		NULL},
};

int connected = 0;
int donecmdline = 0;
volatile int cancelled = 0;
pid_t child;
int sock;
u_int32_t version;
char *ftpscript = NULL;
int noreadline = 0;
int verbose = 0;

extern int data_fd;
extern int pipe_from_par, pipe_to_par;
extern int parent_ipc(int, int, const char*, int);


#define RSHERRORSTR "[-P remotepath] [-l user] [-n]"
#define SSHERRORSTR RSHERRORSTR " [-s sshargs]"
#define COMMANDLINESTR "[-e] [-f file] [-v]"

void
usage() {
	printf("Usage:\n");
	printf("    sftp %s [host]\n", SSHERRORSTR " " COMMANDLINESTR);
	printf("    rsftp %s [host]\n", RSHERRORSTR " " COMMANDLINESTR);
	exit(-1);
}

static char *
safe_strcat(char *dest, int dlen, const char *src) {
	if (strlen(dest) + strlen(src) + 1 > dlen) {
		fprintf(stderr, "String too long\n");
		exit(2);
	}
	return strcat(dest, src);
}

static char *
safe_strcpy(char *dest, int dlen, const char *src) {
	if (strlen(src) + 1 > dlen) {
		fprintf(stderr, "String too long\n");
		exit(2);
	}
	return strcpy(dest, src);
}

int
do_help (char **args, int nargs) {
	int i;
	printf("Supported commands are:\n");
	for (i = 0; action[i].str != NULL; i++) {
		printf(action[i].str);
		if (i % 8 == 7)
			putchar('\n');
		else
			putchar('\t');
	}
	if (i % 8 != 0)
		putchar('\n');
	return 0;
}

int
do_command(char **args, int nargs, FILE *output) {
	char buf[BUFSIZE];
	int i, len = 0;
	message m;

	check_connected();
	for (i = 0; i < nargs; i++) {
		if (i==0 && strcmp(args[i], "exec") == 0)
			continue;
		safe_strcpy(buf + len, BUFSIZE - len, args[i]);
		len += strlen(args[i]);
		buf[len++] = 0;
	}
	send_message(sock, _message(EXEC, buf, len));
	while (1) {
		int n;
		u_int8_t c = 1;
		n = recv_message(sock, &m);
		if (cancelled && version >= SFTP_VERSION(0, 8, 0)) {
			send_message(sock, _data_message(c, CANCEL, NULL, 0));
			cancelled = 0;
		}
		if (n < 0)
			return -1;
		switch (m.command) {
			case ERROR:
				puts((char *)m.data);
				xfree(m.data);
				return -1;
			case SUCCESS:
				return 0;
			case STREAM:
				break;
			case DATA:
				fwrite(m.data, 1, m.len, output);
				xfree(m.data);
				break;
			case ENDDATA:
				return 0;
		}
	}
}

int
do_exec(char **args, int nargs) {
	return do_command(args, nargs, stdout);
}

int
do_delete(char **args, int nargs) {
	args[0] = "rm";
	return do_command(args, nargs, stdout);
}

#define openerror(msg) \
	do { \
		if (msg != NULL) \
			printf("%s\n", (char *)msg); \
		if (!donecmdline) \
			usage(); \
		else { \
			if (strcmp(shell, "ssh") == 0) \
				printf("open %s host\n", SSHERRORSTR); \
			else \
				printf("open %s host\n", RSHERRORSTR); \
			return -1; \
		} \
	} while(0)

int
do_open (char **args, int nargs) {
	int s[2];
#ifdef HAVE_FORKPTY
	int master;
	int didecho = 0;
	struct termios tio, otio;
#endif
	char *remotepath = NULL;
	char *remoteuser = NULL;
	char *remotehost = NULL;
	char *remoteport = NULL;
	char *sargs = NULL;
	int compress = 0;
	int i;
	int nonetrc = 0;
	int openflag = 0;

	if (connected)
		do_close(NULL, 0);
			
	for (i = 1; i < nargs; i++) {
		if (strcmp(args[i], "-e") != 0 &&
		    strcmp(args[i], "-f") != 0 &&
		    strcmp(args[i], "-v") != 0)
			openflag = 1;
		if (strcmp(args[i], "-P") == 0) {
			if (i == nargs - 1)
				openerror(NULL);
			remotepath = args[++i];
		}
		else if (strcmp(args[i], "-l") == 0) {
			if (i == nargs - 1)
				openerror(NULL);
			remoteuser = args[++i];
		}
		else if (strcmp(args[i], "-p") == 0) {
			if (i == nargs - 1)
				openerror(NULL);
			if (strcmp(shell, "ssh") != 0)
				openerror("-p can only be used with ssh");
			remoteport = args[++i];
			if (atoi(remoteport) <= 0)
				openerror("-p must specify a numeric port");
		}
		else if (strcmp(args[i], "-C") == 0) {
			if (strcmp(shell, "ssh") != 0)
				openerror("-C can only be used with ssh");
			compress = 1;
		}
		else if (strcmp(args[i], "-s") == 0) {
			if (strcmp(shell, "ssh") != 0)
				openerror("-s can only be used with ssh");
			sargs = args[++i];
		}
		else if (strcmp(args[i], "-f") == 0) {
			if (donecmdline)
				openerror("-f is not an open flag");
			if (i == nargs - 1)
				openerror(NULL);
			ftpscript = args[++i];
		}
		else if (strcmp(args[i], "-e") == 0) {
			if (donecmdline)
				openerror("-e is not an open flag");
			noreadline = 1;
		}
		else if (strcmp(args[i], "-v") == 0) {
			if (donecmdline)
				openerror("-v is not an open flag");
			verbose = 1;
		}
		else if (strcmp(args[i], "-n") == 0) {
			nonetrc = 1;
		}
		else if (args[i][0] == '-') {
			openerror(NULL);
		}
		else {
			remotehost = args[i];
			break;
		}
	}
	if (remotehost == NULL) {
		if (openflag)
			openerror(NULL);
		else
			return 0;
	}
	if (socketpair(PF_UNIX, SOCK_STREAM, 0, s) < 0) {
		perror("socketpair");
		return -1;
	}
#ifdef HAVE_FORKPTY
	if ((child = forkpty(&master, NULL, NULL, NULL)))
#else
	if ((child = fork()))
#endif
	{
		int n = 0;
		char buffer[BUFSIZE];

		close(s[1]);
		sock = s[0];
#ifdef HAVE_FORKPTY
		if (tcgetattr(0, &tio) == 0 && (tio.c_lflag & ECHO)) {
			didecho = 1;
			otio = tio;
			tio.c_lflag &= ~(ECHO | ECHOE | ECHOK | ECHONL);
			tcsetattr(0, TCSANOW, &tio);
		}
		while (1) {
			fd_set fds;
			FD_ZERO(&fds);
			FD_SET(0, &fds);
			FD_SET(sock, &fds);
			FD_SET(master, &fds);
			n = select (master + 1, &fds, NULL, NULL, NULL);
			if (n < 0) {
				perror("select");
				if (didecho)
					tcsetattr(0, TCSANOW, &otio);
				return -1;
			}
			if (FD_ISSET(0, &fds)) {
				n = read(0, buffer, BUFSIZE - 1);
				if (n < 0) {
					perror("read");
					if (didecho)
						tcsetattr(0, TCSANOW, &otio);
					return -1;
				}
				write(master, buffer, n);
			}
			if (FD_ISSET(master, &fds)) {
				n = read(master, buffer, BUFSIZE - 1);
				if (n < 0) {
					perror("read");
					if (didecho)
						tcsetattr(0, TCSANOW, &otio);
					return -1;
				}
				write(1, buffer, n);
			}
			if (FD_ISSET(sock, &fds)) {
				n = read(sock, buffer, BUFSIZE - 1);
				if (n < 0) {
					perror("read");
					if (didecho)
						tcsetattr(0, TCSANOW, &otio);
					return -1;
				}
				buffer[n] = 0;
				if (n >= 4 && strstr(buffer, "sftp") != NULL) {
					connected = 1;
					if (didecho)
						tcsetattr(0, TCSANOW, &otio);
					break;
				}
				write(1, buffer, n);
			}
		}
#else
		while (1) {
			n = read(sock, buffer, BUFSIZE);
			if (n < 0) {
				perror("read");
				return -1;
			}
			if (n == 0)
				return -1;
			if (n >= 4 && strstr(buffer, "sftp") != NULL) {
				connected = 1;
				break;
			}
			fputs (buffer, stdout);
			printf("\n");
		}
#endif
		if (connected) {
			if (n >= 5 && strstr(buffer, "sftp") != NULL) {
				message m;
				u_int32_t myversion = htonl(SFTP_MY_VERSION);
				send_message(sock, _message(VERSION,
							    &myversion, 4));
				recv_message(sock, &m);
				if (m.command != VERSION || m.len != 4) {
					version = SFTP_VERSION_OLD;
					return 0;
				}
				version = ntohl(*(u_int32_t*)m.data);
				xfree(m.data);
			}
			else
				version = SFTP_VERSION_OLD;
			return 0;
		}
	}
	else {
		char command[1024];
		int plen;
		char *newargs[20];
		int nnewargs = 0;
		char user[20];

#ifdef HAVE_FORKPTY
		close(master);
		setpgid(0, 0);
#endif

		close(s[0]);
		dup2(s[1], 0);
		dup2(s[1], 1);
/*		dup2(s[1], 2);*/

		command[0] = '\0';
		if (remotepath != NULL)
			safe_strcat(command, sizeof(command), remotepath);
		else
			safe_strcat(command, sizeof(command), SFTPSERV_PATH);
		plen = strlen(command);
		if (plen < 8 || strcmp("sftpserv", command + plen - 8) != 0) {
			if (command[plen - 1] != '/')
				safe_strcat(command, sizeof(command), "/");
			safe_strcat(command, sizeof(command), "sftpserv");
		}

		if (remoteuser == NULL && !nonetrc) {
			if (netrc_user(remotehost, user, sizeof(user)) == 0)
				remoteuser = user;
		}
		newargs[nnewargs++] = shell;
		newargs[nnewargs++] = "-e";
		newargs[nnewargs++] = "none";
		if (compress)
			newargs[nnewargs++] = "-C";
		if (remoteuser != NULL) {
			newargs[nnewargs++] = "-l";
			newargs[nnewargs++] = remoteuser;
		}
		if (remoteport != 0) {
			newargs[nnewargs++] = "-p";
			newargs[nnewargs++] = remoteport;
		}
		if (sargs != NULL) {
			int n;
			char* s = strdup(sargs);
			n = tokenize(s, &newargs[nnewargs],
				     sizeof(newargs) - nnewargs - 3);
			nnewargs += n;
		}
		newargs[nnewargs++] = remotehost;
		newargs[nnewargs++] = command;
		newargs[nnewargs++] = NULL;
		execvp(shell, newargs);
	}
	return 0;
}

int
do_close (char **args, int nargs) {
	check_connected();
	send_message(sock, _message(CLOSE, NULL, 0));
	connected = 0;
	return 0;
}

int
do_get (char **args, int nargs) {
	message m;
	int i;
	int resume = 0;
	int argstart = 1;

	sftp_out_buffer =0;
	
	check_connected();
	if (nargs > 1 && strcmp(args[1], "-C") == 0) {
		if (version < SFTP_VERSION(0, 7, 0)) {
			sftp_out_buffer = "get: -C not supported by server\n";
			return -1;
		}
		else {
			sftp_out_buffer = "resume code is _very_ experimental!\n";
			resume = 1;
			argstart++;
		}
	}
	if (strcasecmp(args[0], "LIST") == 0 
			|| strcasecmp(args[0], "NLST") == 0) {
		FILE* file = fdopen(data_fd, "w");
		char* arrlist[] = { "ls",  "-l", "|", "grep", "-v", "^total" };
		char* arrnlst[] = { "ls" };
		int ret;

		parent_ipc(pipe_from_par, pipe_to_par,
				"file list", -1);
		if (strcasecmp(args[0], "LIST") == 0) {
			ret = do_command(arrlist, 6, file);
		} else {
			ret = do_command(arrnlst, 1, file);
		}
		fclose(file);
		return ret;
	}
	if (argstart == nargs) {
		sftp_out_buffer = "get: no files specified\n";
		return -1;
	}
	for (i = argstart; i < nargs; i++) {
		int gotten = 0;
		send_message(sock,
			     _message(REQUEST, args[i], strlen(args[i])+1));
		while (!cancelled) {
			recv_message(sock, &m);
			if (m.command == NOFILEMATCH) {
				if (gotten == 0)
					sftp_out_buffer = (char*)m.data;
					parent_ipc(pipe_from_par, pipe_to_par,
							sftp_out_buffer, -2);
				break;
			}
			else if (m.command != SENDFILE)
				continue;
			if (m.data) {
				struct timeval tv1, tv2;
				int size;
				u_int8_t channel = *(u_int8_t *)m.data;

				gettimeofday(&tv1, NULL);
				size = do_recvfile(sock, channel, resume);
				gettimeofday(&tv2, NULL);
				if (size < 0) {
					sftp_out_buffer = "Failed to get file";
					return -1;
				}
				else if (cancelled)
					sftp_out_buffer = "Transfer aborted";
				xfree(m.data);
				gotten++;
			}
		}
	}
	return 0;
}

int
do_put (char **args, int nargs) {
	int i, n, total = 0;

	check_connected();
	if (strcmp(args[0], "STOR") == 0) {
		n = do_send1file(sock, args[1]);
		return 0;
	}
	for (i = 1; i < nargs; i++) {
		n = do_sendfile(sock, args[i]);
		if (n < 0) {
			printf("error sending %s\n", args[i]);
			return -1;
		}
		total += n;
	}
	if (total == 0)
		printf("No files specified\n");
	else if (cancelled)
		printf("Transfer aborted\n");
	else
		printf("%d file%s sent\n", total, (total == 1) ? "" : "s");
	return 0;
}

int
do_cd (char **args, int nargs) {
	message m;

	check_connected();
	if (nargs > 1)
		send_message(sock, _message(CHDIR, args[1], strlen(args[1])+1));
	else
		send_message(sock, _message(CHDIR, NULL, 0));
	recv_message(sock, &m);
	switch (m.command) {
		case ERROR:
			sftp_out_buffer = (char*)m.data;
			return -1;
		case SUCCESS:
			sftp_out_buffer = 0;
			return 0;
	}
	return 0;
}

int
do_pwd (char **args, int nargs) {
	message m;

	check_connected();
	send_message(sock, _message(GETDIR, NULL, 0));
	recv_message(sock, &m);
	switch (m.command) {
		case ERROR:
			sftp_out_buffer = (char*)m.data;
			return -1;
		case TELLDIR:
			sftp_out_buffer = (char*)m.data;
			return 0;
	}
	return 0;
}

int
do_lcd (char **args, int nargs) {
	int ret;
	if (nargs > 1) {
		char *dir = args[1];

		if (dir[0] == '~') {
			dir = tildeexpand(dir);
			if (dir == NULL) {
				printf ("No such directory\n");
				return -1;
			}
		}
		ret = chdir(dir);
	}
	else
		ret = chdir(getenv("HOME"));
	if (ret < 0) {
                if (errno == EACCES)
                        printf("Permission denied\n");
                else
                        printf("No such directory\n");
	}
	return ret;
}

int
do_dir (char **args, int nargs) {
	int i;
	if (nargs == MAXARGS) {
		printf("too many arguments\n");
		return (-1);
	}
	for (i = nargs; i >= 1; i--)
		args[nargs] = args[nargs - 1];
	args[0] = "ls";
	args[1] = "-l";
	return do_exec(args, nargs + 1);
}

int
do_quit (char **args, int nargs) {
	if (connected)
		do_close(NULL, 0);
	exit(0);
}

int
do_version (char **args, int nargs) {
	printf("sftp version %d.%d.%d\n", SFTP_MAJOR(SFTP_MY_VERSION),
	       SFTP_MINOR(SFTP_MY_VERSION), SFTP_SUB(SFTP_MY_VERSION));
	if (connected) {
		printf("connected to sftpserv version ");
		if (version == SFTP_VERSION_OLD)
			printf("(old)\n");
		else
			printf("%d.%d.%d\n", SFTP_MAJOR(version),
			       SFTP_MINOR(version), SFTP_SUB(version));
	}
	printf("author: Brian Wellington <bwelling@xbill.org>\n");
	return 0;
}

int
do_binary (char **args, int nargs) {
	printf("binary mode is always used\n");
	return 0;
}

void
reap() {
	int status;
	int pid;
	while ((pid = wait3(&status, WNOHANG, NULL)) > 0) {
		if (pid != child)
			continue;
		if (WIFSIGNALED(status))
			printf ("child died with signal %d\n",
					WTERMSIG(status));
		if (connected == 1)
			printf ("Connection closed\n");
		connected = 0;
	}
}

void
handlebreak() {
	if (cancelled)
		do_quit(NULL, 0);
	cancelled = 1;
	if (verbose)
		printf("cancelling...\n");
}


#ifndef LINK_AS_LIBRARY

int
main(int argc, char **argv) {
	char *base;
	int ret;
	struct sigaction act, act2;
	sigset_t set;
	FILE *fp;

	base = findbasename(argv[0]);
	if (base[0] == 'r')
		shell = "rsh";
	else
		shell = "ssh";

	sigemptyset(&set);
	memset(&act, 0, sizeof(struct sigaction));
	act.sa_handler = reap;
	act.sa_mask = set;
	act.sa_flags = SA_RESTART;
	sigaction (SIGCHLD, &act, NULL);
	
	memset(&act2, 0, sizeof(struct sigaction));
	act2.sa_handler = handlebreak;
	act2.sa_mask = set;
	act2.sa_flags = SA_RESTART;
	sigaction (SIGINT, &act2, NULL);

	if (argc > 1)
		do_open(argv, argc);

	donecmdline = 1;

#ifdef HAVE_LIBREADLINE
	if (ftpscript == NULL && !noreadline && isatty(0)) {
		/* This while statement will not return. */
		while (1) {
			char *buffer;
			buffer = readline( "sftp> ");
			if (buffer == NULL)
				do_quit(NULL, 0);
			if (*buffer != 0) {
				add_history(buffer);
				cancelled = 0;
				ret = do_action(buffer, action );
			}
			free(buffer);
		}
	}
#endif
	if (ftpscript != NULL) {
		fp = fopen (ftpscript, "r");
		if (!fp) {
			printf("script file not found\n");
			usage();
		}
	}
	else
		fp = stdin;

	while (1) {
		char buffer[LINESIZE];
		if (fp == stdin)
			printf ("sftp> ");
		if (fgets (buffer, LINESIZE, fp) == NULL) {
			if (fp != stdin)
				fclose(fp);
			do_quit(NULL, 0);
		}
		cancelled = 0;
		ret = do_action(buffer, action);
	}
}




#endif  /* LINK_AS_LIBRARY */

int do_cancel() {
	u_int8_t c = 1;

	send_message(sock, _data_message(c, CANCEL, NULL, 0));

	return 0;
}

