/* 
 * Copyright (C) 1999-2001 Joachim Wieland <joe@mcknight.de>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111, USA.
 */

#include <ctype.h>
#include <stdlib.h>
#include <limits.h>
#include <pwd.h>
#include <syslog.h>
#include "jftpgw.h"

char* valid_keys[] = { "accessfile", "serverport", "defaultmode",
			"runasuser", "defaultforward", "logfile", "forward",
			"throughput", "cmdlogfile", "connectionlogdir",
			"atstartchangerootdir", "changerootdir", "limit",
			"ssh-passwd", "passallauth", "ssh-options",
			"transparent-forward", "pidfile", "welcomeline",
			"listen", "debuglevel", "allowreservedports",
			"allowforeignaddress", "passiveportrange",
			"activeportrange", "logstyle", "transparent-proxy",
			(char*) 0 };

struct optionstruct {
	char* key;
	char* value;
	struct optionstruct* next;
} *optionsbase, *optionscur;

struct forwardstruct {
	char* fwlogin;
	char* fwpass;
	char* destlogin;
	char* destpass;
	struct forwardstruct* next;
} *forwardbase, *forwardcur;

struct throughputstruct {
	char* user;
	struct ip_t src;
	struct ip_t dest;
	char* srcname;
	char* destname;
	float rate;
	struct throughputstruct* next;
} *throughputbase, *throughputcur;

struct sshpasswdstruct {
	char* user;
	char* hostname;
	struct ip_t ip;
	char* password;
	struct sshpasswdstruct* next;
} *sshpasswdbase, *sshpasswdcur;

struct sshoptionsstruct {
	char* user;
	char* hostname;
	struct ip_t ip;
	char* options;
	struct sshoptionsstruct* next;
} *sshoptionsbase, *sshoptionscur;

struct portrangestruct *actvrangebase, *pasvrangebase,
			*actvrangecur, *pasvrangecur;
unsigned int actvportsamount;
unsigned int pasvportsamount;

struct lchar {
	char* entry;
	struct lchar* next;
};


extern struct configstruct config;
extern struct uidstruct runasuser;

/* from jftpgw.c */
extern char* conffilename;

struct loginstruct lsforward;
struct limitstruct *limitbase, *limitcur;

int is_valid_key(const char*);
const char* getaccessfile();
int add_forward(const char*);
int add_throughput(const char*);
int add_limit(const char*);
int add_sshpasswd(const char*);
int add_sshoptions(const char*);
int add_rangespec(const char*);
int match_username(const char*, const char*);
int match_addrs(unsigned long int, const char*, unsigned long int,
                unsigned long int, const char*);
int save_runasuser_uid(void);
void free_llistst(struct lchar*);

int conf_delete_registered = 0;



int set_conffilename(const char* arg) {
	free(conffilename);
	conffilename = strdup(arg);
	enough_mem(conffilename);
	return 0;
}

int read_config(const char* filename) {
	int ret;
	FILE* conffile = open_file(filename);
	if (!conffile) {
		return -1;
	}
	log(9, "opened file");
	/* free the space at the end of the program */
	if (!conf_delete_registered) {
		atexit(conf_delete_config);
		conf_delete_registered = 1;
	}

	optionsbase = optionscur = (struct optionstruct*) 0;
	forwardbase = forwardcur = (struct forwardstruct*) 0;
	throughputbase = throughputcur = (struct throughputstruct*) 0;
	limitbase = limitcur = (struct limitstruct*) 0;
	sshpasswdbase = sshpasswdcur = (struct sshpasswdstruct*) 0;
	sshoptionsbase = sshoptionscur = (struct sshoptionsstruct*) 0;
	pasvrangebase = pasvrangecur = (struct portrangestruct*) 0;
	actvrangebase = actvrangecur = (struct portrangestruct*) 0;
	ret = config_read_options(conffile);
	if (ret) {
		return -1;
	}
	fclose(conffile);
	ret = read_access(getaccessfile());
	return ret;
}

FILE* open_file(const char* fname) {
	FILE* conf;

	conf = fopen(fname, "r");
	if (!conf) {
		perror("Couldn't open the configuration file");
		log(1, "Couldn't open the configuration file %s: %s", fname,
		strerror(errno));

		return 0;
	}
	return conf;
}

FILE* open_logfile(const char* fname) {
	FILE* logf;
	int err;

	logf = fopen(fname, "a");
	err = errno;
	if (!logf) {
		perror("Couldn't open a log file");
		log(1, "Couldn't open the log file %s: %s", fname,
				strerror(errno));
	}
	errno = err;
	return logf;
}

char* config_read_line(FILE* file) {
	/* read 255 bytes at first */
	const int startsize = 5;
	/* therafter increase the buffer again by size bytes */
	const int increase = startsize;
	int times;
	unsigned int size = startsize;
	static char* line;
	char* ret;
	if (line) {
		free(line);
		line = 0;
	}
	size = startsize;
	times = 0;
	do {
		if (!line) {
			line = (char*) malloc(size + 1);
			enough_mem(line);
			line[0] = '\0';
		} else {
			line = (char*) realloc(line, size + 1);
			enough_mem(line);
		}
		/* append the newly read characters to the old ones */
		ret = fgets(line + times*increase, increase + 1, file);
		if (!ret) {
			if (feof(file)) {
				break;
			}
		 	perror("Error reading a line from the "
			       "configuration file");
			log(1, "Error reading a line from the "
			    "configuration file: %s", strerror(errno));
		}
		size += increase;
		times++;
	} while (ret && ret[strlen(ret)-1] != '\n');

	if (feof(file)) {
		free(line);
		line = 0;
		return (char*) 0;
	}
	return line;
}


int config_read_options(FILE* file) {

	int ret;
	char *entry;
#define NORMALTYPE   0
#define FORWARD      1
#define THROUGHPUT   2
#define LIMIT        3
#define SSHPASSWD    4
#define SSHOPTIONS   5
#define RANGESPEC    6

	int entrytype =0;

	while ((entry = config_read_line(file))) {
		if (entry[0] == '\n'     /* empty line  */
		    ||
		    entry[0] == '#') {   /* ignore line */
			continue;
		}
		if (!is_valid_key(entry)) {
			fprintf(stderr, "Invalid key %s in configuration "
					"file\n", entry);
			log(4, "Invalid key %s in configuration file", entry);
			continue;
		}
		entrytype = NORMALTYPE;
		if (strncmp(entry, "forward", strlen("forward")) == 0) {
			entrytype = FORWARD;
		}
		if (strncmp(entry, "throughput", strlen("throughput")) == 0) {
			entrytype = THROUGHPUT;
		}
		if (strncmp(entry, "limit", strlen("limit")) == 0) {
			entrytype = LIMIT;
		}
		if (strncmp(entry, "ssh-passwd", strlen("ssh-passwd")) == 0) {
			entrytype = SSHPASSWD;
		}
		if (strncmp(entry, "ssh-options", strlen("ssh-options")) == 0) {
			entrytype = SSHOPTIONS;
		}
		if (strncmp(entry, "passiveportrange",
			    strlen("passiveportrange")) == 0
		   || strncmp(entry, "activeportrange",
			      strlen("activeportrange")) == 0) {
			entrytype = RANGESPEC;
		}
		switch(entrytype) {
			case FORWARD:
				ret = add_forward(entry);
				break;
			case THROUGHPUT:
				ret = add_throughput(entry);
				break;
			case LIMIT:
				ret = add_limit(entry);
				break;
			case SSHPASSWD:
				ret = add_sshpasswd(entry);
				break;
			case SSHOPTIONS:
				ret = add_sshoptions(entry);
				break;
			case RANGESPEC:
				ret = add_rangespec(entry);
				break;
			default:
				ret = add_to_options(entry);
		}
		if (ret) {
			log(4, "Invalid line %s specified", entry);
			return -1;
		}
	}
	/* cache the UIDs */
	ret = save_runasuser_uid();
	return ret;
}


int add_to_options(char* entry) {
	struct optionstruct *ptr;
	const char* space = strpbrk(entry, WHITESPACES);
	const char* p = 0;
	int add = 0;
	if (!*space) {
		return -1;
	} else {
		int valuesize;
		ptr = (struct optionstruct*)
				malloc(sizeof(struct optionstruct));
		enough_mem(ptr);
		if (optionscur) {
			optionscur->next = ptr;
		}
		if (!optionsbase) {
			optionsbase = ptr;
		}
		optionscur = ptr;

		optionscur->key = (char*) malloc(space - entry + 1);
		enough_mem(optionscur->key);
		strncpy(optionscur->key, entry, space - entry);
		optionscur->key[space - entry] = '\0';

		p = space;
		while (*p && isascii((int)*p) && isspace((int)*p)) {
			p++;
		}

		/*
		 * prepend a space character in front of the command
		 * specifications
		 */

		if (strstr(optionscur->key, "specs")) {
			add = 1;
		}
		valuesize = strlen(p) + 1 + add;
		optionscur->value = (char*) malloc(valuesize);
		if (add) {
			optionscur->value[0] = ' ';
		}
		strncpy(optionscur->value + add, p, valuesize - add);

		while (optionscur->value[strlen(optionscur->value)-1] == '\n') {
			optionscur->value[strlen(optionscur->value)-1] = '\0';
		}

		optionscur->next = (struct optionstruct*) 0;
	}

	return 0;
}


const char* get_option(const char* key) {

	struct optionstruct* osc;

	osc = optionsbase;

	while (osc && strcmp(osc->key, key)) {
		osc = osc->next;
	}

	if (!osc) {
		return 0;
	} else {
		return osc->value;
	}
}

int is_valid_key(const char* keyval) {
	int i;
	i = 0;
	while (valid_keys[i]) {
		if (strlen(keyval) >= strlen(valid_keys[i])) {
			if (strncmp(keyval, valid_keys[i],
						strlen(valid_keys[i])) == 0) {
				return 1;
			}
		}
		i++;
	}
	return 0;
}

/* quotstrtok parses *s and returns a malloc'ed pointer to the tokens. It
 * also respects quotation marks:
 *
 * bla "foo bar" 	bar     bla
 *
 * returns bla, "foo bar", bar and bla
 */

char* quotstrtok(const char* s, const char* delim, int *past_offset) {

	int i = *past_offset;
	const char* q = 0;
	char *ret =0, *r =0;
	const char* ubound, *lbound;

	/*while(s[i] && isspace((int)s[i])) {*/
	/* move forth if s[i] is one of the delimiters */
	while(s[i] && strchr(delim, (int)s[i])) {
		i++;
	}
	if (s[i] == 0)
		return 0;
	if (s[i] == '"') {
		q = strchr(&s[i+1], '"');
		q = strpbrk(++q, delim);
	} else {
		q = strpbrk(&s[i], delim);
	}
	if (!q || !*q) {
		if (strlen(&s[i]) == 0) {
			return 0;
		} else {
			q = s + i + strlen(&s[i]);
		}
	}
	/* chop quotation marks on both sides */
	if (s[i] == '"' && *(q-1) == '"') {
		lbound = &s[i+1];
		ubound = q-1;
	} else {
		lbound = &s[i];
		ubound = q;
	}
	ret = (char*) malloc(ubound - lbound + 1);
	enough_mem(ret);
	strncpy(ret, lbound, ubound - lbound);
	ret[ubound - lbound] = '\0';
	*past_offset = i + (q - &s[i]);  /* q - &s[i] is ubound - lbound,
					    regardless of quotation marks */

	if (strlen(ret)) {
		r = ret + strlen(ret) - 1;
		while (iscntrl((int)*r) && strlen(ret)) {
			*r = '\0';
		r--;
		}
	}

	return ret;
}


int add_forward(const char* entry) {
	struct forwardstruct *ptr;
	char* fwlogin =0, *fwpass =0, *destlogin =0, *destpass =0;
	int offset = strlen("forward");
	fwlogin = quotstrtok(entry, WHITESPACES, &offset);
	if (!fwlogin) {
		return -1;
	}
	destlogin = quotstrtok(entry, WHITESPACES"\n", &offset);
	if (!destlogin) {
		free(fwlogin);
		return -1;
	}
	fwpass = quotstrtok(entry, WHITESPACES"\n", &offset);
	if (!fwpass) {
		fwpass = 0;
		destpass = 0;
	} else {
		destpass = quotstrtok(entry, WHITESPACES"\n", &offset);
	}

	/* the line is invalid if fwpass is "*" and no destpass is 
	   specified */
	if (fwpass && strcmp(fwpass, "*") == 0 && destpass == 0) {
		if (fwlogin) { free(fwlogin); }
		if (fwpass) { free(fwpass); }
		if (destlogin) { free(destlogin); }
		if (destpass) { free(destpass); }
		return -1;
	}

	/* write to the structure */

	ptr = (struct forwardstruct*)
			malloc(sizeof(struct forwardstruct));
	enough_mem(ptr);
	if (forwardcur) {
		forwardcur->next = ptr;
	}
	if (!forwardbase) {
		forwardbase = ptr;
	}
	forwardcur = ptr;

	forwardcur->fwlogin = fwlogin;
	forwardcur->destlogin = destlogin;
	forwardcur->fwpass = fwpass;
	forwardcur->destpass = destpass;
	forwardcur->next = 0;

	return 0;

}

int get_forward(const char* user, struct loginstruct* logininfo) {

	if (!forwardbase) {
		return 0;
	}

	forwardcur = forwardbase;
	do {
		if (strcmp(user, forwardcur->fwlogin) == 0
		/* a "*" matches all usernames */
		 || strcmp("*", forwardcur->fwlogin) == 0) {
			logininfo->fwlogin =
				strdup(forwardcur->fwlogin);
			enough_mem(logininfo->fwlogin);

			logininfo->login =
				strdup(forwardcur->destlogin);
			enough_mem(logininfo->login);

			if (forwardcur->fwpass) {
				logininfo->fwpass = strdup(forwardcur->fwpass);
				enough_mem(logininfo->fwpass);
			} else {
				logininfo->fwpass = 0;
			}

			if (forwardcur->destpass) {
				logininfo->pass =
					strdup(forwardcur->destpass);
				enough_mem(logininfo->pass);
			} else {
				logininfo->pass = 0;
			}

			return 1;
		}
	} while ((forwardcur = forwardcur->next));

	memset(logininfo, 0, sizeof(struct loginstruct));
	return 0;
}

int save_runasuser_uid(void) {
	const char* option;
	struct passwd* pws;

	option = get_option("runasuser");
	if (option) {
		runasuser.username = strdup(option);
		enough_mem(runasuser.username);
		runasuser.username = trim(runasuser.username);
		pws = getpwnam(runasuser.username);
		if (!pws) {
			log(3, "getpwnam(2) could not look up user %s: %s",
				runasuser.username, strerror(errno));
			perror("Could not look up the user name");
			return -1;
		}
		runasuser.uid = pws->pw_uid;
	}
	return 0;
}

int checkfwpass(struct clientinfo* clntinfo) {
	int fw = lsforward.login != 0;
	if (fw == 0) {
		/* no forward for this user. Allow further processing */
		return 1;
	}
 	if (!lsforward.fwpass) {
		/* no password */
		return 1;
	} 
	if (strcmp(lsforward.fwpass, "*") == 0
		|| cryptcmp(lsforward.fwpass, clntinfo->pass) == 0) {
		/* allow any pw but use destpw on dest */
		memset(clntinfo->pass, 0, strlen(clntinfo->pass));
		clntinfo->pass = realloc(clntinfo->pass,
					strlen(lsforward.pass) + 1);
		enough_mem(clntinfo->pass);
		strncpy(clntinfo->pass, lsforward.pass,
				strlen(lsforward.pass) /* always true */);
		return 1;
	}
	return 0;
}



int add_throughput(const char* entry) {
	struct throughputstruct* ptr;
	int offset = strlen("throughput");
	char* user =0, *ipbuf =0, *srcname =0, *destname =0;
	char* endptr;
	struct ip_t src, dest;
	float rate;

	user = quotstrtok(entry, WHITESPACES, &offset);
	if (!user) {
		return -1;
	}
	ipbuf = quotstrtok(entry, WHITESPACES, &offset);
	if (!ipbuf) {
		free(user);
		return -1;
	}
	src = parse_ip(ipbuf);
	if (src.ip == -1 && src.netmask == -1) {
		srcname = ipbuf;
		ipbuf = 0;
	}
	/* can I free ipbuf ??? */
	if (ipbuf) { free(ipbuf); }
	ipbuf = quotstrtok(entry, WHITESPACES, &offset);
	if (!ipbuf) {
		free(user);
		return -1;
	}
	dest = parse_ip(ipbuf);
	if (dest.ip == -1 && dest.netmask == -1) {
		destname = ipbuf;
		ipbuf = 0;
	}
	/* can I free ipbuf??? */
	if (ipbuf) { free(ipbuf); ipbuf = 0; }
	endptr = user; /* this value has no meaning, it's just != 0 */
	errno = 0;
	rate = strtod(&entry[offset], &endptr);
	offset = 0;
	if (rate == 0 || errno /* || rate == HUGE_VAL  */
		|| (ipbuf = quotstrtok(endptr, WHITESPACES"\n", &offset))) {
		if (ipbuf) { free(ipbuf); }
		return -1;
	}

	/* values successfully read */

	ptr = (struct throughputstruct*)
		 malloc(sizeof(struct throughputstruct));
	enough_mem(ptr);
	if (throughputcur) {
		throughputcur->next = ptr;
	}
	if (!throughputbase) {
		throughputbase = ptr;
	}
	throughputcur = ptr;

	throughputcur->user = user;
	throughputcur->src = src;
	throughputcur->dest = dest;
	throughputcur->srcname = srcname;
	throughputcur->destname = destname;
	throughputcur->rate = rate;
	throughputcur->next = 0;

	return 0;
}

float get_rate(const char *fwuser, const char* user,
		unsigned long int src, const char* srcname,
		unsigned long int dest, const char* destname) {

	int matches;
	char* arruser;

	if (!throughputbase) {
		return -1;
	}

	if (user) {
		size_t arrusersize = strlen(user) + 2;
		/* user may be (char*) 0 if no forward is used */
		arruser = malloc(arrusersize);
		enough_mem(arruser);
		snprintf(arruser, arrusersize, ">%s", user);
	} else {
		arruser = (char*) 0;
	}

	throughputcur = throughputbase;
	do {
		matches = 1;
		matches &= match_username(fwuser, throughputcur->user)
			    || match_username(arruser, throughputcur->user);
		matches &= match_addrs(src, srcname,
		  throughputcur->src.ip, throughputcur->src.netmask,
		  throughputcur->srcname);

		matches &= match_addrs(dest, destname,
		  throughputcur->dest.ip, throughputcur->dest.netmask,
		  throughputcur->destname);

		if (matches) {
			break;
		}

	} while ((throughputcur = throughputcur->next));


	if (matches) {
		log(9, "Found a throughput rate.");
		return throughputcur->rate;
	}
	return -1;  /* no rate, transmission at highest possible speed */
}
 
int match_username(const char* is, const char* should) {

	if (!is) {
		return 0;
	}
	if (strlen(should) > 2 && 
		0 == strcmp(should + strlen(should) - 2, "@*")) {

		return !strncmp(is, should, strlen(should) - 1);
	}
	if (!strcmp(should, "*") || !strcmp(should, ">*")) {
		return 1;
	}
	return !strcmp(is, should);
}

int match_addrs(unsigned long int ip, const char* name,
		unsigned long int ipwnet, unsigned long int netmask,
		const char* namepat) {

	struct hostent *host;
	struct in_addr iaddr;
	int i;

	/* try without lookup */

	if (name && namepat && cmp_domains(name, namepat)) {
		return 1;
	}

	if ((ip & netmask) == (ipwnet & netmask)) {
		return 1;
	}

	if (name && namepat) {
		/* look up name and try to match the aliases against
		 * namepat and ipwnet/netmask */
		host = gethostbyname(name);
		if (!host) {
			log(5, "Could not look up name %s", name);
		} else {
			if (host->h_name && namepat) {
				if(cmp_domains(host->h_name, namepat)) {
					return 1;
				}
			}
			i = 0;
			while (host->h_aliases[i] && namepat) {
				if(cmp_domains(host->h_aliases[i], namepat)) {
					return -1;
				}
				i++;
			}
			i = 0;
			do {
				char* ipbuf=gethostentip(host->h_addr_list[i]);
				iaddr.s_addr = inet_addr(ipbuf);
				if ((iaddr.s_addr & netmask)
					 == (ipwnet & netmask)) {
					return 1;
				}
				i++;
			} while (host->h_addr_list[i]);
		}
	}

	/* look up IP and try to match all the host's against the allowed
	   ones */

	host = gethostbyaddr((char*) &ip, sizeof(ip), AF_INET);
	if (!host) {
		iaddr.s_addr = ip;
		log(5, "Could not look up IP %s", inet_ntoa(iaddr));
	} else {
		i = 0;
		do {
			char* ipbuf = gethostentip(host->h_addr_list[i]);
			iaddr.s_addr = inet_addr(ipbuf);
			if ((iaddr.s_addr & netmask) == (ipwnet & netmask)) {
				return 1;
			}
			i++;
		} while (host->h_addr_list[i]);

		if (host->h_name && namepat) {
			if(cmp_domains(host->h_name, namepat)) {
				return 1;
			}
		}
		i = 0;
		while (host->h_aliases[i] && namepat) {
			if(cmp_domains(host->h_aliases[i], namepat)) {
				return 1;
			}
			i++;
		}
	}

	/* didn't find anything */

	return 0;

}

struct lchar* split_line(const char *line, const char* pattern) {
	int offset = strlen(pattern);
	char* buf;
	struct lchar *llistbase =0, *llistcur =0, *ptr =0, *prev =0;

	while ((buf = quotstrtok(line, WHITESPACES"\n", &offset))) {
		ptr = (struct lchar*) malloc (sizeof(struct lchar));
		enough_mem(ptr);
		if (!llistbase) {
			llistbase = ptr;
		}
		if (llistcur) {
			llistcur->next = ptr;
			prev = llistcur;
		}
		llistcur = ptr;

		llistcur->entry = buf;
		llistcur->next = 0;
	}

	return llistbase;
}


int add_limit(const char* entry) {
	struct ip_t ip;
	struct limitstruct *limitptr;
	unsigned long int connamount;

	struct lchar* llistbase = split_line(entry, "limit");
	struct lchar* llistcur = llistbase;
	struct lchar* prev =0;

	/* set llistcur to the last element of the linked list pointed to by
	 * llistbase */

	while (llistcur->next) {
		prev = llistcur;
		llistcur = llistcur->next;
	}

	/* the last entry must be the number of connects allowed */

	connamount = strtol(llistcur->entry, NULL, 10);
	if (errno == ERANGE
			&& (connamount == LONG_MIN || connamount == LONG_MAX)) {
		log(3, "Error reading the amount of allowed connections in %s",
			entry);
		return -1;
	}
	free(llistcur->entry);
	prev->next = 0;
	free(llistcur);

	/* examine the other entries */

	llistcur = llistbase;
	do {
		limitptr = (struct limitstruct*)
				malloc(sizeof(struct limitstruct));
		if (!limitbase) {
			limitbase = limitptr;
		}
		if (limitcur) {
			limitcur->next = limitptr;
		}
		limitcur = limitptr;
		limitcur->connmax = connamount;
		limitcur->connected = 0;
		limitcur->connlist = (struct connliststruct*) 0;
		limitcur->next = (struct limitstruct*) 0;
		ip = parse_ip(llistcur->entry);
		if (ip.ip == -1 && ip.netmask == -1) {
			/* consider llistcur->entry as a hostname */
			limitcur->hostname = llistcur->entry;
		} else {
			limitcur->hostname = 0;
			free(llistcur->entry);
			llistcur->entry = 0;
		}
		limitcur->ip = ip;

		/* destroy the linked list structure */
		prev = llistcur;
		llistcur = llistcur->next;

		/* llistcur->entry is either already freed or the malloc()ed
		 * memory of entry is used for the limitstruct structure */
		free(prev);

	} while (llistcur);

	return 0;
}


int add_sshpasswd(const char* entry) {
	struct ip_t ip;
	struct sshpasswdstruct *sshpasswdptr;
	char* ssh_passwd = 0;
	char* ssh_user = 0;

	struct lchar* llistbase = split_line(entry, "ssh-passwd");
	struct lchar* llistcur = llistbase;
	struct lchar* prev =0;

	if (!llistbase) {
		/* only the keyword has been specified */
		return -1;
	}

	/* the first element of the linked list must be the user name */
	ssh_user = llistbase->entry;
	llistbase = llistbase->next;

	/* llistcur points to the former llistbase */
	free(llistcur);
	llistcur = llistbase;

	if ( ! llistbase) {
		/* only the keyword and one string have been specified */
		return -1;
	}

	if ( ! llistcur->next) {
		/* only the keyword and two strings have been specified */
		free(llistcur->entry);
		free(llistcur);
		return -1;
	}


	/* set llistcur to the last element of the linked list pointed to by
	 * llistbase */

	while (llistcur->next) {
		prev = llistcur;
		llistcur = llistcur->next;
	}

	/* the last entry must be the password */

	ssh_passwd = llistcur->entry;

	prev->next = 0;
	free(llistcur);

	/* examine the other entries */

	llistcur = llistbase;
	do {
		sshpasswdptr = (struct sshpasswdstruct*)
			malloc(sizeof(struct sshpasswdstruct));
		if (!sshpasswdbase) {
			sshpasswdbase = sshpasswdptr;
		}
		if (sshpasswdcur) {
			sshpasswdcur->next = sshpasswdptr;
		}
		sshpasswdcur = sshpasswdptr;
		sshpasswdcur->user = strdup(ssh_user);
		enough_mem(sshpasswdcur->user);
		sshpasswdcur->password = strdup(ssh_passwd);
		enough_mem(sshpasswdcur->password);
		sshpasswdcur->next = (struct sshpasswdstruct*) 0;
		ip = parse_ip(llistcur->entry);
		if (ip.ip == -1 && ip.netmask == -1) {
			/* consider llistcur->entry as a hostname */
			sshpasswdcur->hostname = llistcur->entry;
		} else {
			sshpasswdcur->hostname = 0;
			free(llistcur->entry);
			llistcur->entry = 0;
		}
		sshpasswdcur->ip = ip;

		/* destroy the linked list structure */
		prev = llistcur;
		llistcur = llistcur->next;

		free(prev);

	} while (llistcur);

	free(ssh_user);
	free(ssh_passwd);
	return 0;
}


int add_sshoptions(const char* entry) {
	struct sshoptionsstruct *sshoptionsptr;
	struct ip_t ip;
	char* ssh_options = 0;
	char* ssh_user = 0;

	struct lchar* llistbase = split_line(entry, "ssh-options");
	struct lchar* llistcur = llistbase;
	struct lchar* prev =0;

	if (!llistbase) {
		/* only the keyword has been specified */
		return -1;
	}

	/* the first element of the linked list must be the user name */
	ssh_user = llistbase->entry;
	llistbase = llistbase->next;

	/* llistcur points to the former llistbase */
	free(llistcur);
	llistcur = llistbase;

	if ( ! llistbase) {
		/* only the keyword and one string have been specified */
		return -1;
	}

	if ( ! llistcur->next) {
		/* only the keyword and two strings have been specified */
		free(llistcur->entry);
		free(llistcur);
		return -1;
	}


	/* set llistcur to the last element of the linked list pointed to by
	 * llistbase */

	while (llistcur->next) {
		prev = llistcur;
		llistcur = llistcur->next;
	}

	/* the last entry must be the option field */

	ssh_options = llistcur->entry;

	prev->next = 0;
	free(llistcur);

	/* examine the other entries */

	llistcur = llistbase;
	do {
		sshoptionsptr = (struct sshoptionsstruct*)
				malloc(sizeof(struct sshoptionsstruct));
		if (!sshoptionsbase) {
			sshoptionsbase = sshoptionsptr;
		}
		if (sshoptionscur) {
			sshoptionscur->next = sshoptionsptr;
		}
		sshoptionscur = sshoptionsptr;
		sshoptionscur->user = strdup(ssh_user);
		enough_mem(sshoptionscur->user);
		sshoptionscur->options = strdup(ssh_options);
		enough_mem(sshoptionscur->options);
		sshoptionscur->next = (struct sshoptionsstruct*) 0;
		ip = parse_ip(llistcur->entry);
		if (ip.ip == -1 && ip.netmask == -1) {
			/* consider llistcur->entry as a hostname */
			sshoptionscur->hostname = llistcur->entry;
		} else {
			sshoptionscur->hostname = 0;
			free(llistcur->entry);
			llistcur->entry = 0;
		}
		sshoptionscur->ip = ip;

		/* destroy the linked list structure */
		prev = llistcur;
		llistcur = llistcur->next;

		free(prev);

	} while (llistcur);

	free(ssh_user);
	free(ssh_options);
	return 0;
}


int add_rangespec(const char* entry) {
	int offset = 0;
	unsigned int *portsamount;
	struct portrangestruct** prsbase, **prscur, *prs;
	char* startportstr, *endportstr;
	long startport, endport;
	char* typestr = quotstrtok(entry, WHITESPACES"\n", &offset);
	struct lchar* llistbase = split_line(entry, typestr);
	struct lchar* llistcur = llistbase;
	struct lchar* prevtmp;

	if (strcasecmp(typestr, "passiveportrange") == 0) {
		prsbase = &pasvrangebase;
		prscur  = &pasvrangecur;
		portsamount = &pasvportsamount;
	} else {
		prsbase = &actvrangebase;
		prscur  = &actvrangecur;
		portsamount = &actvportsamount;
	}
	free(typestr);
	typestr = (char*) 0;

	while (llistcur) {
		offset = 0;
		startportstr = quotstrtok(llistcur->entry, ":", &offset);
		endportstr = quotstrtok(llistcur->entry, ":", &offset);
		if (!startportstr || !endportstr) {
			log(3, "invalid port range: %s", llistcur->entry);
			/* free */
			free_llistst(llistbase);
			return -1;
		}
		if (!*startportstr || !*endportstr) {
			log(3, "invalid port range: %s", llistcur->entry);
			/* free */
			free_llistst(llistbase);
			return -1;
		}
		startport = strtol(startportstr, NULL, 10);
		if (errno == ERANGE
				&& (startport == LONG_MIN || startport == LONG_MAX)) {
			log(3, "Error reading the starting port number in %s",
				startportstr);
			free_llistst(llistbase);
			free(startportstr);
			return -1;
		}
		free(startportstr);
		startportstr = (char*) 0;

		endport = strtol(endportstr, NULL, 10);
		if (errno == ERANGE
				&& (endport == LONG_MIN || endport == LONG_MAX)) {
			log(3, "Error reading the ending port number in %s",
				endportstr);
			free_llistst(llistbase);
			free(endportstr);
			return -1;
		}
		free(endportstr);
		endportstr = (char*) 0;

		/* startport really below or equal to endport ? */
		if (startport > endport) {
			log(4, "Port range %d:%d invalid (starting port number"
				" %d is _above_ endport %d)",
					startport, endport,
					startport, endport);
			free_llistst(llistbase);
			return -1;
		}

		/* both must be between (inclusive) 1024 and 65535 */
		if (startport > 65535   ||   startport < /* 1024 */ 0
		   || endport > 65535   ||   endport   < /* 1024 */ 0) {

			log(4, "Port range %d:%d invalid (at least one port "
				"number not in range 1024:65535)",
					startport, endport);
			free_llistst(llistbase);
			return -1;
		}

		/* I want only to allow clear ranges. This is better for the
		 * checking, for the selection of a port and the admin is
		 * forced to specify his ports in a clear manner  :-)
		 *
		 * That means that a portrange like:
		 *
		 * 1000:2000   1500:1600   is invalid, since 1500:1600 is
		 * already in 1000:2000
		 *
		 * Also invalid:
		 *
		 * 1000:2000   1800:2200   (200 portnumbers are the same)
		 * 1000:2000   2000:3000   (1 portnumber is the same)
		 * 1000:2000   500:1000    (1 portnumber is the same)
		 * 1000:2000   799:1200    (200 portnumbers are the same)
		 *
		 * criteria: startport and endport => let's say just pno
		 *
		 * If pno is between (or equal to) the range of another port
		 * specification, state that is invalid.
		 * ------
		 * What happens if we first specify
		 *
		 * 300:400
		 *
		 * and then
		 *
		 * 200:1000 ?
		 *
		 * We have to do the checking for _each_ new range to _all_
		 * ranges. Compare every range to every other range
		 *
		 */

		prs = *prsbase;
		while (prs) {
			if (
			    (startport >= prs->startport &&
			     startport <= prs->endport)
			    ||
			    (endport >= prs->startport &&
			     endport <= prs->endport)
			   ) {
				log(4, "Port range %d:%d invalid (parts "
					"covered by another range (%d:%d))",
						startport, endport,
						prs->startport, prs->endport);
				free_llistst(llistbase);
				return -1;
			}

			if (
			    (prs->startport >= startport &&
			     prs->startport <= endport)
			    ||
			    (prs->endport >= startport &&
			     prs->endport <= endport)
			    ) {
				log(4, "Port range %d:%d invalid (parts "
					"covered by another range (%d:%d))",
						prs->startport, prs->endport,
						startport, endport);
				free_llistst(llistbase);
				return -1;
			}
			prs = prs->next;
		}

		prs = (struct portrangestruct*)
				malloc(sizeof(struct portrangestruct));
		if (*prscur) {
			(*prscur)->next = prs;
		}

		*prscur = prs;

		if (!*prsbase) {
			/* the first one */
			*prsbase = *prscur;
			*portsamount = 0;
		}
		(*prscur)->startport = startport;
		(*prscur)->endport = endport;
		(*prscur)->next = (struct portrangestruct*) 0;

		/* startport: 7
		 * endport:  10
		 *
		 * We have    10 - 7 + 1    =    3 + 1    =    4      ports
		 *
		 * 7, 8, 9, 10
		 * */
		*portsamount += (endport - startport) + 1;

		/* iterate through and always chop off the first element */
		prevtmp = llistcur;
		llistcur = llistcur->next;
		llistbase = llistcur;
		free(prevtmp->entry);
		free(prevtmp);
		prevtmp = 0;
	}

	return 0;
}


int is_below_limit(unsigned long int iaddr, struct limitstruct** ls) {

	int ret;

	if (!limitbase) {
		/* no limit options specified */
		*ls = 0;
		return 1;
	}
	limitcur = limitbase;

	/* lets see if there is a rule for the source */

	do {
		if (limitcur->hostname == NULL) {
			/* compare IPs */
			ret = match_addrs(iaddr, (const char*) 0,
					limitcur->ip.ip, limitcur->ip.netmask,
					(const char*) 0);
		} else {
			/* compare hostnames */
			ret = match_addrs(iaddr, (const char*) 0,
					-1, -1,
					limitcur->hostname);
		}
		if (ret) {
			/* found a match */
			break;
		}
	} while ((limitcur = limitcur->next));

	if (!ret) {
		/* no match found, the client may come in */
		*ls = 0;
		return 1; 
	}

	/* okay, lets see if we fit in */

	if (limitcur->connected == limitcur->connmax) {
		/* ooops... */
		*ls = 0;
		return 0;
	}

	*ls = limitcur;
	return 1;
}


const char* get_sshpasswd(unsigned long int iaddr, const char* user) {

	int ret = 0;

	if (!sshpasswdbase) {
		/* no sshpasswd options specified */
		return (const char*) 0;
	}
	sshpasswdcur = sshpasswdbase;

	/* lets see if there is a rule for the source */

	do {
		if (strcmp(user, sshpasswdcur->user) == 0) {
			if (sshpasswdcur->hostname == NULL) {
				/* compare IPs */
				ret = match_addrs(iaddr, (const char*) 0,
						sshpasswdcur->ip.ip,
						sshpasswdcur->ip.netmask,
						(const char*) 0);
			} else {
				/* compare hostnames */
				ret = match_addrs(iaddr, (const char*) 0,
						-1, -1,
						sshpasswdcur->hostname);
			}
			if (ret) {
				/* found a match */
				break;
			}
		}
	} while ((sshpasswdcur = sshpasswdcur->next));

	if (!ret) {
		/* no match found */
		return (const char*) 0; 
	}

	return sshpasswdcur->password;
}


const char* get_sshoptions(unsigned long int iaddr, const char* user) {
	int ret =0;
/*	int offset;
	struct lchar* lstcur =0, *lstbase =0, *prev =0;
	char* ent;
*/
	if (!sshoptionsbase) {
		/* no sshoptions options specified */
		return (const char*) 0;
	}
	sshoptionscur = sshoptionsbase;

	/* lets see if there is a rule for the source */

	do {
		if (strcmp(user, sshoptionscur->user) == 0) {
			if (sshoptionscur->hostname == NULL) {
				/* compare IPs */
				ret = match_addrs(iaddr, (const char*) 0,
						sshoptionscur->ip.ip,
						sshoptionscur->ip.netmask,
						(const char*) 0);
			} else {
				/* compare hostnames */
				ret = match_addrs(iaddr, (const char*) 0,
						-1, -1,
						sshoptionscur->hostname);
			}
			if (ret) {
				/* found a match */
				break;
			}
		}
	} while ((sshoptionscur = sshoptionscur->next));

	if (!ret) {
		/* no match found */
		return (const char*) 0; 
	}

	/* parse sshoptionscur->options into tokens */
/*	offset = 0;
	while((ent = quotstrtok(sshoptionscur->options, WHITESPACES,
					&offset))) {
		lstcur = malloc(sizeof(struct lchar));
		(*optn)++;
		lstcur->entry = ent;
		lstcur->next = 0;
		if (prev) {
			prev->next = lstcur;
		}
		prev = lstcur;
		if ( ! lstbase) {
			lstbase = lstcur;
		}
	}

	options = (char**) malloc(sizeof(char*) * *optn);

	lstcur = lstbase;
	i =0;
	do {
		options[i] = lstcur->entry;
		i++;
	} while (lstcur->next);

	lstcur = lstbase;
	do {
		lstbase = lstcur->next;
		free(lstcur);
	} while (lstbase);
*/
	return sshoptionscur->options;
}



int register_pid(struct limitstruct* ls, pid_t pid) {

	struct connliststruct *cls, *tmp;

	/* well, let me in */

	/* maybe we don't have an entry for that source */
	if (!ls) {
		return 0;
	}

	cls = (struct connliststruct*) malloc(sizeof(struct connliststruct));
	enough_mem(cls);

	if (!ls->connlist) {
		ls->connlist = cls;
	} else {
		tmp = ls->connlist;
		while (tmp->next) {
			tmp = tmp->next;
		}
		tmp->next = cls;
	}
	cls->next = 0;
	cls->pid = pid;
	log(9, "Added pid %d", cls->pid);
	ls->connected++;

	return 0;
}

int unregister_pid(pid_t pid) {

	struct connliststruct* cls, *prev = 0;

	if (!limitbase) {
		/* no limit options specified */
		/* or we are in a fork()ed server
		 * NOTE: Only the server that listens on the port and
		 * fork()s at every connnection attempt may have a
		 * limitbase != 0  */
		return 0;
	}

	limitcur = limitbase;

	do {
		cls = limitcur->connlist;
		while (cls) {
			log(8, "checking %d against %d", cls->pid, pid);
			if (cls->pid == pid) {
				if (prev) {
					prev->next = cls->next;
				} else {
					limitcur->connlist = cls->next;
				}
				log(9, "removed a connlist structure entry");
				limitcur->connected--;
				free(cls);
				return 0;
			}
			prev = cls;
			cls = cls->next;
		} 
	} while ((limitcur = limitcur->next));

	/* pid was not found. Maybe we don't have a limit entry for that
	 * source */

	return 0;
}


int getservermode() {
	/* readonce is 0, if the function is executed the first time */
	static int readonce;
	/* servmode of 0 means asclient */
	static int servmode;
	const char* opt;
	if (readonce) {
		return servmode;
	}
	readonce = 1;
	opt = get_option("defaultmode");

	if (opt) {
		if (0 == strcasecmp(opt, "passive")) {
			servmode = PASSIVE;
			return servmode;
		}
		if (0 == strcasecmp(opt, "active")) {
			servmode = ACTIVE;
			return servmode;
		}
		if (0 == strcasecmp(opt, "asclient")) {
			servmode = ASCLIENT;
			return servmode;
		}
		if (0 == strcasecmp(opt, "secure-sftp")) {
			servmode = SECURE_SFTP;
			return servmode;
		}
	}

	/* an error occured */

	fprintf(stderr, "No defaultmode specified, using `asclient' as default value for `defaultmode'\n");
	log(4, "No defaultmode specified, using `asclient' as default value for `defaultmode'");
	servmode = ASCLIENT;     /* as client */
	return servmode;
}

unsigned int getserverport() {
	unsigned int port;
	const char* opt = get_option("serverport");
	if (!opt || !(port = atoi(opt))) {
		fprintf(stderr, "Invalid `serverport' port in the"
				"configuration file: %s\n", opt);
		log(4, "Invalid `serverport' port in the configuration "
			    "file: %s", opt);
		fprintf(stderr, "Assuming %d as default value for "
				"`serverport'\n", DEFAULTSERVERPORT);
                log(4, "Assuming %d as default value for `serverport'",
				DEFAULTSERVERPORT);
		port = DEFAULTSERVERPORT;
	}
	return port;
}


const char* getlisten() {
	const char* address;
	const char* opt = get_option("listen");
	if (!opt) {
		fprintf(stderr, "Invalid `bindaddress' address in the configuration "
				"file: %s\n", opt);
		log(4, "Invalid `bindaddress' address in the configuration "
				"file: %s", opt);
		fprintf(stderr, "Assuming %s as default value for `bindaddress'\n",
				DEFAULTBINDADDRESS);
		log(4, "Assuming %s as default value for `bindaddress'",
				DEFAULTBINDADDRESS);
		address = DEFAULTBINDADDRESS;
	} else {
		address = opt;
	}
	return address;
}

const char* getaccessfile() {
	const char* opt = get_option("accessfile");
	if (!opt) {
		char *b;
		size_t bsize;
		fprintf(stderr, "No accessfile option specified. Assuming "
				"%s as default\n", DEFAULTACCESSFILE);
		log(4, "No accessfile option specified. Assuming "
		    "%s as default", DEFAULTACCESSFILE);
		bsize = strlen("accessfile") + strlen(DEFAULTACCESSFILE) + 3;
		b = (char*) malloc(bsize);
		enough_mem(b);
		snprintf(b, bsize, "accessfile %s\n", DEFAULTACCESSFILE);
		add_to_options(b);
		free(b);
		return DEFAULTACCESSFILE;
	} else {
		return get_option("accessfile");
	}
}

void free_optst(struct optionstruct* os) {
	if (!os) {
		return;
	}
	free_optst(os->next);
	if (os->key) { free(os->key); }
	if (os->value) { free(os->value); }
	free(os);
}

void free_fwst(struct forwardstruct* fs) {
	if (!fs) {
		return;
	}
	free_fwst(fs->next);
	if (fs->fwlogin) { free(fs->fwlogin); }
	if (fs->fwpass) {
		memset(fs->fwpass, 0, strlen(fs->fwpass));
	}
	if (fs->fwpass) { free(fs->fwpass); }
	if (fs->destlogin) { free(fs->destlogin); }
	if (fs->destpass) {
		memset(fs->destpass, 0, strlen(fs->destpass));
		free(fs->destpass);
	}
	free(fs);
}

void free_throughst(struct throughputstruct* ts) {
	if (!ts) {
		return;
	}
	free_throughst(ts->next);
	if (ts->user) { free(ts->user); }
	if (ts->srcname) { free(ts->srcname); }
	if (ts->destname) { free(ts->destname); }
	free(ts);
}

void free_cmdlogentst(struct cmdlogent_t* cls, struct cmdlogent_t* base) {
	if (!cls) {
		return;
	}
	free_cmdlogentst(cls->next, base);
	if (cls->logf_name) {
		free(cls->logf_name);
	}
	if (cls->specs) {
		free(cls->specs);
	}
	if (cls->logf) {
		fclose(cls->logf);
		cls->logf = (FILE*) 0;
	}
	if (cls == base) {
		cls->logf_name = (char*) 0;
		cls->specs = (char*) 0;
	} else {
		free(cls);
	}
}

/* this is called by every client at the beginning, only the main process
   keeps track of the logins
*/

void free_limitst(struct limitstruct* ls) {
	if (!ls) {
		return;
	}
	free_limitst(ls->next);
	if (ls->hostname) { free(ls->hostname); }
	free(ls);
}

void free_sshpasswdst(struct sshpasswdstruct* ps) {
	if (!ps) {
		return;
	}
	free_sshpasswdst(ps->next);
	if (ps->password) { free(ps->password); }
	if (ps->user) { free(ps->user); }
	if (ps->hostname) { free(ps->hostname); }
	free(ps);
}

void free_sshoptionsst(struct sshoptionsstruct* ps) {
	if (!ps) {
		return;
	}
	free_sshoptionsst(ps->next);
	if (ps->options) { free(ps->options); }
	if (ps->user) { free(ps->user); }
	if (ps->hostname) { free(ps->hostname); }
	free(ps);
}

void free_portrangest(struct portrangestruct* prs) {
	if (!prs) {
		return;
	}
	free_portrangest(prs->next);
	free(prs);
}

void free_llistst(struct lchar* ls) {
	if (!ls) {
		return;
	}
	free_llistst(ls->next);
	free(ls);
}

void reset_confst(struct configstruct* cs) {
	if (!cs) {
		log(1, "configstruct* cs was NULL, this should NEVER happen!");
		return;
	}
	if (cs->logf_name) {
		free(cs->logf_name);
		cs->logf_name = (char*) 0;
	}
	if (cs->logf) {
		fclose(cs->logf);
		cs->logf = (FILE*) 0;
	}
	if (cs->syslog) {
		closelog();
	}
	free_cmdlogentst(&cs->cmdlogfiles, &cs->cmdlogfiles);
	free_cmdlogentst(&cs->cmdlogdirs, &cs->cmdlogdirs);
	cs->cmdlogfiles.next = (struct cmdlogent_t*) 0;
	cs->cmdlogdirs.next = (struct cmdlogent_t*) 0;
}

void conf_delete_config(void) {
	free_optst(optionsbase);
	free_fwst(forwardbase);
	free_throughst(throughputbase);
	free_limitst(limitbase);
	free_sshpasswdst(sshpasswdbase);
	free_sshoptionsst(sshoptionsbase);
	free_portrangest(actvrangebase);
	free_portrangest(pasvrangebase);
	/* reset_confst(&config);  - moved to jftpgw.c */
	if (runasuser.username) {
		free(runasuser.username);
		runasuser.username = 0;
	}
	optionsbase = (struct optionstruct*) 0;
	forwardbase = (struct forwardstruct*) 0;
	throughputbase = (struct throughputstruct*) 0;
	limitbase = (struct limitstruct*) 0;
	sshpasswdbase = (struct sshpasswdstruct*) 0;
	sshoptionsbase = (struct sshoptionsstruct*) 0;
	actvrangebase = (struct portrangestruct*) 0;
	pasvrangebase = (struct portrangestruct*) 0;
}

