/* 
 * Copyright (C) 1999-2001 Joachim Wieland <joe@mcknight.de>
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111, USA.
 */

/*  ----- from tinyproxy for jftpgw
 *
 * Logs the various messages which tinyproxy produces to either a log file or
 * the syslog daemon. Not much to it...
 *
 * Copyright (C) 1998  Steven Young
 * Copyright (C) 1999  Robert James Kaes (rjkaes@flarenet.com)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * log.c - For the manipulation of log files.
 */

#include <stdio.h>
#include <stdarg.h>
#include <time.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>

#include "jftpgw.h"
#include "log.h"

#define LENGTH 64
#define LOGSIZE 800

#ifndef HAVE_SNPRINTF
#   include "snprintf.c"
#else
#   ifndef HAVE_VSNPRINTF
#     include "snprintf.c"
#   endif
#endif

extern struct configstruct config;

/*
 * This routine logs messages to either the log file or the syslog function.
 */
void log(int level, char *fmt, ...)
{
	va_list args;
	time_t nowtime;
	FILE *cf;

	static char time_string[LENGTH];
	static char str[LOGSIZE];

	if (level > config.debuglevel) {
		return;
	}

	va_start(args, fmt);
	if (!config.syslog) {
		/* log via files */
		nowtime = time(NULL);
		/* Format is month day hour:minute:second (24 time) */
		strftime(time_string, LENGTH, "%b %d %H:%M:%S", localtime(&nowtime));
		if (!(cf = config.logf))
			cf = stderr;

		fprintf(cf, "%s [%ld]: ", time_string, (long int) getpid());
		vfprintf(cf, fmt, args);
		fprintf(cf, "\n");
		fflush(cf);
	} else {
		int logtype = LOG_DEBUG;
		if (level < 8) {
			logtype = LOG_INFO;
		}
		if (level < 6) {
			logtype = LOG_WARNING;
		}
		if (level < 4) {
			logtype = LOG_ERR;
		}
		vsnprintf(str, 800 - 1, fmt, args);
		syslog(logtype, str);
	}

	va_end(args);
}

void log_cmd_ent(struct cmdlogent_t* lent, struct loginfo* li) {

	char* commandpattern;
	int commandpatternsize;
	static char time_string[LENGTH];
	time_t nowtime;
	char* ws;

	commandpatternsize = strlen(li->cmd) + 3;
	commandpattern = (char*) malloc(commandpatternsize);
	enough_mem(commandpattern);

	if ((ws = strpbrk(li->cmd, " \t")) == NULL) {
		/* Command consisting of a single word */
		snprintf(commandpattern, commandpatternsize, " %s ", li->cmd);
	} else {
		commandpattern[0] = ' ';
		strncpy(&commandpattern[1], li->cmd, ws - li->cmd);
		commandpattern[ws-li->cmd+1] = ' ';
		commandpattern[ws-li->cmd+2] = '\0';
	}
	toupstr(commandpattern);

	if (strstr(lent->specs, commandpattern)) {
		/* found, log it */
		nowtime = time(NULL);

		if (lent->style == LS_COMMONLOG) {
			/* [18/Feb/2001:11:42:46 CET] */
			strftime(time_string, LENGTH,
				"[%d/%b/%Y:%H:%M:%S %Z]",
				localtime(&nowtime));
			fprintf(lent->logf, "%s UNKNOWN %s %s ",
				li->host, li->user, time_string);
			if (checkbegin(li->cmd, "PASS")) {
				fprintf(lent->logf, "\"PASS *\"");
			} else {
				fprintf(lent->logf, "\"%s\"", li->cmd);
			}
			fprintf(lent->logf, " %d %ld\n",
					li->respcode, li->transferred);
		}

		if (lent->style == LS_XFERLOG) {
			/* Wed Feb 14 01:41:28 2001 */
			strftime(time_string, LENGTH,
				"%a %b %d %H:%M:%S %Y",
				localtime(&nowtime));
			fprintf(lent->logf, "%s %d %s %ld %s %c _ %c %c %s %s 0 * %c\n",
					time_string, li->transfer_duration, li->host, li->transferred, li->name, li->type, li->direction, strcmp(li->user, "anonymous") == 0 || strcmp(li->user, "ftp") ? 'a' : 'r', li->user, li->service, li->complete ? 'c' : 'i');
			li->name = (char*) 0;
		}
		fflush(lent->logf);
	}
}



void log_cmd(struct loginfo* li) {

	/* go through the specifications of the logfiles and write
	 * them if they match */

	int i;
	struct cmdlogent_t* files = &config.cmdlogfiles;
	struct cmdlogent_t* dirs  = &config.cmdlogdirs;

	i = 0;
	while (files && files->logf_name) {	
		i++;
		log_cmd_ent(files, li);
		files = files->next;
	}

	i = 0;
	while (dirs && dirs->logf_name) {
		i++;
		log_cmd_ent(dirs, li);
		dirs = dirs->next;
	}
}

