/*-
 * Copyright (c) 2003 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: loop-ulog.c,v 1.7 2004/03/19 08:12:31 vlm Exp $
 */

#include "ipcad.h"
#include "opt.h"


#if !defined(NETLINK_NFLOG)

void *
loop_ulog(void *psp) {
	(void)psp;
	assert(!"Can't be here");
	return NULL;
}

#else


/*
 * This is the interface loop - a busy horse handling interface statistics.
 */
void *
loop_ulog(void *psp) {
	struct nlmsghdr *nlh;
	ulog_packet_msg_t *upkt;
	packet_source_t *ps = psp;
	ssize_t len = 0;
	int	addrlen;
	sigset_t set, oset;

	assert(ps->iface_type == IFACE_ULOG);

	sigemptyset(&set);
	sigaddset(&set, SIGALRM);

	assert(ps->iface.ulog.bufsize);

	while(1) {

		if(signoff_now)
			break;

		if(len == -1) {
			if(errno != EAGAIN) {
				/* We can't deal with permanent errors.
				 * Just sleep a bit.
				 */
				printf("System call returned %d: %s\n",
					errno, strerror(errno));
				sleep(1);
			}
		}
		
		if(ps->state != PST_READY || ps->iface.ulog.fd == -1) {
			if(init_packet_source_ulog(ps, 0)) {
				sigprocmask(SIG_UNBLOCK, &set, &oset);
				sleep(5);
				sigprocmask(SIG_SETMASK, &oset, NULL);
				continue;
			}
			assert(ps->state == PST_READY);
			fprintf(stderr, "\n");
		}

		sigprocmask(SIG_UNBLOCK, &set, &oset);

		addrlen = sizeof(ps->iface.ulog.peer);
		len = recvfrom(ps->iface.ulog.fd,
			ps->iface.ulog.buf, ps->iface.ulog.bufsize, 0,
			(struct sockaddr*)&ps->iface.ulog.peer, &addrlen);

		sigprocmask(SIG_SETMASK, &oset, NULL);
		
		/*
		 * Read timeout or failure condition.
		 */
		if(len <= 0) continue;
		if(addrlen != sizeof(ps->iface.ulog.peer))
			continue;
			
		/*
		 * Processing packets
		 */
		nlh = (struct nlmsghdr*)ps->iface.ulog.buf;
		if((nlh->nlmsg_flags & MSG_TRUNC)
			||(len > ps->iface.ulog.bufsize))
			continue;

		while(NLMSG_OK(nlh, len)) {

			upkt = NLMSG_DATA(nlh);

			process_packet_data(ps, upkt->payload, upkt->data_len);
			
			if(nlh->nlmsg_type == NLMSG_DONE
				|| !(nlh->nlmsg_flags & NLM_F_MULTI)) {
				/* Last part of the multilink message */
				break;
			}

			nlh = NLMSG_NEXT(nlh, len);
		}
	}

	return 0;
}

#endif	/* ULOG support */
