/*-
 * Copyright (c) 2001, 2002 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: loop-bpf.c,v 1.4 2004/03/19 08:12:31 vlm Exp $
 */

#include "ipcad.h"
#include "opt.h"

#define	bpf_header	((struct bpf_hdr *)packet)


/*
 * This is the interface loop - a busy horse handling interface statistics.
 */
void *
loop_bpf(void *psp) {
	packet_source_t *ps = psp;
	ssize_t rd = 0;
	char *packet;
	struct pollfd pfd;
	sigset_t set, oset;
	
	assert(ps->iface_type == IFACE_BPF);

	pfd.fd = ps->iface.bpf.fd;
	pfd.events = POLLIN;

	sigemptyset(&set);
	sigaddset(&set, SIGALRM);

	while(1) {

		if(signoff_now)
			break;

		if(rd == -1) {
			if(errno != EAGAIN) {
				/* We can't deal with permanent errors.
				 * Just sleep a bit.
				 */
				printf("System call returned %d: %s\n",
					errno, strerror(errno));
				sleep(1);
			}
		}

		if(ps->state != PST_READY || ps->iface.bpf.fd == -1) {
			if(init_packet_source(ps)) {
				sigprocmask(SIG_UNBLOCK, &set, &oset);
				sleep(5);
				sigprocmask(SIG_SETMASK, &oset, NULL);
				continue;
			}
			assert(ps->state == PST_READY);
			fprintf(stderr, "\n");
			pfd.fd = ps->iface.bpf.fd;
		}

		sigprocmask(SIG_UNBLOCK, &set, &oset);

		rd = poll(&pfd, 1, -1);

		sigprocmask(SIG_SETMASK, &oset, NULL);

		/* Poll timeout almost impossible thing */
		if(rd == 0) {
			printf("Unexpected timeout on interface %s."
			" Please report to vlm@lionet.info."
			" Sleeping.\n",
			ps->ifName);
			sleep(1);
			continue;
		}

		if((rd == -1) || !(pfd.revents & POLLIN)) {
			rd = -1;
			continue;
		}

		if(signoff_now)
			break;

		rd = read(ps->iface.bpf.fd,
			ps->iface.bpf.buf,
			ps->iface.bpf.bufsize);

		/*
		 * Read timeout or failure condition
		 */
		if(rd <= 0)
			continue;
	
		/*
		 * Processing packets
		 */
		for(packet = ps->iface.bpf.buf;
			(packet - ps->iface.bpf.buf) < rd;
			  packet += BPF_WORDALIGN(bpf_header->bh_hdrlen
				  + bpf_header->bh_caplen) )
		{
			process_packet_data(ps,
				packet + bpf_header->bh_hdrlen,
				bpf_header->bh_caplen);
		}

	}

	return 0;

}

