/*-
 * Copyright (c) 2001 Lev Walkin <vlm@spelio.net.ru>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: rshp.c,v 1.11 2001/08/30 08:40:24 vlm Exp $
 */

#include "ipcad.h"
#include "cfgvar.h"
#include "got.h"
#include "csparse.h"
#include "rw.h"
#include "servers.h"
#include "service.h"
#include "opt.h"

void rsh_usage(FILE *f);
int process_rsh_command(int sock, char *command, int privlevel);

int rsh_rw_timeout = 30000;

int
rsh_client_options(int client, struct sockaddr_in *srem) {
	struct linger lng = { 1, 5 }; 
	struct timeval tv = { rsh_rw_timeout / 1000, 0 };
#ifdef	HAVE_SOCKLEN_T
	socklen_t sl;
#else
	size_t sl;
#endif
	long l;

	/* Set lingering on close */
	if(setsockopt(client, SOL_SOCKET, SO_LINGER, &lng, sizeof(lng)))
		return -1;

	/* Set receive and send timeouts */
	/* Unsupported by the Linux on which I've tested it, so ignore errors. */
#ifdef	SO_RCVTIMEO
	setsockopt(client, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
#endif
#ifdef	SO_SNDTIMEO
	setsockopt(client, SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv));
#endif

#ifdef	SO_KEEPALIVE
	/* Set keepalive against broken connections */
	l = 1;
	setsockopt(client, SOL_SOCKET, SO_KEEPALIVE, &l, sizeof(l));
#endif

#ifdef	IPPROTO_IP

#ifdef	TCP_NODELAY
	/* Disable Nagle algorithm for efficiency */
	l = 1;
	setsockopt(client, IPPROTO_IP, TCP_NODELAY, &l, sizeof(l));
#endif

#ifdef	IP_TTL
	/* Set TTL reasonably low to avoid remote attacks */
	l = default_ttl;
	setsockopt(client, IPPROTO_IP, IP_TTL, &l, sizeof(l));
#endif

#endif

	/* Get remote address */
	sl = sizeof(struct sockaddr_in);
	if( getpeername(client, (struct sockaddr *)srem, &sl)
		|| (sl != sizeof(*srem))) {
		return -1;
	}

	return 0;
}


int
getstr(char *buf, size_t bufsize, int sock) {
	char c;
	ssize_t sz;

	*buf = 0;

	do {
		sz = read_timeout(sock, &c, 1, rsh_rw_timeout);

		if(signoff_now)
			return -1;

		if(sz != 1)
			return -1;

		*buf++ = c;

		if (--bufsize == 0)
			return -1;

	} while (c != 0);

	return 0;
};



/*
** remote shell thread:
** [port]\0
** remuser\0
** locuser\0
** command\0
** data
*/

int
process_rsh_request(int client) {
	struct sockaddr_in srem;
	char port_s[16];
	char remuser[16];
	char locuser[16];
	char command[256];
	int port;
	int rsock = -1;
	int privlevel;
	char c;

	/* Set socket options */
	if( rsh_client_options(client, &srem) )
		return -1;

	if( getstr(port_s, sizeof(port_s), client) )
		return -1;

	if(*port_s) {

		port = atoi(port_s);
		if(port >= 1024 || port < 1)
			return -1;

		srem.sin_port = htons(port);

		port = IPPORT_RESERVED - 1;
		if( (rsock = rresvport(&port)) == -1) {
			return -1;
		};
	
		if( connect(rsock, (struct sockaddr *)&srem,
			sizeof(struct sockaddr_in)) == -1) {
			close(rsock);
			return -1;
		}

	} /* if(*port_s) */;

	if( getstr(remuser, sizeof(remuser), client) ) {
		if(rsock != -1)
			close(rsock);
		return -1;
	};

	if( getstr(locuser, sizeof(locuser), client) ) {
		if(rsock != -1)
			close(rsock);
		return -1;
	};

	if( getstr(command, sizeof(command), client) ) {
		if(rsock != -1)
			close(rsock);
		return -1;
	};


	c = '\0';
	safe_write(client, &c, 1);

	if( (privlevel = cfg_check_rsh(remuser, &srem.sin_addr)) == 0 ) {
		char msg[] = "Permission denied\n";
		safe_write(client, msg, sizeof(msg) - 1);
		if(rsock != -1)
			close(rsock);
		return -1;
	};

	if( process_rsh_command(client, command, privlevel) == -1) {
		if(rsock != -1)
			close(rsock);
		return -1;
	};

	if(rsock != -1)
		close(rsock);

	return 0;
};

static int
check_permission(int privlevel, int operation) {

	/* Priviledge level required by operation */
	operation >>= 4;

	if(privlevel < operation)
		return 0;

	return 1;
}

int
process_rsh_command(int sock, char *command, int privlevel) {
	FILE *f;
	char *param = NULL;
	int Oper = 0;

	if(!(f=fdopen(sock, "w"))) {
		char msg[] = "Internal error.\n";
		safe_write(sock, msg, sizeof(msg) - 1);
		shutdown(sock, SHUT_RDWR); close(sock);
		return -1;
	};

	Oper = csparse(command, &param);

	/* Check command against privlevel */

	if(abs(Oper) < CS_HELP) {
		if(param)
			free(param);
		fprintf(f, "Syntax error. Type 'help' for usage.\n");
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return -1;
	}

	if(!check_permission(privlevel, Oper)) {
		fprintf(f, "Permission denied\n");
		if(param)
			free(param);
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return -1;
	}

	if(Oper == CS_HELP) {
		rsh_usage(f);
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	if(Oper == CS_STAT) {
		show_stats(f);
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	if(Oper == CS_SVER) {
		show_version(f);
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	if(Oper == CS_DUMP) {
		if(!param)
			param = cfg_dump_file;

		if(param) {
			make_dump(param, f);
			if(param != cfg_dump_file)
				free(param);
		} else {
			fprintf(f, "Dump file is not defined.\n");
			fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
			return -1;
		};
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	if(Oper == CS_IMPT) {

		if(!param)
			param = cfg_dump_file;

		if(param) {
			import_table(param, f, 0);
			if(param != cfg_dump_file)
				free(param);
		} else {
			fprintf(f, "Dump file is not defined.\n");
			fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
			return -1;
		};
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	if(Oper == CS_REST) {

		if(!param)
			param = cfg_dump_file;

		if(param) {
			import_table(param, f, 1);
			if(param != cfg_dump_file)
				free(param);
		} else {
			fprintf(f, "Dump file is not defined.\n");
			fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
			return -1;
		};
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	/* show ip accounting [checkpoint] */
	if( Oper == CS_SIA || Oper == CS_SIAC ) {
		display(f, (Oper==CS_SIAC)?1:0);
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	/* show interface xxx */
	if( Oper == CS_SIN ) {
		if_stat(f, param);
		free(param);
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	/* CLEAR ... */
	if( Oper == CS_CIAC ) {

		pthread_mutex_lock(&checkpoint_mutex);

		clear_ip_accounting(checkpoint);
		checkpoint = NULL;
		checkpoint_entries = 0;
		ex_checkpoint_stalled = 0;
		ex_checkpoint_packets = 0;
		ex_checkpoint_bytes = 0;
		time(&checkpoint_time);

		pthread_mutex_unlock(&checkpoint_mutex);

		fprintf(f, "IP accounting checkpoint cleared\n");

		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	/* clear ip accounting */
	if( Oper == CS_CIA ) {

		pthread_mutex_lock(&checkpoint_mutex);
		pthread_mutex_lock(&ipstream_mutex);

		clear_ip_accounting(checkpoint);
		checkpoint = ipstream;

		ipstream = NULL;
		checkpoint_entries = ipstream_entries;
		ipstream_entries = 0;

		checkpoint_time = ipstream_time;
		time(&ipstream_time);

		ex_checkpoint_stalled = ex_current_stalled;
		ex_checkpoint_packets = ex_current_packets;
		ex_checkpoint_bytes = ex_current_bytes;
		ex_current_stalled = 0;
		ex_current_packets = 0;
		ex_current_bytes = 0;

		pthread_mutex_unlock(&ipstream_mutex);
		pthread_mutex_unlock(&checkpoint_mutex);

		fprintf(f, "IP accounting cleared\n");

		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		return 0;
	};

	if( Oper == CS_SHUT ) {
		fprintf(f, "Shutdown process started\n");
		fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);
		signoff_now = 1;
		return 0;
	}

	fprintf(f, "Invalid request.\n");
	fflush(f); shutdown(sock, SHUT_RDWR); fclose(f);

	return 0;
};


